from __future__ import annotations

import os
from datetime import datetime
from pathlib import Path
from xml.etree.ElementTree import (
    Element,
    SubElement,
    XMLParser,
    indent,
    parse,
    tostring,
)

from qtpy.QtCore import QEvent, QObject, Signal
from qtpy.QtWidgets import QFileDialog, QMessageBox

from .model import pvItemModel

DATE_FORMAT = r"%Y-%m-%d %H:%M:%S"


def now():
    return datetime.today().strftime(DATE_FORMAT)


class pvProject(QObject):
    stateChanged = Signal(bool)

    def safe(callback):
        """Decorator to handle detatched/dirty state."""

        def wrapper(self, *args, **kwargs):
            abort = not self.ok
            if abort:
                match QMessageBox.warning(
                    None,
                    "Detatched state",
                    "Do you want to save the pending changes ?",
                    buttons=QMessageBox.StandardButtons(
                        QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel
                    ),
                    defaultButton=QMessageBox.Yes,
                ):
                    case QMessageBox.Yes:
                        abort = not self.save()
                    case QMessageBox.No:
                        abort = False
            if abort:  # reject close events !
                for arg in list(args) + list(kwargs.values()):
                    if isinstance(arg, QEvent) and arg.type() == QEvent.Close:
                        arg.ignore()
                return None
            return callback(self, *args, **kwargs)

        return wrapper

    def __init__(
        self,
        model: pvItemModel,
        title: str = "New project",
        description: str = "Awsome 3D visualisation project",
        author: str = Path.home().stem,
        date: str = now(),
        parent: QObject = None,
    ):
        super().__init__(parent)

        if not isinstance(model, pvItemModel):
            raise TypeError(pvItemModel)

        self.model = model
        self.title: str = str(title)
        self.description: str = str(description)
        self.author: str = str(author)
        self.date: str = str(date)

        self._ok: bool = True
        self._file: str = None

    @property
    def ok(self):
        return self._ok

    @ok.setter
    def ok(self, value: bool):
        value = bool(value)
        if value != self._ok:
            self._ok = value
            self.stateChanged.emit(value)

    def invalidate(self):
        self.ok = False

    @safe
    def closeEvent(self, event):
        event.accept()

    def package(self, file: str | None = None):
        """Serialize project, hardcopy data and leave project state unchanged.

        Args:
            file (str, optional): Destination of the export. Defaults to None.
        """
        file = (
            file
            or QFileDialog.getSaveFileName(
                None,
                "Export project",
                dir=str(Path(self.title).with_suffix(".xml")),
                filter=("XML (*.xml)"),
            )[0]
        )

        if not file:
            return

        xml = self.to_xml(include_data=True)
        indent(xml)
        payload = tostring(
            xml,
            xml_declaration=True,
            encoding="utf8",
        )
        Path(file).write_bytes(payload)

    def save(self, include_data: bool = False) -> bool:
        """Serialize project.

        Args:
            include_data (bool, optional): Hardcopy data into the project file if True. Defaults to False.

        Returns:
            bool: Serialization succes flag.
        """
        if self._file:
            self.relativize()
            temp_layers = [el for el in self.model.layers() if el.is_temp]
            if temp_layers and (
                QMessageBox.warning(
                    None,
                    "Temporary layers",
                    f"There are {len(temp_layers)} temporary layers in your project !\n"
                    + "Their geometries will be lost if not saved, do you want to save them along the project ?",
                    QMessageBox.Yes | QMessageBox.No,
                )
                == QMessageBox.Yes
            ):
                path = Path(self._file)
                prefix = path.stem
                for el in temp_layers:
                    el.save(prefix=prefix)
            xml = self.to_xml(include_data=include_data)
            indent(xml)
            payload = tostring(
                xml,
                xml_declaration=True,
                encoding="utf8",
            )
            Path(self._file).write_bytes(payload)
            self.ok = True
        else:
            self.saveAs()
        return self.ok

    def saveAs(self, file=None, include_data: bool = False) -> bool:
        """Serialize project and update current location if changes.

        Args:
            file (str, optional): Where to save the project. Defaults to None.
            include_data (bool, optional): Hardcopy data into the project file if True. Defaults to False.

        Returns:
            bool: Serialization succes flag.
        """
        if file := QFileDialog.getSaveFileName(
            None,
            "Save project",
            dir=str(Path(self.title).with_suffix(".xml")),
            filter=("XML (*.xml)"),
        )[0]:
            self._file = file
            return self.save(include_data)
        return self.ok

    @safe
    def clear(self) -> bool:
        self.model.clear()
        other = self.__class__(self.model)
        self.title = other.title
        self.description = other.description
        self.date = other.date
        self._file = other._file
        self.ok = True
        return self.ok

    def to_xml(self, include_data: bool = False) -> Element:
        root = Element(
            "Project",
            title=str(self.title),
            author=str(self.author),
            creation_date=str(self.date),
            last_modified=str(now()),
        )
        details = SubElement(root, "Description")
        details.text = self.description
        for el in self.model.to_xml(include_data=include_data):
            root.append(el)
        return root

    @safe
    def load(self, file: str | None = None) -> bool:
        if not file:
            file, *_ = QFileDialog.getOpenFileName(
                None,
                "Open project",
                dir=str(Path(self._file or Path.home())),
                filter="XML files (*.xml);; All files (*.*)",
            )

        if not Path(file).is_file():
            return False

        xml = parse(file, XMLParser(encoding="utf-8")).getroot()
        self.clear()
        # project attributes
        for k, v in xml.attrib.items():
            setattr(self, k, v)
        desc = xml.find("Description")
        if desc is not None:
            self.description = desc.text
        # project data
        os.chdir(Path(file).parent)
        self.model.load_xml(xml)
        self._file = Path(file).resolve()
        self.ok = True
        return None
        # except Exception as err:  # TODO: handle errors
        #     raise OSError(f"Invalid project file !\n{err}")

    def relativize(self):
        project = Path(self._file).resolve()
        if not project.is_file():
            return

        new_loc = project.parent.resolve()
        for layer in self.model.layers():
            if source := layer.context.source:
                try:
                    layer.context.source = (
                        Path(source).resolve().relative_to(new_loc, walk_up=True)
                    )
                except Exception():
                    pass
            if texture := layer.symbology._texture:
                try:
                    layer.symbology._texture = (
                        Path(texture).resolve().relative_to(new_loc, walk_up=True)
                    )
                except Exception():
                    pass
        os.chdir(new_loc)
