from __future__ import annotations

import re
from enum import IntEnum, auto
from typing import Any

from pyvista import (
    DataObject,
    ExplicitStructuredGrid,
    ImageData,
    MultiBlock,
    PointSet,
    PolyData,
    RectilinearGrid,
    StructuredGrid,
    UnstructuredGrid,
)
from vtkmodules import vtkIOXML

vtkIOXMLWriters = tuple(filter(re.compile(r"vtkXML\w+?Writer").match, dir(vtkIOXML)))


def vtkIOXMLFileExtension(obj: DataObject) -> str:
    for ext, writer in obj._WRITERS.items():  # noqa: B007
        if writer.__name__ in vtkIOXMLWriters:
            break
    return ext


def vtkIOXMLFileExtensions(obj: DataObject) -> list[str]:
    return [
        ext
        for ext, writer in obj._WRITERS.items()
        if writer.__name__ in vtkIOXMLWriters
    ]


class pvGeometryType(IntEnum):
    MULTI = -1
    INVALID = 0
    MESH = auto()
    GRID = auto()
    IMAGE = auto()

    @staticmethod
    def of(obj: Any):
        if isinstance(obj, PointSet | PolyData | UnstructuredGrid):
            return pvGeometryType.MESH
        if isinstance(
            obj,
            ExplicitStructuredGrid | RectilinearGrid | StructuredGrid,
        ):
            return pvGeometryType.GRID
        if isinstance(obj, ImageData):
            return pvGeometryType.IMAGE
        if isinstance(obj, MultiBlock):
            return pvGeometryType.MULTI
        return pvGeometryType.INVALID
