from __future__ import annotations

from typing import Any

from qtpy.QtCore import Qt
from qtpy.QtWidgets import (
    QCheckBox,
    QComboBox,
    QDialog,
    QDoubleSpinBox,
    QFileDialog,
    QFormLayout,
    QLabel,
    QPushButton,
    QSlider,
)

from . import (
    pvGeometryType,
    pvIcon,
    pvItemModel,
    pvLayer,
    pvLayerContext,
    pvLayerSymbology,
)
from .colors import QICSColorButton


class pvLayerSettings(QDialog):
    def __init__(self, model: pvItemModel, layers: list[pvLayer] | None = None):
        if layers is None:
            layers = []
        super().__init__()

        self.setModal(False)  # ignored by self.exec() method !
        self.setWindowIcon(pvIcon("mActionOptions"))
        self.setWindowTitle("Layer properties")
        self.setLayout(QFormLayout())

        self.model = model
        self.layers = layers or model.layers

        assert isinstance(self.model, pvItemModel)
        assert all(isinstance(el, pvLayer) for el in self.layers)

        self._init_ui()

    def get(self, name: str):
        if name == "scalars":
            values = []
            for el in self.layers:
                if el.geometry:
                    values += el.geometry.array_names
            default = None
        elif hasattr(pvLayer, name):
            values = [getattr(el, name) for el in self.layers]
            default = None
        elif hasattr(pvLayerSymbology, name):
            values = [getattr(el.symbology, name) for el in self.layers]
            default = getattr(pvLayerSymbology, name)
        elif hasattr(pvLayerContext, name):
            values = [getattr(el.context, name) for el in self.layers]
            default = getattr(pvLayerContext, name)
        else:
            raise AttributeError(name)

        v = set(values)
        return v.pop() if len(v) == 1 else default

    def set(self, name: str, value: Any):
        if hasattr(pvLayerSymbology, name):
            for el in self.layers:
                setattr(el.symbology, name, value)
                if (
                    name == "_texture"
                    and value
                    and el.geometry.active_texture_coordinates is None
                ):
                    el.geometry.texture_map_to_plane(inplace=True)
        else:
            raise AttributeError(name)
        self.model.layersChanged.emit(self.layers)

    def _init_ui(self) -> None:
        self.layout().addRow("uid:", QLabel(self.get("uid")))
        self.layout().addRow("source:", QLabel(self.get("source")))
        self.layout().addRow(
            "geometry:",
            QLabel((self.get("geom_t") or pvGeometryType.MULTI).name),
        )

        scalars = QComboBox()
        options = [""]
        for el in self.layers:
            if el.geometry:
                options += el.geometry.array_names
        options = list(set(options))
        scalars.addItems(options)
        scalars.setCurrentIndex(
            options.index(self.get("scalars")) if self.get("scalars") in options else 0
        )
        scalars.currentTextChanged.connect(lambda s: self.set("_scalars", s))
        self.layout().addRow("scalars:", scalars)

        log_scale = QCheckBox()
        log_scale.setCheckState(Qt.Checked if self.get("log_scale") else Qt.Unchecked)
        log_scale.stateChanged.connect(lambda v: self.set("_log_scale", v))
        self.layout().addRow("log scale:", log_scale)

        cmap = QComboBox()
        options = list(
            {
                "",
                "binary",
                "gray",
                "terrain",
                "ocean",
                "jet",
                "viridis",
                "hsv",
                "inferno",
            }
            | {self.get("cmap")}
        )
        cmap.addItems(options)
        scalars.setCurrentIndex(
            options.index(self.get("cmap")) if self.get("cmap") in options else 0
        )
        cmap.currentTextChanged.connect(lambda s: self.set("_cmap", s))
        self.layout().addRow("colormap:", cmap)

        color = QICSColorButton(self.get("color"))
        color.colorChanged.connect(lambda c: self.set("color", c.name()))
        self.layout().addRow("color:", color)

        opacity = QSlider(Qt.Horizontal)
        opacity.setMinimum(0)
        opacity.setMaximum(100)
        opacity.setValue(int(self.get("opacity") * 100))
        opacity.valueChanged.connect(lambda v: self.set("opacity", float(v) / 100.0))
        self.layout().addRow("opacity:", opacity)

        show_edges = QCheckBox()
        show_edges.setCheckState(Qt.Checked if self.get("show_edges") else Qt.Unchecked)
        show_edges.stateChanged.connect(lambda v: self.set("show_edges", v))
        self.layout().addRow("show edges:", show_edges)

        edge_color = QICSColorButton(self.get("edge_color"))
        edge_color.colorChanged.connect(lambda c: self.set("edge_color", c.name()))
        self.layout().addRow("edges color:", edge_color)

        render_points_as_spheres = QCheckBox()
        render_points_as_spheres.setCheckState(
            Qt.Checked if self.get("render_points_as_spheres") else Qt.Unchecked
        )
        render_points_as_spheres.stateChanged.connect(
            lambda v: self.set("render_points_as_spheres", v)
        )
        self.layout().addRow("render points as spheres:", render_points_as_spheres)

        point_size = QDoubleSpinBox()
        point_size.setMinimum(0)
        point_size.setMaximum(100)
        point_size.setDecimals(1)
        point_size.setValue(self.get("point_size"))
        point_size.valueChanged.connect(lambda v: self.set("point_size", v))
        self.layout().addRow("points size:", point_size)

        render_lines_as_tubes = QCheckBox()
        render_lines_as_tubes.setCheckState(
            Qt.Checked if self.get("render_lines_as_tubes") else Qt.Unchecked
        )
        render_lines_as_tubes.stateChanged.connect(
            lambda v: self.set("render_lines_as_tubes", v)
        )
        self.layout().addRow("render lines as tubes:", render_lines_as_tubes)

        line_width = QDoubleSpinBox()
        line_width.setMinimum(0)
        line_width.setMaximum(100)
        line_width.setDecimals(1)
        line_width.setValue(self.get("line_width"))
        line_width.valueChanged.connect(lambda v: self.set("line_width", v))
        self.layout().addRow("lines size:", line_width)

        texture = QPushButton("Browse")

        def pick_texture():
            self.set(
                "_texture",
                QFileDialog.getOpenFileName(
                    None,
                    "Pick a texture file",
                    directory=self.layers[0].source,
                    filter=("Texture image (*.png *.tif *.jpg);; All files (*.*)"),
                )[0],
            )

        texture.clicked.connect(pick_texture)
        # texture.setEnabled(False)
        self.layout().addRow("texture:", texture)
