import xml.etree.ElementTree as ET
from pathlib import Path

from pyvista import MultiBlock


def rename_blocks(vtm_file, mapper: dict | None = None):
    """pyvista.MultiBlock.save arbitrary name each using using:
    f"block_{i}.vt{*}" pattern, discarding the "name" attribute.

    This function tries to rename block using the block attribute
    """
    if mapper is None:
        mapper = {}
    vtm_file = Path(vtm_file)
    if not vtm_file.is_file():
        return
    path, stem = vtm_file.parent, vtm_file.stem
    tree = ET.parse(vtm_file)
    root = tree.getroot()
    routing = {}
    for el in root.iter("DataSet"):
        file = Path(el.attrib["file"])
        name = el.attrib["name"]
        unique, i = mapper(name, name), 1
        dst = Path(stem) / (unique + file.suffix)
        while unique in routing.values() or dst in routing:
            unique = f"{mapper(name, name)}({i})"
            dst = Path(stem) / (unique + file.suffix)
        routing[file] = Path(stem) / (unique + file.suffix)
    for block in root.iter("DataSet"):
        org = Path(el.attrib["file"])
        dst = routing[file]
        name = dst.stem
        block.set("name", name)
        block.set("file", dst.as_posix())
        (path / org).replace(path / dst)
    tree.write(vtm_file)


def save(multiblock: MultiBlock, filename: Path, mapper: dict | None = None):
    if mapper is None:
        mapper = {}
    multiblock.save(filename)
    rename_blocks(filename)
