from __future__ import annotations

from functools import cached_property
from logging import warning
from typing import Any

from pyvistaqt import QtInteractor
from qtpy.QtCore import QPoint, QSize, Qt
from qtpy.QtWidgets import QMenu, QWidget

from ..core import pvGroup, pvLayer, pvSettings
from . import pvAction, pvIcon
from .renderer_dlg import pvRendererDialog


class pvRenderer(QtInteractor):
    def __init__(
        self,
        parent: QWidget = None,
        *,
        title: str | None = None,
        off_screen: bool | None = None,
        multi_samples: int | None = None,
        line_smoothing: bool = False,
        point_smoothing: bool = False,
        polygon_smoothing: bool = False,
        auto_update: float | bool = 5,
        **kwargs: Any,
    ) -> None:
        super().__init__(
            parent,
            title,
            off_screen,
            multi_samples,
            line_smoothing,
            point_smoothing,
            polygon_smoothing,
            auto_update,
            **kwargs,
        )
        self.setMinimumSize(QSize(64, 64))
        self.init_actions()
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self.on_context_menu)

        for attr, args in self.settings.to_dict(self.group).items():
            try:
                self.set(attr, **args)
            except AttributeError as err:
                warning(err)

    @property
    def group(self) -> str:
        return "renderer"

    @cached_property
    def settings(self):
        return pvSettings()

    @property
    def menu(self):
        menu = QMenu("View", parent=self)
        menu.addAction(self.action_reset_camera)
        menu_views = menu.addMenu("Orthogonal views")
        menu_views.addAction(pvAction("XY view", callback=self.view_xy, parent=self))
        menu_views.addAction(pvAction("YZ view", callback=self.view_yz, parent=self))
        menu_views.addAction(pvAction("XZ view", callback=self.view_xz, parent=self))
        menu.addAction(self.action_renderer_settings)
        return menu

    def on_context_menu(self, point: QPoint):
        self.menu.exec(self.mapToGlobal(point))

    def init_actions(self):
        self.action_reset_camera = pvAction(
            "Reset camera",
            icon=pvIcon("mActionZoomTo"),
            shortcut="R",
            callback=lambda: self.reset_camera(),
            parent=self,
        )

        self.action_renderer_settings = pvAction(
            "Settings...",
            icon=pvIcon("settings"),
            shortcut="Ctrl+,",
            callback=pvRendererDialog(self).exec,
            parent=self,
        )

    def add_layer(self, layer: pvLayer):
        if isinstance(layer, pvGroup):
            return  # ignore groups
        if not isinstance(layer, pvLayer):
            raise TypeError(layer)
        if not layer.is_valid:
            return  # ignore empty/invalid geometries
        super().add_mesh(
            layer.geometry,
            name=str(layer),
            render=layer.state,
            texture=layer.symbology.texture,
            **layer.symbology.kwargs,
        )

    def add_layers(self, layers: list[pvLayer]):
        for el in layers:
            self.add_layer(el)

    def remove_layer(self, layer: pvLayer):
        if isinstance(layer, pvGroup):
            for el in layer:
                self.remove_layer(el)
            self.update()
        elif actor := self.actors.get(str(layer), None):
            super().remove_actor(actor)
            self.update()

    def remove_layers(self, layers: list[pvLayer]):
        for el in layers:
            self.remove_layer(el)

    def reset_camera(
        self,
        bounds: Any = None,
        render: bool = True,  # noqa: ARG002
    ) -> None:
        super().reset_camera(render=True, bounds=bounds)

    def set(self, name: str, **kwargs):
        if name == "background":
            super().set_background(**kwargs)
        elif name == "scale":
            super().set_scale(**kwargs)
        elif name == "orientation_widget":
            shape = kwargs.get("geometry")
            color = kwargs.get("color")
            if not shape or shape.lower() == "none":
                super().hide_axes()
            elif shape.lower() in ("arrow", "north"):
                super().add_north_arrow_widget(color=color or "red", edge_color=color)
            elif shape.lower() == "axes":
                super().add_axes(color=color)
        else:
            raise AttributeError(name)

        for key, value in kwargs.items():
            self.settings.setValue(key, value)

    def redraw_layers(self, layers: list[pvLayer] | None = None):
        if layers is None:
            layers = []
        for layer in layers:
            if actor := self.actors.get(str(layer), None):
                actor.visibility = layer.is_enabled
                actor.texture = layer.symbology.texture
                for key, value in layer.symbology.kwargs.items():
                    setattr(actor.prop, key, value)
        super().update()
