from __future__ import annotations

from itertools import chain


def ravel_cells(nested_cells: list[list[int]]) -> list:
	"""Converts nested lists of indices into a flat list of connectivities.

	e.g. [[0,1,2],[0,2,3]]  -> [3,0,1,2,3,0,2,3]
	"""
	flat_connectivity = chain.from_iterable([(len(c), *c) for c in nested_cells])
	return list(flat_connectivity)


def unravel_cells(flat_cells: list[int]) -> list[list[int]]:
	"""Converts a flat list of connectivities into nested lists of indices.

	e.g. [3,0,1,2,3,0,2,3] -> [[0,1,2],[0,2,3]]
	"""
	nested_cells = []
	i, stop = 0, len(flat_cells)
	while i < stop:
		n = flat_cells[i]
		i += 1
		nested_cells.append(flat_cells[i : i + n])
		i += n
	return nested_cells


def mesh2coords(
	points: list[tuple[float]], cells: list[int]
) -> list[list[tuple[float]]]:
	coords = []
	i, stop = 0, len(cells)
	while i < stop:
		n = cells[i]
		i += 1
		coords.append(points[cells[i : i + n]])
		i += n
	return coords
