from __future__ import annotations

from configparser import ConfigParser
from contextlib import contextmanager
from pathlib import Path
from typing import Any

from qgis.PyQt.QtCore import QSettings
from qgis.PyQt.QtCore import pyqtSignal as QSignal


class qvSettings(QSettings):
	"""Convenient QSettings singleton wrapping `metadata.txt`.

	Featuring:
	    - Absolute and relative key paths support (set/get/find keys from anywhere)
	    - Group focus context manager (select a sub group for a certain job)
	    - Signal emission at value change (with key absolute path)
	    - Python dictionnary export
	"""

	valueChanged = QSignal(str, object)
	_instance = None

	def __init__(self):
		# make it a singleton
		self.__class__.__new__ = lambda _: self
		# set/get aliases
		self.get = self.value
		self.set = self.setValue
		self.setdefault = self.setDefault
		# make sure QSettings.__init__() is called only once
		if not self._instance:
			self.__default_init__()
			self._instance = self

	# default values
	def __default_init__(self, manifest: Path = Path(__file__).parent / "metadata.txt"):
		# parse metadata.txt as a dictionnary
		conf = ConfigParser(allow_no_value=True)
		conf.read(manifest)
		super().__init__("QGIS", application=f"{conf.get('general', 'name')}")
		# register metadata content
		for attr in conf["general"]:
			self.setValue(attr, conf.get("general", attr))
		# register paths
		self.setValue("path", manifest.parent.as_posix())

	# change active group
	def _seek(self, group: str = "/") -> tuple[str, str]:
		"""Sets the active group using absolute or relative path.

		Args:
		    group (str, optional): QSettings group to set as active. Defaults to top-level group.

		Returns:
		    Tuple[str, str]: Both (previous, new) groups absolute paths.
		"""

		# close all opened groups, track closed path
		old = ""
		while curr := self.group():
			old = curr + "/" + old
			self.endGroup()
		old = "/" + old

		# resolve seeked group
		prefix, _, suffix = group.strip("/").partition("/")
		if set(prefix) == {"."}:  # relative paths
			n = len(prefix) - 1
			prefix = old.rsplit("/", maxsplit=n)[0]
		new = "/".join((prefix, suffix))
		if new[-1] != "/":
			new += "/"

		# open seeked group
		self.beginGroup(new)

		return old, new

	# get active group absolute path
	def path(self) -> str:
		old = self._seek(".")[0]
		return old or "/"

	@contextmanager
	def focus(self, group: str = "/"):
		"""Context manager that helps focusing a specific group.

		WARNING: Since pvSettings is a singleton, calling the `_seek()` method is risky.
		You could end up leaving the instance in an undefined state (i.e. unwanted active group)!

		This context manager allows you to focus temporarily on a specific subgroup,
		using the syntax:
		    `with self.focus(group="my/sub/group") as sub: ...`
		wich will ensure you get back to the previous state

		Args:
		    group (str, optional): QSettings group to set as active. Defaults to top-level group.

		Yields:
		    pvSettings: Current instance with the seeked group focused.
		"""
		state = self._seek(group)[0]
		try:
			yield self
		finally:
			state = self._seek(state)

	# better value()
	def value(self, key: str, default: Any = None) -> Any:
		"""Returns the value for absolute `key`. If the setting doesn't exist, returns `default`."""
		if "/" in key:
			group, key = key.rsplit("/", maxsplit=1)
			with self.focus(group) as this:
				return this.value(key, default)
		else:
			return super().value(key, default)

	# better setValue()
	def setValue(self, key: str, value: Any):
		"""Sets the value of absolute `key` to `value`. If the `key` already exists, the previous value is overwritten."""
		if "/" in key:
			group, key = key.rsplit("/", maxsplit=1)
			with self.focus(group) as this:
				return this.setValue(key, value)
		else:
			self.valueChanged.emit(self.path() + key, value)
			return super().setValue(key, value)

	# soft setValue()
	def setDefault(self, key: str, value: Any) -> Any:
		"""Sets the value of absolute `key` to `value`. If the `key` already exists, the previous value is preserved."""
		if "/" in key:
			group, key = key.rsplit("/", maxsplit=1)
			with self.focus(group) as this:
				return this.setdefault(key, value)
		elif key in self.allKeys():
			return self.value(key)
		else:
			self.setValue(key, value)
			return value

	# better remove()
	def remove(self, key: str):
		"""Removes the setting `key` and any sub-settings of the key."""
		if "/" in key:
			group, key = key.rsplit("/", maxsplit=1)
			with self.focus(group) as this:
				return this.remove(key)
		else:
			return super().remove(key)

	# better allKeys()
	def allKeys(self, group: str = ".") -> list[str]:
		"""Returns a list of all keys, including subkeys, relative to a group."""
		with self.focus(group):
			return super().allKeys()

	# dict view of QSettings
	def to_dict(self, group: str = ".") -> dict:
		"""Cast QSettings into Python `dict[str:Union[str,dict]`.

		Also cast numeric strings into numeric variables (`int` or `float`).
		"""
		dct = {}
		with self.focus(group) as this:
			for k in this.childKeys():
				value = this.value(k)
				if isinstance(value, str):
					v = value.lstrip("-").replace("e", "", 1)
					if v.isdecimal():
						value = int(value)
					elif v.replace(".", "", 1).isdecimal():
						value = float(value)
				dct[k] = value
			for g in this.childGroups():
				dct[g] = this.to_dict(f"./{g}")
		return dct
