from __future__ import annotations

import numpy as np
from qtpy.QtCore import QPoint, Qt, Signal
from qtpy.QtGui import QColor, QKeySequence, QMouseEvent
from qtpy.QtWidgets import (
	QFileDialog,
	QFrame,
	QHeaderView,
	QMenu,
	QMessageBox,
	QSlider,
	QTreeView,
	QWidget,
	QWidgetAction,
)

from . import pvAction, pvGroup, pvIcon, pvItemModel, pvLayer, pvLayerSymbology
from .colors import QICSColorDialog
from .layer_dlg import pvLayerSettings


class pvTreeView(QTreeView):
	requestFocus = Signal(list)

	def __init__(self, parent: QWidget = None) -> None:
		super().__init__(parent)
		self.setFrameStyle(QFrame.Plain)
		self.setHeaderHidden(True)
		self.header().setSectionResizeMode(QHeaderView.ResizeToContents)
		self.setSelectionMode(QTreeView.ExtendedSelection)
		self.setDragDropMode(QTreeView.InternalMove)  # DragDrop)
		self.setContextMenuPolicy(Qt.CustomContextMenu)
		self.customContextMenuRequested.connect(self.on_context_menu)

	def setModel(self, model: pvItemModel) -> None:
		if not isinstance(model, pvItemModel):
			raise TypeError(model)
		super().setModel(model)

	def on_context_menu(self, point: QPoint):
		self.menu.exec(self.viewport().mapToGlobal(point))

	def mousePressEvent(self, event: QMouseEvent):
		index = self.indexAt(event.position().toPoint())
		if not index.isValid():
			self.setCurrentIndex(self.indexAt(event.position().toPoint()))
		super().mousePressEvent(event)

	def keyPressEvent(self, event):
		if event.key() == Qt.Key_Delete:
			self.action_remove_selection.trigger()
		return super().keyPressEvent(event)

	def currentLayer(self):
		item = self.model.item(self.currentIndex())
		return item if isinstance(item, pvLayer) else None

	def selectedLayers(self) -> list[pvLayer]:
		selection = []
		for index in self.selectedIndexes():
			selection += self.model().layers(index)
		return selection

	@property
	def menu(self):
		# proper menu instanciation
		menu = QMenu("Edit", parent=self)
		menu.addAction(self.action_new_group)
		menu.addSeparator()
		# zoom to selection
		menu.addAction(self.action_focus_selection)
		menu.addSeparator()
		# color shortcut
		menu.addAction(self.action_paint_selection)
		# opacity slider
		menu.addAction(self.action_slider_opacity)
		# edit selection
		menu.addAction(self.action_edit_selection)
		menu.addSeparator()
		menu.addAction(self.action_export_selection)
		menu.addSeparator()
		# allow removal
		menu.addAction(self.action_remove_selection)
		return menu

	@property
	def action_new_group(self) -> pvAction:
		def callback():
			group = self.model().add(pvGroup(), parent=self.currentIndex())
			selection = list(
				filter(None, [el.internalPointer() for el in self.selectedIndexes()])
			)
			if selection:
				self.model().moveItems(selection, parent=group)

		return pvAction(
			text="New group",
			icon=pvIcon("mActionAddGroup"),
			callback=callback,
			parent=self,
		)

	@property
	def action_remove_selection(self) -> pvAction:
		def callback():
			n, m = len(self.selectedIndexes()), len(self.selectedLayers())
			if (
				QMessageBox.warning(
					self,
					"Remove items",
					f"Remove {n} item{'s' if n > 1 else ''} ({m} layer{'s' if m > 1 else ''}) ?",
					QMessageBox.Ok | QMessageBox.Cancel,
				)
				== QMessageBox.Ok
			):
				for row in self.selectedIndexes():
					self.model().remove(row)

		return pvAction(
			text="Delete",
			icon="mActionRemove",
			shortcut=QKeySequence.Delete,
			callback=callback,
			parent=self,
			enable=bool(self.selectedIndexes()),
		)

	@property
	def action_focus_selection(self) -> pvAction:
		def callback():
			bounds = [el.geometry.bounds for el in self.selectedLayers() if el.geometry]
			if bounds:
				bounds = np.array(bounds).reshape((-1, 6))  # reshape as numpy array
				bounds[np.abs(bounds) == 1e299] = np.nan  # filter pyvista's nans
				bounds = np.ravel(  # interleaves min/max boundaries to get largest bbox
					(np.nanmin(bounds, axis=0)[::2], np.nanmax(bounds, axis=0)[1::2]),
					order="F",
				).tolist()
				self.requestFocus.emit(bounds)
			else:
				self.requestFocus.emit(None)

		return pvAction(
			text="Zoom to",
			icon=pvIcon("mActionZoomTo"),
			callback=callback,
			parent=self,
		)

	@property
	def action_edit_selection(self) -> pvAction:
		def callback():
			pvLayerSettings(self.model(), self.selectedLayers()).open()

		return pvAction(
			text="Properties...",
			icon=pvIcon("symbology"),
			callback=callback,
			parent=self,
			enable=bool(self.selectedLayers()),
		)

	@property
	def action_export_selection(self) -> pvAction:
		def callback():
			layers = self.selectedLayers()
			if len(layers) > 1:
				prefix = QFileDialog.getExistingDirectory()
				if not prefix:
					return
			else:
				prefix = None
			for layer in layers:
				layer.save(prefix=prefix)

		return pvAction(
			text="Export data ...",
			callback=callback,
			parent=self,
			enable=bool(self.selectedLayers()),
		)

	@property
	def action_slider_opacity(self) -> QWidgetAction:
		# direct selected layers setter / helpers
		layers = self.selectedLayers()
		first = layers[0] if layers else None

		def change(value):
			for layer in layers:
				layer.symbology.opacity = float(value) / 100.0
			self.model().layersChanged.emit(layers)

		slider = QSlider(Qt.Horizontal)
		slider.setMinimum(0)
		slider.setMaximum(100)
		slider.setValue(
			int(
				float(first.symbology.opacity if first else pvLayerSymbology.opacity)
				* 100
			)
		)
		slider.valueChanged.connect(change)
		action = QWidgetAction(self)
		action.setDefaultWidget(slider)
		action.setText("opacity")
		action.setEnabled(bool(layers))
		return action

	@property
	def action_paint_selection(self) -> pvAction:
		# direct selected layers setter / helpers
		layers = self.selectedLayers()
		first = layers[0] if layers else None

		def callback():
			color = QICSColorDialog.getColor(
				initial=QColor(
					first.symbology.color if first else pvLayerSymbology.color
				)
			)
			if color:
				for layer in layers:
					layer.symbology.color = color.name()
				self.model().layersChanged.emit(layers)

		return pvAction(
			text="Change color ...",
			icon=pvIcon("color"),
			callback=callback,
			parent=self,
			enable=bool(self.selectedLayers()),
		)
