from __future__ import annotations

from qtpy.QtCore import Qt
from qtpy.QtGui import QKeySequence
from qtpy.QtWidgets import QMainWindow, QMenu, QMenuBar, QSplitter, QWidget

from ..core import pvItemModel, pvLayer, pvLoader, pvProject, pvSettings
from .action import pvAction
from .icons import pvIcon
from .loader_menu import pvLoaderMenu
from .project_menu import pvProjectMenu
from .renderer import pvRenderer
from .treeview import pvTreeView


class pvInterface(QMainWindow):
	settings = pvSettings()

	def __init__(
		self,
		parent: QWidget = None,
		flags: Qt.QWindowFlags = Qt.Window,
		*,
		project: str | None = None,
		dummy: bool = False,
	) -> None:
		super().__init__(parent, flags)

		# INITIALIZE WINDOW
		self.setWindowTitle(self.settings.value("name", "Viewer 3D"))
		self.setWindowIcon(pvIcon())
		self.setMenuBar(QMenuBar(parent=self))

		# CREATE UI COMPONENTS (WIDGETS)
		#   our 2 main components (tree view and 3D renderer)
		self.tree = pvTreeView(parent=self)
		self.renderer = pvRenderer(parent=self)
		splitter = QSplitter(Qt.Horizontal, parent=self)
		splitter.setChildrenCollapsible(False)
		splitter.addWidget(self.tree)
		splitter.addWidget(self.renderer)
		self.setCentralWidget(splitter)

		# INITIALIZE DATA MODEL
		#   how data are handled
		self.model = pvItemModel(parent=self)
		self.tree.setModel(self.model)  # view display model
		self.project = pvProject(self.model, parent=self)
		self.installEventFilter(self.project)  # project allows quit/close
		self.loader = pvLoader(self.model, self.tree.currentIndex)

		# CONNECT LOGIC
		self.tree.requestFocus.connect(self.renderer.reset_camera)
		# model <-> renderer
		self.model.layersAdded.connect(self.renderer.add_layers)
		self.model.layersRemoved.connect(self.renderer.remove_layers)
		self.model.layersChanged.connect(self.renderer.redraw_layers)
		# model <-> project
		self.model.layersAdded.connect(self.project.invalidate)
		self.model.layersRemoved.connect(self.project.invalidate)
		self.model.layersChanged.connect(self.project.invalidate)
		# project <-> window
		self.project.stateChanged.connect(self.updateTitle)

		# FINALIZE MENUBAR
		p = self.menuBar().addMenu(pvProjectMenu(self.project, parent=self))
		p.menu().addSeparator()
		p.menu().addAction(
			pvAction(
				"Exit",
				shortcut=QKeySequence("Ctrl+Q"),
				callback=self.close,
				parent=self,
			)
		)
		self.menuBar().addMenu(pvLoaderMenu(self.loader, parent=self))
		self.menuBar().addMenu(self.renderer.menu)
		if dummy:
			self.menuBar().addMenu(self.dummy_options)

		# END OF INITIALIZATION
		if project:  # load a project
			self.project.load(project)

	@property
	def dummy_options(self) -> QMenu:
		from pyvista import Box, Cone, Plane, Sphere

		menu = QMenu(self.tr("Dummy"), parent=self)
		menu.addAction(
			pvAction(
				"Add sphere",
				icon=pvIcon("mActionAddBasicCircle"),
				callback=lambda: self.model.add(
					pvLayer.from_memory(Sphere(), name="Dummy sphere"),
					parent=self.tree.currentIndex(),
				),
				parent=self,
			)
		)
		menu.addAction(
			pvAction(
				"Add cube",
				icon=pvIcon("mActionAddLayer"),
				callback=lambda: self.model.add(
					pvLayer.from_memory(Box(), name="Dummy cube"),
					parent=self.tree.currentIndex(),
				),
				parent=self,
			)
		)
		menu.addAction(
			pvAction(
				"Add cone",
				icon=pvIcon("mActionAddBasicTriangle"),
				callback=lambda: self.model.add(
					pvLayer.from_memory(Cone(), name="Dummy cone"),
					parent=self.tree.currentIndex(),
				),
				parent=self,
			)
		)
		menu.addAction(
			pvAction(
				"Add square",
				icon=pvIcon("mActionAddBasicRectangle"),
				callback=lambda: self.model.add(
					pvLayer.from_memory(Plane(), name="Dummy square"),
					parent=self.tree.currentIndex(),
				),
				parent=self,
			)
		)
		return menu

	def updateTitle(self, state: bool = True):
		title = "" if bool(state) else "* "
		title += self.project.title
		self.setWindowTitle(title)

	def closeEvent(self, event):
		# project allows/refuses close
		self.project.closeEvent(event)

	def show(self):
		self.setEnabled(True)
		self.setWindowState(Qt.WindowActive)
		super().show()
