from __future__ import annotations

import codecs
import pickle
from dataclasses import dataclass
from pathlib import Path
from typing import Any
from xml.etree.ElementTree import Element, SubElement

from numpy import fromstring
from pyvista import Texture
from qtpy.QtGui import QColor


@dataclass
class pvLayerSymbology:
	color: str = "white"
	edge_color: str = "white"
	opacity: float = 1.0
	show_edges: bool = False
	render_points_as_spheres: bool = True
	point_size: float = 5.0
	render_lines_as_tubes: bool = False
	line_width: float = 2.0
	style: str = "Surface"
	_texture: str = ""  # pv.add_mesh requires a pv.Texture not a file !

	def __setattr__(self, name: str, value: Any):
		# handle texture (cast source to array, track reference)
		if name == "_texture":
			self.__dict__.pop("texture", None)
		# handle colors (multiple formats allowed ...)
		if "color" in name:
			if QColor(value).isValid():
				value = QColor(value).name()
			else:
				rgba = fromstring(value, sep=" ")
				if QColor(*rgba * 255).isValid():
					value = QColor(*rgba * 255).name()
				elif QColor(*rgba).isValid():
					value = QColor(*rgba).name()
				else:
					raise ValueError()
		# safe type-casting (in XML reading everything comes as str)
		v_type = type(self.__getattribute__(name))
		if (
			name == "_texture" and value
		):  # texture should be rendered over neutral color:
			self.color = "white"
		elif v_type is bool:
			value = (
				False
				if str(value).lower() in ("n", "no", "none", "false", "null", "0")
				else bool(value)
			)
		elif v_type is not type(None):
			value = v_type(value)
		super().__setattr__(name, value)

	@property
	def texture(self) -> Texture:
		if not self._texture:
			return None
		if isinstance(self._texture, str | Path):
			tex = Texture(self._texture)
			if tex.n_components > 3:
				tex = Texture(tex.to_array()[:, :, :3])
			if Path(self._texture).suffix.lower().startswith(".tif"):
				tex = tex.flip_y()
			return tex

		return Texture(self._texture)

	@property
	def kwargs(self):
		attributes = [
			attr for attr in self.__dataclass_fields__ if not attr.startswith("_")
		]
		return {attr: getattr(self, attr) for attr in attributes}

	def to_xml(self, include_data: bool = False) -> Element:
		attribs = {k: str(v) for k, v in self.kwargs.items()}
		xml = Element("Symbology", **attribs)
		if self.texture and include_data:
			tex = SubElement(xml, "Texture")
			tex.text = codecs.encode(
				pickle.dumps(self.texture.to_image()), "base64"
			).decode()
		else:
			xml.set("texture", str(self._texture))
		return xml

	@staticmethod
	def from_xml(el: Element):
		if el is None:
			return pvLayerSymbology()
		kwargs = dict(el.attrib)
		kwargs["_texture"] = kwargs.pop("texture", "")
		tex = el.find("Texture")
		if tex is not None:
			kwargs["_texture"] = pickle.loads(
				codecs.decode(tex.text.encode(), "base64")
			)
		return pvLayerSymbology(**kwargs)
