from __future__ import annotations

from functools import cached_property
from typing import ClassVar

from qgis import processing
from qgis.core import (
	QgsApplication,
	QgsProcessingProvider,
	QgsRasterLayer,
	QgsVectorLayer,
)
from qgis.PyQt.QtCore import QEvent, QObject, Qt
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QToolBar, QWidget

from .pvgis import pvIcon, pvInterface, pvLayer, pvSettings
from .settings import qvSettings
from .toolbox import MeshifyRaster, MeshifyVector, VectorifyMesh, qvToolbox


class qvGIS(QObject):
	"""pvGIS - QGIS plugin instance."""

	toolbox: QgsProcessingProvider = qvToolbox()
	viewer: QWidget = pvInterface()

	converters: ClassVar = {
		QgsVectorLayer: ":".join((qvToolbox.id(), MeshifyVector.name())),
		QgsRasterLayer: ":".join((qvToolbox.id(), MeshifyRaster.name())),
		pvLayer: ":".join((qvToolbox.id(), VectorifyMesh.name())),
	}

	def __init__(self, iface):
		"""Constructor (called at plugin load).

		:param iface: An interface instance that will be passed to this class
		    which provides the hook by which you can manipulate the QGIS
		    application at run time.
		:type iface: QgsInterface
		"""
		# Call the Qt.QObject constructor
		super().__init__(QgsApplication.instance())
		self.iface = iface
		self.name = self.objectName = self.settings.get("name")
		self.viewer.installEventFilter(self)
		self.iface.layerTreeView().selectionModel().selectionChanged.connect(
			self.qgs_layer_changed
		)
		self.viewer.tree.selectionModel().selectionChanged.connect(self.pv_item_changed)

	@property
	def pvsettings(self) -> pvSettings:
		return pvSettings()

	@property
	def settings(self) -> qvSettings:
		return qvSettings()

	def initProcessing(self):
		"""Register the plugin toolbox (processings provider)"""
		self.toolbox = qvToolbox()  # force update on plugin reloading
		QgsApplication.instance().processingRegistry().addProvider(self.toolbox)

	def initGui(self):
		"""Called at plugin start (not load)"""

		self.initProcessing()

		# MENU entries
		#   inside Extensions > pvGIS
		self.qv_menu = self.iface.pluginMenu().addMenu(
			pvIcon(), self.settings.value("name")
		)
		self.qv_menu.addAction(self.show)
		self.qv_menu.addSeparator()
		self.qv_menu.addAction(self.about)
		#   inside pvGIS > pvGIS
		self.pv_menu = self.viewer.menuBar().addMenu("QGIS")
		self.pv_menu.addAction(self.send_to_2d)
		# TOOLBAR shortcuts
		self.iface.addToolBarIcon(self.show)
		# self.iface.addToolBarIcon(self.send_to_3d)
		self.layer_toolbar = next(
			(
				w
				for w in self.iface.layerTreeView().parent().children()
				if isinstance(w, QToolBar)
			),
			None,
		)
		if self.layer_toolbar:
			self.layer_toolbar.addAction(self.send_to_3d)

	def unload(self):
		"""Removes the plugin menus/toolbars/toolboxes/events"""
		# remove the menu entries
		self.iface.pluginMenu().removeAction(self.qv_menu.menuAction())
		# remove toolbar shortcuts
		self.iface.pluginToolBar().removeAction(self.show)
		# self.iface.pluginToolBar().removeAction(self.send_to_3d)
		if self.layer_toolbar:
			self.layer_toolbar.removeAction(self.send_to_3d)
		# remove the toolbox
		if self.toolbox:
			QgsApplication.instance().processingRegistry().removeProvider(self.toolbox)
		# delete
		self.deleteLater()

	def eventFilter(self, sender: QObject, event: QEvent) -> bool:
		# safe guard around quit/close events
		if event.type() == QEvent.Close and sender is self.viewer:
			self.viewer.hide()
			event.ignore()
			return True
		return False

	def qgs_layer_changed(self):
		enable = False
		for layer in self.iface.layerTreeView().selectedLayers():
			if layer == self.viewer.tree.model().root:
				enable = False
			elif type(layer) in self.converters:
				enable = True
			else:
				continue
			break
		self.send_to_3d.setEnabled(enable)

	def pv_item_changed(self):
		enable = False
		for layer in self.viewer.tree.selectedLayers():
			if layer.is_temp or not layer.is_valid:
				continue
			if type(layer) in self.converters:
				enable = True
				break
		self.send_to_2d.setEnabled(enable)

	@property
	def loader(self):
		return self.viewer.loader

	@cached_property
	def about(self) -> QAction:
		"""Display plugin details"""

		def callback():
			dlg = QWidget(self.iface.mainWindow())
			dlg.setWindowIcon(pvIcon())

			content = f"<b>{self.name}</b><br>"
			content += f"<i>{self.settings.get('description')}</i><br><br>"
			content += f"{self.settings.get('about')}<br><br>"

			content += f"<b>Author</b> : <a href=mailto:{self.settings.get('email')}>{self.settings.get('author')}</a><br>"
			if homepage := self.settings.get("homepage", None):
				content += f"<b>Homepage</b> : <a href={homepage}>GitLab</a><br>"
			content += f"<b>Source code</b> : <a href={self.settings.get('repository')}>GitLab</a><br>"
			if tracker := self.settings.get("tracker", None):
				content += f"<b>Repport issue</b> : <a href={tracker}>GitLab</a><br>"
			if doc := self.settings.get("documentation", None):
				content += f"<b>Documentation</b> : <a href={doc}>GitLab</a><br>"

			content += f"<br><b>Version</b> : {self.settings.get('version')}<br>"
			content += f"<i><b>Requires</b> QGIS &#x02265; {self.settings.get('qgisMinimumVersion')}</i><br>"
			if plugins := self.settings.get("plugin_dependencies"):
				content += f"<i><b>Depends on</b> : {plugins.strip(',')}</i><br>"
			if deps := self.settings.get("pip_dependencies"):
				content += f"<i><b>Requirements</b> : {deps.strip(',')}</i><br>"

			QMessageBox.about(dlg, "About", content)
			dlg.deleteLater()

		action = QAction(pvIcon("helpContents"), f"About {self.name}", parent=self)
		action.triggered.connect(callback)
		return action

	@cached_property
	def show(self) -> QAction:
		def callback():
			self.viewer.setEnabled(True)
			self.viewer.setWindowState(Qt.WindowActive)
			self.viewer.show()

		action = QAction(pvIcon(), "Show 3D viewer", parent=self)
		action.triggered.connect(callback)
		return action

	@cached_property
	def send_to_3d(self) -> QAction:
		"""Import a QGIS layer into the 3D viewer"""

		def callback():
			for layer in self.iface.layerTreeView().selectedLayers():
				if converter := self.converters.get(type(layer), None):
					processing.run(
						converter,
						{"INPUT": layer, "OUTPUT": "TEMPORARY_OUTPUT"},
					)
			self.viewer.show()

		action = QAction(pvIcon("mActionSendTo3D"), f"Send to {self.name}", parent=self)
		action.triggered.connect(callback)
		action.setEnabled(False)
		return action

	@cached_property
	def send_to_2d(self) -> QAction:
		"""Import a 3D layer into QGIS"""

		def callback():
			for layer in self.viewer.tree.selectedLayers():
				if layer.is_temp or not layer.is_valid:
					continue
				if converter := self.converters.get(type(layer), None):
					processing.run(
						converter,
						{"INPUT": layer.source, "OUTPUT": "memory:"},
					)

		action = QAction(pvIcon("mAddToProject"), "Send to QGIS", parent=self)
		action.triggered.connect(callback)
		action.setEnabled(False)
		return action
