import math
import operator

class Raster(object):

   def __init__(self, xll, yll, dc, dr, nc, nr, angle=0.0, nodata=-9999, data=[]):
   
      self.xll = xll
      self.yll = yll
      self.dc = dc
      self.dr = dr
      self.nc = nc
      self.nr = nr
      self.angle = angle
      self.nodata = nodata
      
      self.data = []
      # check data size
      if len(data) == 0:
         for i in range(nr):
            self.data.append([nodata for j in range(nc)])      
      else:
         if len(data) != self.nr:
            raise ValueError('Unexpected number of raster rows: %d instead of %d' % (len(data), self.nr))
         for row in range(self.nr):
            if len(data[row]) != self.nc:
               raise ValueError('Unexpected number of raster columns: %d instead of %d' % (len(data[row]), self.nc))
         self.data = data
      
   def num_cells(self):
      return self.nc * self.nr
      
   def cell_value(self, idx):
   
      r_idx, c_idx = self.cell(idx)
      
      return self.data[r_idx][c_idx]
      
   def set_cell_value(self, idx, value):
     
      r_idx, c_idx = self.cell(idx)
      
      self.data[r_idx][c_idx] = value 
      
   def cell_center(self, idx):
      
      r_idx, c_idx = self.cell(idx)
      
      dc = ((0.5 + c_idx) * self.dc)
      dr = ((0.5 + r_idx) * self.dr)
      
      return (self.xll + (dc * math.cos(self.angle) - dr * math.sin(self.angle)),
              self.yll + (dr * math.cos(self.angle) + dc * math.sin(self.angle)))
              
   def cell(self, idx):
   
      r_idx = int(math.floor(idx/self.nc))
      c_idx = int(idx % self.nc)
      
      if r_idx < 0 or r_idx >= self.nr:
         raise IndexError('raster row index out of range: %d, num. rows = %d' % (r_idx, self.nr))
      if c_idx < 0 or c_idx >= self.nc:
         raise IndexError('raster column index out of range: %d, num. columns = %d' % (c_idx, self.nc))
         
      return (r_idx, c_idx)
      
   def idx(self, cell):
      return cell[0] * self.nc + cell[1]
      
   def save_as_prm(self, filename):
   
      prm = open(filename, 'w+')
      
      prm.write('# This file was automatically generated by %s\n' % __file__)
      prm.write('# The following metadata can be copied to an promaides .ilm\n')
      prm.write('# file to use this raster with promaides.\n\n')
      
      prm.write('#!GENERAL = <SET>\n')
      prm.write('#\t$NX          = %d\n' % self.nc)
      prm.write('#\t$NY          = %d\n' % self.nr)
      prm.write('#\t$LOWLEFTX    = %f\n' % self.xll)
      prm.write('#\t$LOWLEFTY    = %f\n' % self.yll)
      prm.write('#\t$ELEMWIDTH_X = %f\n' % self.dc)
      prm.write('#\t$ELEMWIDTH_Y = %f\n' % self.dr)
      prm.write('#\t$NOINFOVALUE = %f\n' % self.nodata)
      
      # CAUTION! The positive rotation direction is defined opposite in ProMaIDEs
      angle = self.angle / math.pi * 180.0
      prm.write('#\t$ANGLE       = %f\n' % -angle)
      
      prm.write('#</SET>\n\n')
      
      prm.write('# --- Value description --- #\n')
      prm.write('# element-nr. [-]\n')
      prm.write('# geod. height [m]\n')
      prm.write('# material id [-] (must match one of the ids in the materials file)\n')
      prm.write('# initial waterlevel [m]\n')
      prm.write('# boundary condition [bool] (whether the cell is a boundary)\n')
      prm.write('# stationary boundary condition [bool] (whether the boundary is stationary or instationary)\n')
      prm.write('# boundary condition value [-] (if stationary is true, unit is according to "boundary type", otherwise id of hydrograph)\n')
      prm.write('# boundary type [-] (whether the boundary condition value unit is "point" [m3/s], "area" [m3/(s m2)], "length" [m], or "waterlevel" [m])\n\n')
      
      prm.write('!BEGIN\n')
      for i in range(self.num_cells()):
         prm.write('%d\t%f\t7\t0.0\tfalse\tfalse\t0\n' % (i, self.cell_value(i)))
      prm.write('!END\n')
      
      prm.close()