from PyQt4.QtCore import *
from PyQt4.QtGui import *

from qgis.core import *

from ui_promaides_dem_export import PluginDialog

from rasterinterpolator import RasterInterpolator

import resources
import os, math

class PromaidesDEMExport(object):

   ILM_TMPL_FILE = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'ilm_template.txt')
   
   def __init__(self, iface):
   
      self.iface = iface
      
      self.dialog = None
      self.previewLayer = None
      
      QgsMapLayerRegistry.instance().addMapLayers([self.previewLayer])
      
   def initGui(self):
   
      self.action = QAction(QIcon(":/plugins/promaides_tools/icon.png"), u'ProMaIDes DEM Export', self.iface.mainWindow())
      QObject.connect(self.action, SIGNAL("triggered()"), self.invokeDialog)
      
      self.iface.addToolBarIcon(self.action)
      self.iface.addPluginToMenu('&ProMaIDes Tools', self.action)
      
   def unload(self):
   
      self.iface.removePluginMenu('&ProMaIDes Tools', self.action)
      self.iface.removeToolBarIcon(self.action)
      
   def invokeDialog(self):
   
      self.previewLayer = QgsVectorLayer('Polygon', 'ProMaIDes Raster', 'memory')
      self.previewLayer.setCrs(self.iface.mapCanvas().mapRenderer().destinationCrs())
      QgsMapLayerRegistry.instance().addMapLayer(self.previewLayer)
   
      self.dialog = PluginDialog(self.iface, self.iface.mainWindow())
      self.dialog.setModal(False)
      
      self.dialog.accepted.connect(self.onDialogAccepted)
      self.dialog.rejected.connect(self.onDialogRejected)
      
      self.dialog.rasterAdded.connect(self.addRasterBounds)
      self.dialog.rasterUpdated.connect(self.updateRasterBounds)
      self.dialog.rasterRemoved.connect(self.removeRasterBounds)
      
      self.dialog.show()
            
   def interpolate(self, baseLayer, band, outRaster, interpolMode):
   
      interpol = RasterInterpolator(baseLayer, interpolMode, band)
      for i in range(outRaster.num_cells()):
      
         if self.exportCanceled:
            return
            
         point = outRaster.cell_center(i)
         value = interpol.interpolate(QgsPoint(point[0], point[1]))
         
         if value:
            outRaster.set_cell_value(i, value)
         else:
            outRaster.set_cell_value(i, outRaster.nodata)
            
         self.progress.setValue(self.progress.value() + 1)

   
   def addRasterBounds(self, id, polygon):
   
      dp = self.previewLayer.dataProvider()
      newPoly = QgsFeature(id)
      newPoly.setGeometry(polygon)
      dp.addFeatures([newPoly])
      
      self.previewLayer.updateExtents()
      self.previewLayer.triggerRepaint()
      # self.iface.mapCanvas().refresh()
      
   def updateRasterBounds(self, id, polygon):
   
      self.previewLayer.dataProvider().changeGeometryValues({id: polygon})
      
      self.previewLayer.updateExtents()
      self.previewLayer.triggerRepaint()
      # self.iface.mapCanvas().refresh()
      
   def removeRasterBounds(self, id):
   
      self.previewLayer.dataProvider().deleteFeatures([id])
      
      self.previewLayer.updateExtents()
      self.previewLayer.triggerRepaint()
      # self.iface.mapCanvas().refresh()
   
   
   def onDialogAccepted(self):

      baseLayer = self.dialog.baseLayer()
      band = self.dialog.band()
      rasters = self.dialog.rasters()  # list of tuples (raster, interpolMode, filename)
      
      num_cells = 0
      for raster, interpolMode, filename in rasters:
         num_cells += raster.num_cells()
      
      # if there are rasters to interpolate and export
      if num_cells > 0:
      
         text = ''
         if len(rasters) > 1:
            text = 'Interpolating and exporting %d rasters ...' % len(rasters)
         else:
            text = 'Interpolating and exporting raster ...'
            
         self.progress = QProgressDialog(text, 'Abort', 0, num_cells, self.iface.mainWindow())
         self.progress.setWindowTitle('ProMaIDes DEM Export')
         self.progress.canceled.connect(self.cancelExport)
         self.exportCanceled = False
         
         self.progress.show()
         self.progress.setValue(1)
            
         for raster, interpolMode, filename in rasters:          
            self.interpolate(baseLayer, band, raster, interpolMode)
            if self.exportCanceled:
               break
               
            try:
               raster.save_as_prm(filename)
            except IOError:
               QMessageBox.critical(self.iface.mainWindow(), 'I/O Error', 'An I/O error occured during\nraster export to file\n\n%s' % filename)
         
         self.progress.close()
         self.progress = None
         
         if self.dialog.createIlmFile():
            standardLocation = QDesktopServices.storageLocation(QDesktopServices.DocumentsLocation)
            filename = QFileDialog.getSaveFileName(self.iface.mainWindow(), 'Save .ilm File', standardLocation)
            if filename != '':
               try:
                  self.writeIlmFile(filename, rasters)
               except IOError:
                  QMessageBox.critical(self.iface.mainWindow(), 'I/O Error', 'An I/O error occured during\n.ilm export to file\n\n%s' % filename)
      
      
      QgsMapLayerRegistry.instance().removeMapLayer(self.previewLayer.id())
      
      del self.dialog
      del self.previewLayer
    
    
   def cancelExport(self):
   
      self.exportCanceled = True
   
   
   def onDialogRejected(self):
   
      QgsMapLayerRegistry.instance().removeMapLayer(self.previewLayer.id())
      
      del self.dialog
      del self.previewLayer
   
   
   def writeIlmFile(self, filename, rasters):
   
      ilm_tmpl_file = open(self.ILM_TMPL_FILE, 'r')
      ilm_tmpl = ilm_tmpl_file.read()
      ilm_tmpl_file.close()
      
      ilm = open(filename, 'w+')
      
      # write general settings to ilm file
      ilm.write(ilm_tmpl.format(len(rasters)))
      
      counter = 0
      for raster, interpolMode, filename in rasters:
      
         # CAUTION! The positive rotation direction is defined opposite in ProMaIDEs
         angle = -raster.angle / math.pi * 180.0
      
         ilm.write('!$BEGINFPMODEL = %d "raster-%d"\n' % (counter, counter+1))
         ilm.write('!GENERAL= <SET> \n')
         ilm.write('  $NX          = %d\n' % raster.nc)
         ilm.write('  $NY          = %d\n' % raster.nr)
         ilm.write('  $LOWLEFTX    = %f\n' % raster.xll)
         ilm.write('  $LOWLEFTY    = %f\n' % raster.yll)
         ilm.write('  $ELEMWIDTH_X = %f\n' % raster.dc)
         ilm.write('  $ELEMWIDTH_Y = %f\n' % raster.dr)
         ilm.write('  $NOINFOVALUE = %f\n' % raster.nodata)
         ilm.write('  $ANGLE       = %f\n' % angle)
         ilm.write('</SET> \n')

         ilm.write('!2DOUTPUT = "./results"\n')
         ilm.write('!FLOODPLAINFILE = "%s"\n' % filename)
                        
         ilm.write('!LIMITS = <SET>  # numerical limits for 2D simulation\n')
         ilm.write('  $RTOL = 5e-7   # relative tolerances   [optional, standard value = 5e-7]\n')
         ilm.write('  $ATOL = 5e-7   # absolute tolerances   [optional, standard value = 5e-7]\n')
         ilm.write('  $WET  = 0.01   # wet and dry parameter [optional, standard value = 1e-2]\n')
         ilm.write('</SET>\n')                                                                                      	    
         ilm.write('!$ENDFPMODEL\n\n')
         
         counter += 1
         
      ilm.write('!$ENDDESCRIPTION\n')
      ilm.close()
      
         
      
   