"""
/***************************************************************************
Name                 : Profile from Line
Description          : samples rasters along vector lines
Date                 : 05/Aug/10 
copyright            : (C) 2010 by Ricardo Garcia Silva
email                : ricardo.garcia.silva@gmail.com 
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

#TODO 

# Import other python modules
from shapely.wkb import loads
from shapely.wkb import dumps
import shapely.geos 

# Import the PyQt and QGIS libraries
from PyQt4.QtCore import * 
from PyQt4.QtGui import *
from qgis.core import *

# Initialize Qt resources from file resources.py
import resources

# Import the code for the dialog
from ProfileFromLineDialog import ProfileFromLineDialog

class ProfileFromLine: 

    def __init__(self, iface):
        # Save reference to the QGIS interface
        self.iface = iface
        self.legIface = iface.legendInterface()
        self.layerRegistry = QgsMapLayerRegistry.instance()
        self.canvas = self.iface.mapCanvas()
        self.qgisSettings = QSettings()
        self.shapelyVersionError = "The Shapely python package currently" \
                                   " installed doesn't support the features" \
                                   " needed by this plugin. Please install" \
                                   " version 1.2 or later and make sure" \
                                   " your GEOS library is also at version" \
                                   " 3.2 or later."
        self.shapelyGEOSError = "The shapely python package uses the GEOS" \
                                " library. Your current version of GEOS" \
                                " doesn't support the features needed by" \
                                " this plugin. Please install a more" \
                                " recent version of GEOS (version 3.2 or" +\
                                " later)."

    def initGui(self):  
        self.action = QAction(QIcon(":/plugins/ProfileFromLine/icon.png"), \
            "Profile from line", self.iface.mainWindow())
        self.toggle_plugin_availability()
        QObject.connect(self.action, SIGNAL("activated()"), self.run) 
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&Profile from line", self.action)
        QObject.connect(self.layerRegistry,
                        SIGNAL("layerWasAdded(QgsMapLayer *)"),
                        self.toggle_plugin_availability)
        QObject.connect(self.layerRegistry,
                        SIGNAL("layerWillBeRemoved(QString)"),
                        self.toggle_plugin_availability)

    def unload(self):
        self.iface.removePluginMenu("&Profile from line",self.action)
        self.iface.removeToolBarIcon(self.action)

    def run(self): 
        try:
            geosCAPIVersion = shapely.geos.geos_capi_version
        except AttributeError:
            geosCAPIVersion = (-1,-1,-1)
        if int(geosCAPIVersion[0]) >= 1:
            if int(geosCAPIVersion[1]) >= 6:
                availableLayers = self._get_loaded_layers()
                dlg = ProfileFromLineDialog(availableLayers["vectorLine"],
                        availableLayers["raster"], self.create_profile_layer)
                result = dlg.exec_() 
            else:
                QMessageBox.warning(None, "ProfileFromLine: Error",
                                    self.shapelyGEOSError)
        else:
            QMessageBox.warning(None, "ProfileFromLine: Error", 
                                self.shapelyVersionError)

    def _get_loaded_layers(self):
        """
        Return a dictionary with the loaded vector and raster layers.
        
        Only vectors of type 'line' are returned.
        """

        availableLayers = {"vectorLine" : [], "raster" : []}
        loadedLayers = self.legIface.layers()
        for layerObj in loadedLayers:
            layerType = layerObj.type()
            if layerType == 0:
                if layerObj.geometryType() == 1: # it's a line
                    availableLayers["vectorLine"].append(layerObj)
            elif layerType == 1: # it's a raster layer
                availableLayers["raster"].append(layerObj)
        return availableLayers

    def create_profile_layer(self, linesLayer, rasterLayers, 
            pointInterval, useSelectedFeatures, progressBar):
        """
        Create the new point layer.
        """

        print("create_profile_layer method called.")
        progressBar.setVisible(True)
        progressBar.setValue(0)
        linesProvider = linesLayer.dataProvider()
        newFields, fieldMapper, rasterFieldsStart = self._get_new_fields(
                linesProvider, rasterLayers)
        print("newFields: %s" % newFields)
        print("fieldMapper: %s" % fieldMapper)
        print("rasterFieldsStart: %s" % rasterFieldsStart)
        projectionSettingKey = "Projections/defaultBehaviour"
        oldProjectionSetting = self.qgisSettings.value(projectionSettingKey)
        self.qgisSettings.setValue(projectionSettingKey, "useGlobal")
        self.qgisSettings.sync()
        #pointLayer = QgsVectorLayer("Point", "temporary_points", "memory")
        pointLayerURI = "Point?crs=%s&index=yes" % (linesLayer.crs().toProj4())
        for field in newFields:
            fieldMap = {2: "integer", 6: "double", 10 : "string"}
            fieldName = fieldMap.get(field.type())
            pointLayerURI += "&field=%s:%s" % (field.name(), fieldName)
        pointLayer = QgsVectorLayer(pointLayerURI, "temporary_points", "memory")
        pointLayer.setCrs(linesLayer.crs())
        self.qgisSettings.setValue(projectionSettingKey, oldProjectionSetting)
        pointProvider = pointLayer.dataProvider()
        pointProvider.addAttributes(newFields)
        lineFeat = QgsFeature()
        allLineAttrs = linesProvider.attributeIndexes()
        linesProvider.select(allLineAttrs)
        lineCounter = 0
        pointNum = 0
        currentStep = 10
        progressBar.setValue(currentStep)
        remainingSteps = 100 - currentStep
        numberOfLines = linesProvider.featureCount()
        stepInterval = int(remainingSteps / numberOfLines)
        if useSelectedFeatures:
            featList = linesLayer.selectedFeatures()
            for lineFeat in featList:
                pointNum = self.create_points_from_line(lineFeat, pointLayer, 
                                pointProvider, pointInterval, pointNum, fieldMapper, 
                                rasterLayers, rasterFieldsStart)
                currentStep += stepInterval
                progressBar.setValue(currentStep)
        else:
            while linesProvider.nextFeature(lineFeat):
                pointNum = self.create_points_from_line(lineFeat, pointLayer, pointProvider,
                                                        pointInterval, pointNum, fieldMapper, 
                                                        rasterLayers, rasterFieldsStart)
                currentStep += stepInterval
                progressBar.setValue(currentStep)
        pointLayer.updateExtents()
        self.layerRegistry.addMapLayer(pointLayer, True)
        progressBar.setValue(100)
        print("create_profile_layer method exiting.")

    def create_points_from_line(self, lineFeat, pointLayer, pointProvider,
                                pointInterval, pointNum, fieldMapper, 
                                rasterLayers, rasterFieldsStart):
        shapelyGeom = loads(lineFeat.geometry().asWkb())
        attrMap = lineFeat.attributeMap()
        accumDist = 0
        while accumDist <= shapelyGeom.length:
            newFeat = self.create_new_point(pointLayer, accumDist,
                                            pointNum, attrMap, shapelyGeom, 
                                            fieldMapper, rasterLayers, 
                                            rasterFieldsStart)
            pointProvider.addFeatures([newFeat])
            accumDist += pointInterval
            pointNum += 1
        lastLineFeat = self.create_new_point(pointLayer, shapelyGeom.length,
                                             pointNum, attrMap, 
                                             shapelyGeom, fieldMapper, 
                                             rasterLayers, rasterFieldsStart, 
                                             normalized=False)
        pointProvider.addFeatures([lastLineFeat])
        pointNum += 1
        return pointNum

    def _get_new_fields(self, linesProvider, rasterLayers):
        """
        Construct a new list containing the field definitions for the new point layer.

        Returns: A tuple containing the new fields list as first item,
                 a dictionary with the mappings between the id of the
                 fields from the line layer and the new id of the same
                 fields in the points layer, and an integer specifying
                 the number of the first field that holds raster values.
        """

        newFields = [QgsField("pointNum", QVariant.String),
                     QgsField("accumDst", QVariant.Double)]
        counter = 2
        linesFields = linesProvider.fields()
        newFieldsMapping = dict()
        for numFieldLine, field in linesFields.iteritems():
            newFields.append(QgsField(field.name(), field.type(), field.typeName(), field.length(), field.precision()))
            newFieldsMapping[numFieldLine] = counter
            counter +=1
        rasterFieldsStart = counter
        for rLayer in rasterLayers:
            newFields.append(QgsField(rLayer.name(), QVariant.Double))
            counter +=1
        return (newFields, newFieldsMapping, rasterFieldsStart)

    def create_new_point(self, pointLayer, distanceFromLine, pointNumber, 
                        lineAttrMap, shapelyLine, fieldMapper, rasterLayers,
                        rasterFieldsStart, normalized=False):
        """
        Create a new point feature.
        """

        newFeat = QgsFeature()
        shapelyPoint = shapelyLine.interpolate(distanceFromLine, normalized)
        newPoint = dumps(shapelyPoint)
        newGeom = QgsGeometry()
        newGeom.fromWkb(newPoint)
        newFeat.setGeometry(newGeom)
        newFeat.addAttribute(0, QVariant(pointNumber))
        newFeat.addAttribute(1, QVariant(distanceFromLine))
        for key,attr in lineAttrMap.iteritems():
            newFeat.addAttribute(fieldMapper[key], attr)
        rasterFieldsCounter = 0
        for rLayer in rasterLayers:
            noDataValue, validNoData = rLayer.noDataValue()
            if validNoData:
                rasterValue = noDataValue
            else:
                #assign an arbitrary value of 9999 for now
                rasterValue = 9999
            result, identifyDict = rLayer.identify(newGeom.asPoint())
            if result:
                for bandName, pixelValue in identifyDict.iteritems():
                    # only accepting the first band's value at the moment
                    if str(bandName) == rLayer.bandName(1):
                        #if "out of extent" in strValue or "null" not in strValue:
                        try:
                            rasterValue = float(pixelValue)
                        except ValueError:
                            pass
            newFeat.addAttribute(rasterFieldsStart + rasterFieldsCounter, QVariant(rasterValue))
            rasterFieldsCounter += 1
        return newFeat

    def toggle_plugin_availability(self, arg=None):
        numLinesLayers = len(self._get_loaded_layers()["vectorLine"])
        if numLinesLayers == 0:
            self.action.setEnabled(False)
        else:
            self.action.setEnabled(True)
