/*
 * Decompiled with CFR 0.152.
 */
package org.h2gis.graalvm;

import java.lang.reflect.Field;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.sql.DataSource;
import org.graalvm.nativeimage.IsolateThread;
import org.graalvm.nativeimage.c.function.CEntryPoint;
import org.graalvm.nativeimage.c.type.CCharPointer;
import org.graalvm.nativeimage.c.type.CTypeConversion;
import org.graalvm.word.WordBase;
import org.graalvm.word.WordFactory;
import org.h2.Driver;
import org.h2gis.functions.factory.H2GISDBFactory;
import org.h2gis.functions.factory.H2GISFunctions;
import org.h2gis.graalvm.ResultSetWrapper;
import org.h2gis.utilities.JDBCUtilities;
import sun.misc.Unsafe;

public class GraalCInterface {
    private static final Logger LOGGER = Logger.getLogger(GraalCInterface.class.getName());
    private static final Map<Long, Connection> connections = new ConcurrentHashMap<Long, Connection>();
    private static final Map<Long, Statement> statements = new ConcurrentHashMap<Long, Statement>();
    private static final Map<Long, ResultSet> results = new ConcurrentHashMap<Long, ResultSet>();
    private static final AtomicLong handleCounter = new AtomicLong(1L);
    private static final ThreadLocal<String> lastError = new ThreadLocal();
    private static final Unsafe unsafe = GraalCInterface.getUnsafe();

    @CEntryPoint(name="h2gis_get_last_error")
    public static CCharPointer h2gisGetLastError(IsolateThread thread) {
        String error = lastError.get();
        lastError.remove();
        if (error == null) {
            error = "";
        }
        return CTypeConversion.toCString((CharSequence)error).get();
    }

    @CEntryPoint(name="h2gis_connect")
    public static long h2gisConnect(IsolateThread thread, CCharPointer filePathPointer, CCharPointer usernamePointer, CCharPointer passwordPointer) {
        if (filePathPointer.isNull() || usernamePointer.isNull() || passwordPointer.isNull()) {
            GraalCInterface.logAndSetError("Null pointer received in connection parameters", null);
            return 0L;
        }
        try {
            String filePath = CTypeConversion.toJavaString((CCharPointer)filePathPointer);
            String username = CTypeConversion.toJavaString((CCharPointer)usernamePointer);
            String password = CTypeConversion.toJavaString((CCharPointer)passwordPointer);
            String url = "jdbc:h2:" + filePath;
            Properties properties = new Properties();
            properties.setProperty("url", url);
            properties.setProperty("user", username);
            properties.setProperty("password", password);
            Connection connection = JDBCUtilities.wrapSpatialDataSource((DataSource)H2GISDBFactory.createDataSource((Properties)properties)).getConnection();
            long handle = handleCounter.getAndIncrement();
            connections.put(handle, connection);
            return handle;
        }
        catch (Exception e) {
            GraalCInterface.logAndSetError("Failed to open H2 connection", e);
            return 0L;
        }
    }

    @CEntryPoint(name="h2gis_load")
    public static long h2gisLoad(IsolateThread thread, long connectionHandle) {
        Connection conn = connections.get(connectionHandle);
        if (conn == null) {
            GraalCInterface.logAndSetError("Invalid connection handle: " + connectionHandle, null);
            return 0L;
        }
        try {
            H2GISFunctions.load((Connection)conn);
            return 1L;
        }
        catch (Exception e) {
            GraalCInterface.logAndSetError("Failed to load H2GIS functions", e);
            return 0L;
        }
    }

    @CEntryPoint(name="h2gis_fetch")
    public static long h2gisFetch(IsolateThread thread, long connectionHandle, CCharPointer queryPointer) {
        if (queryPointer.isNull()) {
            GraalCInterface.logAndSetError("Null pointer received for query", null);
            return 0L;
        }
        Connection conn = connections.get(connectionHandle);
        if (conn == null) {
            GraalCInterface.logAndSetError("Invalid connection handle: " + connectionHandle, null);
            return 0L;
        }
        try {
            String query = CTypeConversion.toJavaString((CCharPointer)queryPointer);
            Statement stmt = conn.createStatement(1004, 1007);
            ResultSet rs = stmt.executeQuery(query);
            long handle = handleCounter.getAndIncrement();
            statements.put(handle, stmt);
            results.put(handle, rs);
            return handle;
        }
        catch (Exception e) {
            GraalCInterface.logAndSetError("Failed to execute query", e);
            return 0L;
        }
    }

    @CEntryPoint(name="h2gis_execute")
    public static int h2gisExecute(IsolateThread thread, long connectionHandle, CCharPointer queryPointer) {
        int n;
        block10: {
            if (queryPointer.isNull()) {
                GraalCInterface.logAndSetError("Null pointer received for update query", null);
                return -1;
            }
            Connection conn = connections.get(connectionHandle);
            if (conn == null) {
                GraalCInterface.logAndSetError("Invalid connection handle: " + connectionHandle, null);
                return -1;
            }
            String query = CTypeConversion.toJavaString((CCharPointer)queryPointer);
            Statement stmt = conn.createStatement();
            try {
                n = stmt.executeUpdate(query);
                if (stmt == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (stmt != null) {
                        try {
                            stmt.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    GraalCInterface.logAndSetError("Failed to execute update", e);
                    return -1;
                }
            }
            stmt.close();
        }
        return n;
    }

    @CEntryPoint(name="h2gis_close_query")
    public static void h2gisCloseQuery(IsolateThread thread, long queryHandle) {
        ResultSet rs = results.remove(queryHandle);
        Statement stmt = statements.remove(queryHandle);
        try {
            if (rs != null) {
                rs.close();
            }
        }
        catch (Exception e) {
            GraalCInterface.logAndSetError("Failed to close ResultSet", e);
        }
        try {
            if (stmt != null) {
                stmt.close();
            }
        }
        catch (Exception e) {
            GraalCInterface.logAndSetError("Failed to close Statement", e);
        }
    }

    @CEntryPoint(name="h2gis_close_connection")
    public static void h2gisCloseConnection(IsolateThread thread, long connectionHandle) {
        Connection conn = connections.remove(connectionHandle);
        try {
            if (conn != null) {
                conn.close();
            } else {
                GraalCInterface.logAndSetError("Invalid or already closed connection handle: " + connectionHandle, null);
            }
        }
        catch (Exception e) {
            GraalCInterface.logAndSetError("Failed to close connection", e);
        }
    }

    @CEntryPoint(name="h2gis_delete_database_and_close")
    public static void h2gisDeleteClose(IsolateThread thread, long connectionHandle) {
        Connection conn = connections.get(connectionHandle);
        try {
            if (conn == null) {
                GraalCInterface.logAndSetError("Invalid or already closed connection handle: " + connectionHandle, null);
                return;
            }
            try (Statement stmt = conn.createStatement();){
                ResultSet rs = stmt.executeQuery("SELECT DATABASE()");
                if (!rs.next()) {
                    throw new SQLException("Could not find database name");
                }
                String dbName = rs.getString(1);
                stmt.executeUpdate("DROP DATABASE `" + dbName + "`");
                rs.close();
            }
            conn.close();
            connections.remove(connectionHandle);
        }
        catch (Exception e) {
            GraalCInterface.logAndSetError("Failed to delete and close database", e);
        }
    }

    @CEntryPoint(name="h2gis_fetch_all")
    public static WordBase h2gisFetchAll(IsolateThread thread, long queryHandle, WordBase bufferSize) {
        try {
            ResultSet rs = results.get(queryHandle);
            if (rs == null) {
                if (bufferSize.rawValue() != 0L) {
                    unsafe.putLong(bufferSize.rawValue(), 0L);
                }
                return WordFactory.zero();
            }
            ResultSetWrapper resultSetWrapper = ResultSetWrapper.from(rs);
            byte[] arr = resultSetWrapper.serialize();
            if (bufferSize.rawValue() != 0L) {
                unsafe.putLong(bufferSize.rawValue(), arr.length);
            }
            long addr = unsafe.allocateMemory(arr.length);
            for (int i = 0; i < arr.length; ++i) {
                unsafe.putByte(addr + (long)i, arr[i]);
            }
            rs.close();
            return WordFactory.pointer((long)addr);
        }
        catch (Exception e) {
            System.err.println("Error in h2gis_fetch_all: " + e.getMessage());
            if (bufferSize.rawValue() != 0L) {
                unsafe.putLong(bufferSize.rawValue(), 0L);
            }
            return WordFactory.zero();
        }
    }

    @CEntryPoint(name="h2gis_fetch_one")
    public static WordBase h2gisFetchOne(IsolateThread thread, long queryHandle, WordBase bufferSize) {
        try {
            ResultSet rs = results.get(queryHandle);
            if (rs == null) {
                if (bufferSize.rawValue() != 0L) {
                    unsafe.putLong(bufferSize.rawValue(), 0L);
                }
                return WordFactory.zero();
            }
            ResultSetWrapper resultSetWrapper = ResultSetWrapper.fromOne(rs);
            byte[] arr = resultSetWrapper.serialize();
            if (bufferSize.rawValue() != 0L) {
                unsafe.putLong(bufferSize.rawValue(), arr.length);
            }
            long addr = unsafe.allocateMemory(arr.length);
            for (int i = 0; i < arr.length; ++i) {
                unsafe.putByte(addr + (long)i, arr[i]);
            }
            return WordFactory.pointer((long)addr);
        }
        catch (Exception e) {
            System.err.println("Error in h2gis_fetch_all: " + e.getMessage());
            if (bufferSize.rawValue() != 0L) {
                unsafe.putLong(bufferSize.rawValue(), 0L);
            }
            return WordFactory.zero();
        }
    }

    @CEntryPoint(name="h2gis_get_column_types")
    public static WordBase h2gisGetColumnTypes(IsolateThread thread, long queryHandle, WordBase colCountOut) {
        try {
            ResultSet rs = results.get(queryHandle);
            if (rs == null) {
                if (colCountOut.rawValue() != 0L) {
                    unsafe.putLong(colCountOut.rawValue(), 0L);
                }
                return WordFactory.zero();
            }
            ResultSetMetaData meta = rs.getMetaData();
            int colCount = meta.getColumnCount();
            if (colCount < 1) {
                if (colCountOut.rawValue() != 0L) {
                    unsafe.putLong(colCountOut.rawValue(), 0L);
                }
                return WordFactory.zero();
            }
            ByteBuffer buffer = ByteBuffer.allocate(colCount * 4).order(ByteOrder.LITTLE_ENDIAN);
            for (int i = 1; i <= colCount; ++i) {
                int typeCode;
                String sqlTypeName = meta.getColumnTypeName(i).toLowerCase();
                int jdbcType = meta.getColumnType(i);
                switch (jdbcType) {
                    case -6: 
                    case 4: 
                    case 5: {
                        typeCode = 1;
                        break;
                    }
                    case -5: {
                        typeCode = 2;
                        break;
                    }
                    case 6: 
                    case 7: {
                        typeCode = 3;
                        break;
                    }
                    case 2: 
                    case 3: 
                    case 8: {
                        typeCode = 4;
                        break;
                    }
                    case -7: 
                    case 16: {
                        typeCode = 5;
                        break;
                    }
                    case -1: 
                    case 1: 
                    case 12: {
                        typeCode = 6;
                        break;
                    }
                    case 91: 
                    case 92: 
                    case 93: {
                        typeCode = 7;
                        break;
                    }
                    default: {
                        typeCode = sqlTypeName.startsWith("geometry") ? 8 : 99;
                    }
                }
                buffer.putInt(typeCode);
            }
            byte[] arr = buffer.array();
            if (colCountOut.rawValue() != 0L) {
                unsafe.putLong(colCountOut.rawValue(), colCount);
            }
            long addr = unsafe.allocateMemory(arr.length);
            for (int i = 0; i < arr.length; ++i) {
                unsafe.putByte(addr + (long)i, arr[i]);
            }
            return WordFactory.pointer((long)addr);
        }
        catch (Exception e) {
            System.err.println("Error in h2gis_get_column_types: " + e.getMessage());
            return WordFactory.zero();
        }
    }

    @CEntryPoint(name="h2gis_free_result_set")
    public static long freeResultResultSet(IsolateThread thread, long queryHandle) {
        try {
            results.get(queryHandle).close();
            return 0L;
        }
        catch (SQLException e) {
            System.err.println(e.toString());
            return 1L;
        }
    }

    @CEntryPoint(name="h2gis_free_result_buffer")
    public static void freeResultBuffer(IsolateThread thread, WordBase ptr) {
        if (ptr.rawValue() != 0L) {
            unsafe.freeMemory(ptr.rawValue());
        }
    }

    private static void logAndSetError(String message, Exception e) {
        if (e != null) {
            LOGGER.log(Level.SEVERE, message, e);
            lastError.set(message + ": " + e.getMessage());
        } else {
            LOGGER.severe(message);
            lastError.set(message);
        }
    }

    private static Unsafe getUnsafe() {
        try {
            Field f = Unsafe.class.getDeclaredField("theUnsafe");
            f.setAccessible(true);
            return (Unsafe)f.get(null);
        }
        catch (Exception e) {
            throw new RuntimeException("Unable to get Unsafe instance", e);
        }
    }

    static {
        try {
            DriverManager.registerDriver((java.sql.Driver)new Driver());
        }
        catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to register H2 Driver", e);
        }
    }
}

