/*
 * Decompiled with CFR 0.152.
 */
package org.h2gis.functions.io.geojson;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import org.h2.util.geometry.JTSUtils;
import org.h2gis.api.EmptyProgressVisitor;
import org.h2gis.api.ProgressVisitor;
import org.h2gis.functions.io.geojson.GeoJsonField;
import org.h2gis.utilities.JDBCUtilities;
import org.h2gis.utilities.TableLocation;
import org.h2gis.utilities.dbtypes.DBTypes;
import org.h2gis.utilities.dbtypes.DBUtils;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.CoordinateXYZM;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.PrecisionModel;
import org.locationtech.jts.geom.impl.CoordinateArraySequence;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GeoJsonReaderDriver {
    private final File fileName;
    private final Connection connection;
    private static GeometryFactory GF;
    private final String encoding;
    private final boolean deleteTable;
    private PreparedStatement preparedStatement = null;
    private JsonFactory jsFactory;
    private int nbFeature = 1;
    private int featureCounter = 1;
    private ProgressVisitor progress = new EmptyProgressVisitor();
    private static final int AVERAGE_NODE_SIZE = 500;
    boolean hasGeometryField = false;
    private static final Logger log;
    private int parsedSRID = 0;
    private DBTypes dbType = DBTypes.H2GIS;
    private String tableLocation;
    private LinkedHashMap<String, Integer> cachedColumnNames;
    private LinkedHashMap<String, Integer> cachedColumnIndex;
    private static final int BATCH_MAX_SIZE = 100;
    private Set finalGeometryTypes;
    private JsonEncoding jsonEncoding;
    private int coordinateDimension = 2;

    public GeoJsonReaderDriver(Connection connection, File fileName, String encoding, boolean deleteTable) {
        this.connection = connection;
        this.fileName = fileName;
        this.encoding = encoding;
        this.deleteTable = deleteTable;
    }

    public String read(ProgressVisitor progress, String tableReference) throws SQLException, IOException {
        String fileNameLower = this.fileName.getName().toLowerCase();
        if (this.fileName != null && (fileNameLower.endsWith(".geojson") || fileNameLower.endsWith(".json"))) {
            if (!this.fileName.exists()) {
                throw new SQLException("The file " + this.tableLocation + " doesn't exist ");
            }
            this.dbType = DBUtils.getDBType((Connection)this.connection);
            this.tableLocation = TableLocation.parse((String)tableReference, (DBTypes)this.dbType).toString();
            if (this.deleteTable) {
                Statement stmt = this.connection.createStatement();
                stmt.execute("DROP TABLE IF EXISTS " + this.tableLocation);
                stmt.close();
            }
            if (this.fileName.length() > 0L) {
                this.parseGeoJson(progress);
                return this.tableLocation;
            }
            JDBCUtilities.createEmptyTable((Connection)this.connection, (String)this.tableLocation);
            return this.tableLocation;
        }
        if (this.fileName != null && this.fileName.getName().toLowerCase().endsWith(".gz")) {
            if (!this.fileName.exists()) {
                throw new SQLException("The file " + this.tableLocation + " doesn't exist ");
            }
            this.dbType = DBUtils.getDBType((Connection)this.connection);
            this.tableLocation = TableLocation.parse((String)tableReference, (DBTypes)this.dbType).toString();
            if (this.deleteTable) {
                Statement stmt = this.connection.createStatement();
                stmt.execute("DROP TABLE IF EXISTS " + this.tableLocation);
                stmt.close();
            }
            if (this.fileName.length() > 0L) {
                this.progress = progress.subProcess(100);
                this.init();
                FileInputStream fis = new FileInputStream(this.fileName);
                if (this.parseMetadata(new GZIPInputStream(fis))) {
                    this.connection.setAutoCommit(false);
                    GF = new GeometryFactory(new PrecisionModel(), this.parsedSRID);
                    fis = new FileInputStream(this.fileName);
                    this.parseData(new GZIPInputStream(fis));
                    this.connection.setAutoCommit(true);
                    return this.tableLocation;
                }
                throw new SQLException("Cannot create the table " + this.tableLocation + " to import the GeoJSON data");
            }
            JDBCUtilities.createEmptyTable((Connection)this.connection, (String)this.tableLocation);
            return this.tableLocation;
        }
        throw new SQLException("The geojson read driver supports only geojson or gz extensions");
    }

    private void parseGeoJson(ProgressVisitor progress) throws SQLException, IOException {
        this.progress = progress.subProcess(100);
        this.init();
        if (!this.parseMetadata(new FileInputStream(this.fileName))) {
            throw new SQLException("Cannot create the table " + this.tableLocation + " to import the GeoJSON data");
        }
        this.connection.setAutoCommit(false);
        GF = new GeometryFactory(new PrecisionModel(), this.parsedSRID);
        this.parseData(new FileInputStream(this.fileName));
        this.connection.setAutoCommit(true);
    }

    private boolean parseMetadata(InputStream is) throws SQLException, IOException {
        block43: {
            try {
                this.cachedColumnNames = new LinkedHashMap();
                this.finalGeometryTypes = new HashSet();
                try (JsonParser jp = this.jsFactory.createParser((Reader)new InputStreamReader(is, this.jsonEncoding.getJavaName()));){
                    jp.nextToken();
                    jp.nextToken();
                    String dataType = jp.getText();
                    if (dataType.equalsIgnoreCase(GeoJsonField.TYPE)) {
                        jp.nextToken();
                        dataType = jp.getText();
                        if (dataType.equalsIgnoreCase(GeoJsonField.FEATURECOLLECTION)) {
                            jp.nextToken();
                            this.parseFeaturesMetadata(jp);
                            break block43;
                        }
                        if (dataType.equalsIgnoreCase(GeoJsonField.FEATURE)) {
                            this.parseFeatureMetadata(jp);
                            break block43;
                        }
                        if (dataType.equalsIgnoreCase(GeoJsonField.POINT)) {
                            this.parseGeometryMetadata(jp, jp.getText());
                            this.hasGeometryField = true;
                            break block43;
                        }
                        if (dataType.equalsIgnoreCase(GeoJsonField.LINESTRING)) {
                            this.parseGeometryMetadata(jp, jp.getText());
                            this.hasGeometryField = true;
                            break block43;
                        }
                        if (dataType.equalsIgnoreCase(GeoJsonField.POLYGON)) {
                            this.parseGeometryMetadata(jp, jp.getText());
                            this.hasGeometryField = true;
                            break block43;
                        }
                        if (dataType.equalsIgnoreCase(GeoJsonField.MULTIPOINT)) {
                            this.parseGeometryMetadata(jp, jp.getText());
                            this.hasGeometryField = true;
                            break block43;
                        }
                        if (dataType.equalsIgnoreCase(GeoJsonField.MULTILINESTRING)) {
                            this.parseGeometryMetadata(jp, jp.getText());
                            this.hasGeometryField = true;
                            break block43;
                        }
                        if (dataType.equalsIgnoreCase(GeoJsonField.MULTIPOLYGON)) {
                            this.parseGeometryMetadata(jp, jp.getText());
                            this.hasGeometryField = true;
                            break block43;
                        }
                        if (dataType.equalsIgnoreCase(GeoJsonField.GEOMETRYCOLLECTION)) {
                            this.parseGeometryMetadata(jp, jp.getText());
                            this.hasGeometryField = true;
                            break block43;
                        }
                        throw new SQLException("Malformed GeoJSON file. Found '" + dataType + "'");
                    }
                    if (dataType.equalsIgnoreCase(GeoJsonField.FEATURES)) {
                        this.parseFeaturesMetadata(jp);
                        break block43;
                    }
                    throw new SQLException("Malformed GeoJSON file. Found '" + dataType + "'");
                }
            }
            catch (FileNotFoundException ex) {
                throw new SQLException(ex);
            }
            finally {
                try {
                    if (is != null) {
                        is.close();
                    }
                }
                catch (IOException ex) {
                    throw new IOException(ex);
                }
            }
        }
        if (this.hasGeometryField) {
            StringBuilder createTable = new StringBuilder();
            createTable.append("CREATE TABLE ");
            createTable.append(this.tableLocation);
            createTable.append(" (");
            Object finalGeometryType = GeoJsonField.GEOMETRY;
            if (this.finalGeometryTypes.size() == 1) {
                finalGeometryType = (String)this.finalGeometryTypes.iterator().next();
                switch (this.coordinateDimension) {
                    case 3: {
                        finalGeometryType = (String)finalGeometryType + "Z";
                        break;
                    }
                    case 4: {
                        finalGeometryType = (String)finalGeometryType + "ZM";
                    }
                }
                createTable.append("THE_GEOM GEOMETRY(").append((String)finalGeometryType).append(",").append(this.parsedSRID).append(")");
            } else {
                createTable.append("THE_GEOM GEOMETRY(GEOMETRY,").append(this.parsedSRID).append(")");
            }
            this.cachedColumnIndex = new LinkedHashMap();
            StringBuilder insertTable = new StringBuilder("INSERT INTO ");
            insertTable.append(this.tableLocation).append(" VALUES(?");
            int i = 1;
            for (Map.Entry<String, Integer> columns : this.cachedColumnNames.entrySet()) {
                String columnName = columns.getKey();
                Integer columnType = columns.getValue();
                this.cachedColumnIndex.put(columnName, i++);
                createTable.append(",").append(columns.getKey()).append(" ").append(GeoJsonReaderDriver.getSQLTypeName(columnType));
                if (columnType == 2003) {
                    if (this.dbType == DBTypes.H2 || this.dbType == DBTypes.H2GIS) {
                        insertTable.append(",").append(" ? FORMAT json");
                        continue;
                    }
                    insertTable.append(",").append("cast(? as json)");
                    continue;
                }
                insertTable.append(",").append("?");
            }
            createTable.append(")");
            insertTable.append(")");
            try (Statement stmt = this.connection.createStatement();){
                stmt.execute(createTable.toString());
            }
            this.preparedStatement = this.connection.prepareStatement(insertTable.toString());
            return true;
        }
        throw new SQLException("The geojson file  does not contain any geometry.");
    }

    private void parseFeaturesMetadata(JsonParser jp) throws IOException, SQLException {
        while (!jp.getText().equalsIgnoreCase(GeoJsonField.FEATURES) && !jp.getText().equalsIgnoreCase(GeoJsonField.CRS)) {
            jp.nextToken();
            if (jp.getCurrentToken().equals((Object)JsonToken.START_ARRAY) || jp.getCurrentToken().equals((Object)JsonToken.START_OBJECT)) {
                jp.skipChildren();
            }
            jp.nextToken();
        }
        if (jp.getText().equalsIgnoreCase(GeoJsonField.CRS)) {
            this.parsedSRID = this.readCRS(jp);
        }
        if (jp.getText().equalsIgnoreCase(GeoJsonField.FEATURES)) {
            jp.nextToken();
            JsonToken token = jp.nextToken();
            while (token != JsonToken.END_ARRAY) {
                jp.nextToken();
                jp.nextToken();
                String geomType = jp.getText();
                if (geomType.equalsIgnoreCase(GeoJsonField.FEATURE)) {
                    if (this.progress.isCanceled()) {
                        throw new SQLException("Canceled by user");
                    }
                    this.parseFeatureMetadata(jp);
                    token = jp.nextToken();
                    if (token != JsonToken.START_OBJECT && token != JsonToken.END_ARRAY) {
                        throw new SQLException("Malformed GeoJSON file. Expected 'Start Object or End array', found '" + token + "'");
                    }
                    ++this.nbFeature;
                    continue;
                }
                throw new SQLException("Malformed GeoJSON file. Expected 'Feature', found '" + geomType + "'");
            }
        } else {
            throw new SQLException("Malformed GeoJSON file. Expected 'features', found '" + jp.getText() + "'");
        }
    }

    private void parseFeatureMetadata(JsonParser jp) throws IOException, SQLException {
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            String field = jp.getText();
            if (!(field.equalsIgnoreCase(GeoJsonField.FEATURE_ID) || field.equalsIgnoreCase(GeoJsonField.GEOMETRY) || field.equalsIgnoreCase(GeoJsonField.PROPERTIES))) {
                JsonToken currentToken = jp.nextToken();
                if (!currentToken.equals((Object)JsonToken.START_ARRAY) && !currentToken.equals((Object)JsonToken.START_OBJECT)) continue;
                jp.skipChildren();
                continue;
            }
            if (field.equalsIgnoreCase(GeoJsonField.GEOMETRY)) {
                this.parseParentGeometryMetadata(jp);
                this.hasGeometryField = true;
                continue;
            }
            if (field.equalsIgnoreCase(GeoJsonField.PROPERTIES)) {
                this.parsePropertiesMetadata(jp);
                continue;
            }
            if (!field.equalsIgnoreCase(GeoJsonField.FEATURE_ID)) continue;
            this.processPropertyMetadata(jp);
        }
    }

    private void parseParentGeometryMetadata(JsonParser jp) throws IOException, SQLException {
        if (jp.nextToken() != JsonToken.VALUE_NULL) {
            jp.nextToken();
            jp.nextToken();
            String geometryType = jp.getText();
            this.parseGeometryMetadata(jp, geometryType);
        }
    }

    private void parseGeometryMetadata(JsonParser jp, String geometryType) throws IOException, SQLException {
        if (geometryType.equalsIgnoreCase(GeoJsonField.POINT)) {
            this.parsePointMetadata(jp);
            this.finalGeometryTypes.add(GeoJsonField.POINT);
        } else if (geometryType.equalsIgnoreCase(GeoJsonField.MULTIPOINT)) {
            this.parseMultiPointMetadata(jp);
            this.finalGeometryTypes.add(GeoJsonField.MULTIPOINT);
        } else if (geometryType.equalsIgnoreCase(GeoJsonField.LINESTRING)) {
            this.parseLinestringMetadata(jp);
            this.finalGeometryTypes.add(GeoJsonField.LINESTRING);
        } else if (geometryType.equalsIgnoreCase(GeoJsonField.MULTILINESTRING)) {
            this.parseMultiLinestringMetadata(jp);
            this.finalGeometryTypes.add(GeoJsonField.MULTILINESTRING);
        } else if (geometryType.equalsIgnoreCase(GeoJsonField.POLYGON)) {
            this.parsePolygonMetadata(jp);
            this.finalGeometryTypes.add(GeoJsonField.POLYGON);
        } else if (geometryType.equalsIgnoreCase(GeoJsonField.MULTIPOLYGON)) {
            this.parseMultiPolygonMetadata(jp);
            this.finalGeometryTypes.add(GeoJsonField.MULTIPOLYGON);
        } else if (geometryType.equalsIgnoreCase(GeoJsonField.GEOMETRYCOLLECTION)) {
            this.parseGeometryCollectionMetadata(jp);
            this.finalGeometryTypes.add(GeoJsonField.GEOMETRYCOLLECTION);
        } else {
            throw new SQLException("Unsupported geometry : " + geometryType);
        }
    }

    private void parsePointMetadata(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (!coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
        }
        jp.nextToken();
        this.parseCoordinateMetadata(jp);
    }

    private void parseMultiPointMetadata(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (!coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
        }
        jp.nextToken();
        this.parseCoordinatesMetadata(jp);
        jp.nextToken();
    }

    private void parseLinestringMetadata(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (!coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
        }
        jp.nextToken();
        this.parseCoordinatesMetadata(jp);
        jp.nextToken();
    }

    private void parseMultiLinestringMetadata(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            jp.nextToken();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                this.parseCoordinatesMetadata(jp);
                jp.nextToken();
            }
        } else {
            throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
        }
        jp.nextToken();
    }

    private void parsePolygonMetadata(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            jp.nextToken();
            int linesIndex = 0;
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                if (linesIndex == 0) {
                    this.parseCoordinatesMetadata(jp);
                } else {
                    this.parseCoordinatesMetadata(jp);
                }
                jp.nextToken();
                ++linesIndex;
            }
            if (linesIndex > 1) {
                jp.nextToken();
            } else {
                jp.nextToken();
            }
        } else {
            throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
        }
    }

    private void parseMultiPolygonMetadata(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            jp.nextToken();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                jp.nextToken();
                while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                    this.parseCoordinatesMetadata(jp);
                    jp.nextToken();
                }
                jp.nextToken();
            }
        } else {
            throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
        }
        jp.nextToken();
    }

    private void parseGeometryCollectionMetadata(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.GEOMETRIES)) {
            jp.nextToken();
            jp.nextToken();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                jp.nextToken();
                jp.nextToken();
                String geometryType = jp.getText();
                this.parseGeometryMetadata(jp, geometryType);
                jp.nextToken();
            }
        } else {
            throw new SQLException("Malformed GeoJSON file. Expected 'geometries', found '" + coordinatesField + "'");
        }
        jp.nextToken();
    }

    private void parseCoordinateMetadata(JsonParser jp) throws IOException {
        jp.nextToken();
        jp.nextToken();
        jp.nextToken();
        if (jp.getCurrentToken() != JsonToken.END_ARRAY) {
            jp.nextToken();
            this.coordinateDimension = 3;
        }
        if (jp.getCurrentToken() != JsonToken.END_ARRAY) {
            jp.nextToken();
            this.coordinateDimension = 4;
        }
        jp.nextToken();
    }

    private void parseCoordinatesMetadata(JsonParser jp) throws IOException {
        jp.nextToken();
        while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
            this.parseCoordinateMetadata(jp);
        }
    }

    private void processPropertyMetadata(JsonParser jp) throws IOException {
        String fieldName = TableLocation.capsIdentifier((String)jp.getText(), (DBTypes)this.dbType);
        fieldName = TableLocation.quoteIdentifier((String)fieldName, (DBTypes)this.dbType);
        JsonToken value = jp.nextToken();
        if (null != value) {
            Integer dataType = this.cachedColumnNames.get(fieldName);
            boolean hasField = this.cachedColumnNames.containsKey(fieldName);
            switch (value) {
                case VALUE_STRING: {
                    this.cachedColumnNames.put(fieldName, 12);
                    break;
                }
                case VALUE_TRUE: 
                case VALUE_FALSE: {
                    if (!hasField || dataType == 0) {
                        this.cachedColumnNames.put(fieldName, 16);
                        break;
                    }
                    if (!hasField || dataType == 16) break;
                    this.cachedColumnNames.put(fieldName, 12);
                    break;
                }
                case VALUE_NUMBER_FLOAT: {
                    if (!hasField || dataType == 0) {
                        this.cachedColumnNames.put(fieldName, 8);
                        break;
                    }
                    if (!hasField) break;
                    if (dataType == -5) {
                        this.cachedColumnNames.put(fieldName, 8);
                        break;
                    }
                    if (dataType == 8) break;
                    this.cachedColumnNames.put(fieldName, 12);
                    break;
                }
                case VALUE_NUMBER_INT: {
                    if (!hasField || dataType == 0) {
                        this.cachedColumnNames.put(fieldName, -5);
                        break;
                    }
                    if (!hasField || dataType == -5 || dataType == 8) break;
                    this.cachedColumnNames.put(fieldName, 12);
                    break;
                }
                case START_ARRAY: {
                    if (!hasField || dataType == 0) {
                        this.cachedColumnNames.put(fieldName, 2003);
                    } else if (hasField && dataType != 2003) {
                        this.cachedColumnNames.put(fieldName, 12);
                    }
                    this.parseArrayMetadata(jp);
                    break;
                }
                case START_OBJECT: {
                    if (!hasField || dataType == 0) {
                        this.cachedColumnNames.put(fieldName, 2003);
                    } else if (hasField && dataType != 2003) {
                        this.cachedColumnNames.put(fieldName, 12);
                    }
                    this.parseObjectMetadata(jp);
                    break;
                }
                case VALUE_NULL: {
                    if (hasField) break;
                    this.cachedColumnNames.put(fieldName, 0);
                }
            }
        }
    }

    private void parsePropertiesMetadata(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            this.processPropertyMetadata(jp);
        }
    }

    private void init() throws SQLException {
        this.jsonEncoding = JsonEncoding.UTF8;
        if (this.encoding != null && !this.encoding.isEmpty()) {
            try {
                this.jsonEncoding = JsonEncoding.valueOf((String)this.encoding);
            }
            catch (IllegalArgumentException ex) {
                throw new SQLException("Only UTF-8, UTF-16BE, UTF-16LE, UTF-32BE, UTF-32LE encoding is supported");
            }
        }
        this.jsFactory = new JsonFactory();
        this.jsFactory.configure(JsonParser.Feature.ALLOW_COMMENTS, true);
        this.jsFactory.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
        this.jsFactory.configure(JsonParser.Feature.ALLOW_NON_NUMERIC_NUMBERS, true);
    }

    private Object[] parseFeature(JsonParser jp) throws IOException, SQLException {
        Object[] values = new Object[this.cachedColumnIndex.size() + 1];
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            String field = jp.getText();
            if (!(field.equalsIgnoreCase(GeoJsonField.FEATURE_ID) || field.equalsIgnoreCase(GeoJsonField.GEOMETRY) || field.equalsIgnoreCase(GeoJsonField.PROPERTIES))) {
                JsonToken currentToken = jp.nextToken();
                if (!currentToken.equals((Object)JsonToken.START_ARRAY) && !currentToken.equals((Object)JsonToken.START_OBJECT)) continue;
                jp.skipChildren();
                continue;
            }
            if (field.equalsIgnoreCase(GeoJsonField.GEOMETRY)) {
                this.setGeometry(jp, values);
                continue;
            }
            if (field.equalsIgnoreCase(GeoJsonField.PROPERTIES)) {
                this.parseProperties(jp, values);
                continue;
            }
            if (!field.equalsIgnoreCase(GeoJsonField.FEATURE_ID)) continue;
            this.parseProperty(jp, values);
        }
        return values;
    }

    private void setGeometry(JsonParser jp, Object[] values) throws IOException, SQLException {
        if (jp.nextToken() != JsonToken.VALUE_NULL) {
            jp.nextToken();
            jp.nextToken();
            String geometryType = jp.getText();
            values[0] = JTSUtils.geometry2ewkb((Geometry)this.parseGeometry(jp, geometryType));
        }
    }

    private Geometry parseGeometry(JsonParser jp, String geometryType) throws IOException, SQLException {
        if (geometryType.equalsIgnoreCase(GeoJsonField.POINT)) {
            return this.parsePoint(jp);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.MULTIPOINT)) {
            return this.parseMultiPoint(jp);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.LINESTRING)) {
            return this.parseLinestring(jp);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.MULTILINESTRING)) {
            return this.parseMultiLinestring(jp);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.POLYGON)) {
            return this.parsePolygon(jp);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.MULTIPOLYGON)) {
            return this.parseMultiPolygon(jp);
        }
        if (geometryType.equalsIgnoreCase(GeoJsonField.GEOMETRYCOLLECTION)) {
            return this.parseGeometryCollection(jp);
        }
        throw new SQLException("Unsupported geometry : " + geometryType);
    }

    private void parseProperty(JsonParser jp, Object[] values) throws IOException {
        String fieldName = TableLocation.capsIdentifier((String)jp.getText(), (DBTypes)this.dbType);
        fieldName = TableLocation.quoteIdentifier((String)fieldName, (DBTypes)this.dbType);
        JsonToken value = jp.nextToken();
        if (null != value) {
            switch (value) {
                case VALUE_STRING: {
                    values[this.cachedColumnIndex.get((Object)fieldName).intValue()] = jp.getText();
                    break;
                }
                case VALUE_TRUE: {
                    values[this.cachedColumnIndex.get((Object)fieldName).intValue()] = jp.getValueAsBoolean();
                    break;
                }
                case VALUE_FALSE: {
                    values[this.cachedColumnIndex.get((Object)fieldName).intValue()] = jp.getValueAsBoolean();
                    break;
                }
                case VALUE_NUMBER_FLOAT: {
                    values[this.cachedColumnIndex.get((Object)fieldName).intValue()] = jp.getValueAsDouble();
                    break;
                }
                case VALUE_NUMBER_INT: {
                    if (jp.getNumberType() == JsonParser.NumberType.INT) {
                        values[this.cachedColumnIndex.get((Object)fieldName).intValue()] = jp.getIntValue();
                        break;
                    }
                    values[this.cachedColumnIndex.get((Object)fieldName).intValue()] = jp.getLongValue();
                    break;
                }
                case START_ARRAY: {
                    StringBuilder sb = new StringBuilder();
                    this.parseArray(jp, sb);
                    values[this.cachedColumnIndex.get((Object)fieldName).intValue()] = sb.toString();
                    break;
                }
                case START_OBJECT: {
                    StringBuilder sb = new StringBuilder();
                    this.parseObject(jp, sb);
                    values[this.cachedColumnIndex.get((Object)fieldName).intValue()] = sb.toString();
                    break;
                }
                case VALUE_NULL: {
                    values[this.cachedColumnIndex.get((Object)fieldName).intValue()] = null;
                    break;
                }
            }
        }
    }

    private void parseProperties(JsonParser jp, Object[] values) throws IOException, SQLException {
        jp.nextToken();
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            this.parseProperty(jp, values);
        }
    }

    private void parseFeatures(JsonParser jp) throws IOException, SQLException {
        while (!jp.getText().equalsIgnoreCase(GeoJsonField.FEATURES) && !jp.getText().equalsIgnoreCase(GeoJsonField.CRS)) {
            jp.nextToken();
            if (jp.getCurrentToken().equals((Object)JsonToken.START_ARRAY) || jp.getCurrentToken().equals((Object)JsonToken.START_OBJECT)) {
                jp.skipChildren();
            }
            jp.nextToken();
        }
        String firstParam = jp.getText();
        if (firstParam.equalsIgnoreCase(GeoJsonField.CRS)) {
            firstParam = this.skipCRS(jp);
        }
        if (firstParam.equalsIgnoreCase(GeoJsonField.FEATURES)) {
            this.connection.setAutoCommit(false);
            jp.nextToken();
            JsonToken token = jp.nextToken();
            long batchSize = 0L;
            while (token != JsonToken.END_ARRAY) {
                jp.nextToken();
                jp.nextToken();
                String geomType = jp.getText();
                if (geomType.equalsIgnoreCase(GeoJsonField.FEATURE)) {
                    if (this.progress.isCanceled()) {
                        throw new SQLException("Canceled by user");
                    }
                    Object[] values = this.parseFeature(jp);
                    for (int i = 0; i < values.length; ++i) {
                        this.preparedStatement.setObject(i + 1, values[i]);
                    }
                    this.preparedStatement.addBatch();
                    if (++batchSize >= 100L) {
                        this.preparedStatement.executeBatch();
                        this.connection.commit();
                        this.preparedStatement.clearBatch();
                        batchSize = 0L;
                    }
                    token = jp.nextToken();
                    ++this.featureCounter;
                    this.progress.setStep(this.featureCounter / this.nbFeature * 100);
                    if (batchSize <= 0L) continue;
                    this.preparedStatement.executeBatch();
                    this.connection.commit();
                    this.preparedStatement.clearBatch();
                    continue;
                }
                this.connection.setAutoCommit(true);
                throw new SQLException("Malformed GeoJSON file. Expected 'Feature', found '" + geomType + "'");
            }
        } else {
            throw new SQLException("Malformed GeoJSON file. Expected 'features', found '" + firstParam + "'");
        }
        this.connection.setAutoCommit(true);
        log.debug(this.featureCounter - 1 + " geojson features have been imported.");
    }

    private Point parsePoint(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            return GF.createPoint(this.parseCoordinate(jp));
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private MultiPoint parseMultiPoint(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            MultiPoint mPoint = GF.createMultiPoint(this.parseCoordinates(jp));
            jp.nextToken();
            return mPoint;
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private LineString parseLinestring(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            LineString line = GF.createLineString(this.parseCoordinates(jp));
            jp.nextToken();
            return line;
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private MultiLineString parseMultiLinestring(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            ArrayList<LineString> lineStrings = new ArrayList<LineString>();
            jp.nextToken();
            jp.nextToken();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                lineStrings.add(GF.createLineString(this.parseCoordinates(jp)));
                jp.nextToken();
            }
            MultiLineString line = GF.createMultiLineString(lineStrings.toArray(new LineString[0]));
            jp.nextToken();
            return line;
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private Polygon parsePolygon(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            jp.nextToken();
            jp.nextToken();
            int linesIndex = 0;
            LinearRing linearRing = null;
            ArrayList<LinearRing> holes = new ArrayList<LinearRing>();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                if (linesIndex == 0) {
                    linearRing = GF.createLinearRing(this.parseCoordinates(jp));
                } else {
                    holes.add(GF.createLinearRing(this.parseCoordinates(jp)));
                }
                jp.nextToken();
                ++linesIndex;
            }
            if (linesIndex > 1) {
                jp.nextToken();
                return GF.createPolygon(linearRing, holes.toArray(new LinearRing[0]));
            }
            jp.nextToken();
            return GF.createPolygon(linearRing, null);
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private MultiPolygon parseMultiPolygon(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.COORDINATES)) {
            ArrayList<Polygon> polygons = new ArrayList<Polygon>();
            jp.nextToken();
            jp.nextToken();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                jp.nextToken();
                int linesIndex = 0;
                LinearRing linearRing = null;
                ArrayList<LinearRing> holes = new ArrayList<LinearRing>();
                while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                    if (linesIndex == 0) {
                        linearRing = GF.createLinearRing(this.parseCoordinates(jp));
                    } else {
                        holes.add(GF.createLinearRing(this.parseCoordinates(jp)));
                    }
                    jp.nextToken();
                    ++linesIndex;
                }
                if (linesIndex > 1) {
                    jp.nextToken();
                    polygons.add(GF.createPolygon(linearRing, holes.toArray(new LinearRing[0])));
                    continue;
                }
                jp.nextToken();
                polygons.add(GF.createPolygon(linearRing, null));
            }
            jp.nextToken();
            return GF.createMultiPolygon(polygons.toArray(new Polygon[0]));
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'coordinates', found '" + coordinatesField + "'");
    }

    private GeometryCollection parseGeometryCollection(JsonParser jp) throws IOException, SQLException {
        jp.nextToken();
        String coordinatesField = jp.getText();
        if (coordinatesField.equalsIgnoreCase(GeoJsonField.GEOMETRIES)) {
            jp.nextToken();
            jp.nextToken();
            ArrayList<Geometry> geometries = new ArrayList<Geometry>();
            while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
                jp.nextToken();
                jp.nextToken();
                String geometryType = jp.getText();
                geometries.add(this.parseGeometry(jp, geometryType));
                jp.nextToken();
            }
            jp.nextToken();
            return GF.createGeometryCollection(geometries.toArray(new Geometry[0]));
        }
        throw new SQLException("Malformed GeoJSON file. Expected 'geometries', found '" + coordinatesField + "'");
    }

    private CoordinateSequence parseCoordinates(JsonParser jp) throws IOException {
        jp.nextToken();
        ArrayList<Coordinate> coords = new ArrayList<Coordinate>();
        while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
            coords.add(this.parseCoordinate(jp));
        }
        int hasM = this.coordinateDimension > 3 ? 1 : 0;
        CoordinateArraySequence seq = new CoordinateArraySequence(coords.toArray(new Coordinate[0]), this.coordinateDimension, hasM);
        return seq;
    }

    private static double getOrDefault(List<Double> coordinates, int index, double defaultValue) {
        return index < coordinates.size() ? coordinates.get(index) : defaultValue;
    }

    private Coordinate parseCoordinate(JsonParser jp) throws IOException {
        ArrayList<Double> coordinates = new ArrayList<Double>();
        jp.nextToken();
        while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
            coordinates.add(jp.getDoubleValue());
            jp.nextToken();
        }
        Object coord = this.coordinateDimension == 4 ? new CoordinateXYZM(GeoJsonReaderDriver.getOrDefault(coordinates, 0, 0.0), GeoJsonReaderDriver.getOrDefault(coordinates, 1, 0.0), GeoJsonReaderDriver.getOrDefault(coordinates, 2, 0.0), GeoJsonReaderDriver.getOrDefault(coordinates, 3, 0.0)) : (this.coordinateDimension == 3 ? new Coordinate(GeoJsonReaderDriver.getOrDefault(coordinates, 0, 0.0), GeoJsonReaderDriver.getOrDefault(coordinates, 1, 0.0), GeoJsonReaderDriver.getOrDefault(coordinates, 2, 0.0)) : new Coordinate(GeoJsonReaderDriver.getOrDefault(coordinates, 0, 0.0), GeoJsonReaderDriver.getOrDefault(coordinates, 1, 0.0)));
        jp.nextToken();
        return coord;
    }

    private void parseData(InputStream is) throws IOException, SQLException {
        block28: {
            try (JsonParser jp = this.jsFactory.createParser((Reader)new InputStreamReader(is, this.jsonEncoding.getJavaName()));){
                jp.nextToken();
                jp.nextToken();
                String dataType = jp.getText();
                if (dataType.equalsIgnoreCase(GeoJsonField.TYPE)) {
                    jp.nextToken();
                    dataType = jp.getText();
                    if (dataType.equalsIgnoreCase(GeoJsonField.FEATURECOLLECTION)) {
                        jp.nextToken();
                        this.parseFeatures(jp);
                        break block28;
                    }
                    if (dataType.equalsIgnoreCase(GeoJsonField.FEATURE)) {
                        Object[] values = this.parseFeature(jp);
                        for (int i = 0; i < values.length; ++i) {
                            this.preparedStatement.setObject(i + 1, values[i]);
                        }
                        this.preparedStatement.execute();
                        break block28;
                    }
                    if (dataType.equalsIgnoreCase(GeoJsonField.POINT)) {
                        this.preparedStatement.setObject(1, this.parsePoint(jp));
                        this.preparedStatement.execute();
                        break block28;
                    }
                    if (dataType.equalsIgnoreCase(GeoJsonField.LINESTRING)) {
                        this.preparedStatement.setObject(1, this.parseLinestring(jp));
                        this.preparedStatement.execute();
                        break block28;
                    }
                    if (dataType.equalsIgnoreCase(GeoJsonField.POLYGON)) {
                        this.preparedStatement.setObject(1, this.parsePolygon(jp));
                        this.preparedStatement.execute();
                        break block28;
                    }
                    if (dataType.equalsIgnoreCase(GeoJsonField.MULTIPOINT)) {
                        this.preparedStatement.setObject(1, this.parseMultiPoint(jp));
                        this.preparedStatement.execute();
                        break block28;
                    }
                    if (dataType.equalsIgnoreCase(GeoJsonField.MULTILINESTRING)) {
                        this.preparedStatement.setObject(1, this.parseMultiLinestring(jp));
                        this.preparedStatement.execute();
                        break block28;
                    }
                    if (dataType.equalsIgnoreCase(GeoJsonField.MULTIPOLYGON)) {
                        this.preparedStatement.setObject(1, this.parseMultiPolygon(jp));
                        this.preparedStatement.execute();
                        break block28;
                    }
                    if (dataType.equalsIgnoreCase(GeoJsonField.GEOMETRYCOLLECTION)) {
                        this.preparedStatement.setObject(1, this.parseGeometryCollection(jp));
                        this.preparedStatement.execute();
                        break block28;
                    }
                    throw new SQLException("Malformed GeoJSON file. Found '" + dataType + "'");
                }
                if (dataType.equalsIgnoreCase(GeoJsonField.FEATURES)) {
                    this.parseFeatures(jp);
                    break block28;
                }
                throw new SQLException("Malformed GeoJSON file. Found '" + dataType + "'");
            }
            catch (FileNotFoundException ex) {
                throw new SQLException(ex);
            }
            finally {
                try {
                    if (is != null) {
                        is.close();
                    }
                }
                catch (IOException ex) {
                    throw new SQLException(ex);
                }
            }
        }
    }

    private int readCRS(JsonParser jp) throws IOException, SQLException {
        int srid = 0;
        jp.nextToken();
        jp.nextToken();
        jp.nextToken();
        String firstField = jp.getText();
        if (firstField.equalsIgnoreCase(GeoJsonField.NAME)) {
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            String crsURI = jp.getText();
            if (crsURI.toLowerCase().startsWith(GeoJsonField.CRS_URN_EPSG)) {
                String[] split = crsURI.toLowerCase().split(GeoJsonField.CRS_URN_EPSG);
                if (split != null) {
                    srid = Integer.valueOf(split[1]);
                } else {
                    log.debug("The CRS URN " + crsURI + " is not supported.");
                }
            } else if (crsURI.equalsIgnoreCase(GeoJsonField.CRS_URN_OGC)) {
                log.debug("Specification of coordinate reference systems has been removed,\n i.e., the \"crs\" member of [GJ2008] is no longer used. Assuming WGS84 CRS");
                srid = 4326;
            } else {
                log.debug("The CRS URN " + crsURI + " is not supported.");
            }
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
        } else if (firstField.equalsIgnoreCase(GeoJsonField.LINK)) {
            log.debug("Linked CRS is not supported.");
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
            jp.nextToken();
        } else {
            throw new SQLException("Malformed GeoJSON CRS element.");
        }
        return srid;
    }

    private String skipCRS(JsonParser jp) throws IOException {
        jp.nextToken();
        jp.skipChildren();
        jp.nextToken();
        return jp.getText();
    }

    private void parseArrayMetadata(JsonParser jp) throws IOException {
        JsonToken value = jp.nextToken();
        while (value != JsonToken.END_ARRAY) {
            if (value == JsonToken.START_OBJECT) {
                this.parseObjectMetadata(jp);
            } else if (value == JsonToken.START_ARRAY) {
                this.parseArrayMetadata(jp);
            }
            value = jp.nextToken();
        }
    }

    private void parseObjectMetadata(JsonParser jp) throws IOException {
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            JsonToken value = jp.nextToken();
            if (value == JsonToken.START_OBJECT) {
                this.parseObjectMetadata(jp);
                continue;
            }
            if (value != JsonToken.START_ARRAY) continue;
            this.parseArrayMetadata(jp);
        }
    }

    private void parseArray(JsonParser jp, StringBuilder sb) throws IOException {
        sb.append(jp.currentToken().asCharArray());
        JsonToken value = jp.nextToken();
        String sep = ",";
        while (value != JsonToken.END_ARRAY) {
            if (value == JsonToken.START_OBJECT) {
                this.parseObject(jp, sb);
            } else if (value == JsonToken.END_OBJECT) {
                sb.append(jp.currentToken().asCharArray());
            } else if (value == JsonToken.START_ARRAY) {
                this.parseArray(jp, sb);
            } else if (value == JsonToken.VALUE_NULL) {
                sb.append("null");
            } else if (value == JsonToken.FIELD_NAME) {
                sb.append("\"").append(jp.getValueAsString()).append("\"");
                sep = ":";
            } else if (value == JsonToken.VALUE_STRING) {
                sb.append("\"").append(jp.getValueAsString()).append("\"");
                sep = ",";
            } else {
                sb.append(jp.getValueAsString());
                sep = ",";
            }
            if ((value = jp.nextToken()) == JsonToken.END_ARRAY || value == JsonToken.END_OBJECT) continue;
            sb.append(sep);
        }
        sb.append(jp.currentToken().asCharArray());
    }

    private void parseObject(JsonParser jp, StringBuilder sb) throws IOException {
        sb.append(jp.currentToken().asCharArray());
        JsonToken value = jp.nextToken();
        String sep = ",";
        while (value != JsonToken.END_OBJECT) {
            if (value == JsonToken.START_OBJECT) {
                this.parseObject(jp, sb);
                value = jp.nextToken();
                if (value == JsonToken.END_ARRAY || value == JsonToken.END_OBJECT) continue;
                sb.append(",");
                continue;
            }
            if (value == JsonToken.START_ARRAY) {
                sep = "[";
            } else if (value == JsonToken.FIELD_NAME) {
                sb.append("\"").append(jp.getValueAsString()).append("\"");
                sep = ":";
            } else if (value == JsonToken.VALUE_STRING) {
                sb.append("\"").append(jp.getValueAsString()).append("\"");
                sep = ",";
            } else {
                sb.append(jp.getValueAsString());
                sep = ",";
            }
            if ((value = jp.nextToken()) == JsonToken.END_ARRAY || value == JsonToken.END_OBJECT) continue;
            sb.append(sep);
        }
        sb.append(jp.currentToken().asCharArray());
    }

    private String parseObject(JsonParser jp) throws IOException {
        Object ret = "{";
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            JsonToken value = jp.nextToken();
            if (value == JsonToken.START_OBJECT) {
                this.parseObjectMetadata(jp);
                continue;
            }
            if (value != JsonToken.START_ARRAY) continue;
            this.parseArrayMetadata(jp);
        }
        ret = (String)ret + "}";
        return ret;
    }

    private static String getSQLTypeName(int sqlType) throws SQLException {
        switch (sqlType) {
            case 0: 
            case 12: {
                return "VARCHAR";
            }
            case 16: {
                return "BOOLEAN";
            }
            case 8: {
                return "DOUBLE PRECISION";
            }
            case -5: {
                return "BIGINT";
            }
            case 2003: {
                return "JSON";
            }
        }
        throw new SQLException("Unkown data type");
    }

    static {
        log = LoggerFactory.getLogger(GeoJsonReaderDriver.class);
    }
}

