# -*- coding: utf-8 -*-

"""
/***************************************************************************
 ProcessingUMEP
                                 A QGIS plugin
 UMEP for processing toolbox
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-04-02
        copyright            : (C) 2020 by Fredrik Lindberg
        email                : fredrikl@gvc.gu.se
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Fredrik Lindberg'
__date__ = '2020-04-02'
__copyright__ = '(C) 2020 by Fredrik Lindberg'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (QgsProcessing,
                       QgsProcessingAlgorithm,
                    #    QgsProcessingParameterPoint,
                       QgsProcessingParameterString,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterFolderDestination,
                    #    QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterField,
                    #    QgsProcessingParameterVectorDestination,
                    #    QgsProcessingParameterExtent,
                       QgsProcessingException,
                       QgsVectorLayer,
                       QgsFeature,
                    #    QgsGeometry,
                    #    QgsPointXY,
                       QgsVectorFileWriter,
                       QgsVectorDataProvider,
                       QgsField)

from qgis.PyQt.QtGui import QIcon
from osgeo import gdal, osr, ogr
from osgeo.gdalconst import *
import os
import numpy as np
import inspect
from pathlib import Path
import sys
from ..util import misc
from ..util import landCoverFractions_v1 as land
# # from ..util import RoughnessCalcFunctionV2 as rg
# from ..util import imageMorphometricParms_v1 as morph



class ProcessingLandCoverFractionAlgorithm(QgsProcessingAlgorithm):
    """
    This algorithm is a processing version of Image Morphometric Calculator Point
    """

    INPUT_POLYGONLAYER = 'INPUT_POLYGONLAYER'
    ID_FIELD = 'ID_FIELD'
    SERACH_METHOD = 'SEARCH_METHOD'
    INPUT_DISTANCE = 'INPUT_DISTANCE'
    INPUT_INTERVAL = 'INPUT_INTERVAL'
    INPUT_LCGRID = 'INPUT_LCGRID'
    FILE_PREFIX = 'FILE_PREFIX'
    OUTPUT_DIR = 'OUTPUT_DIR'
    IGNORE_NODATA = 'IGNORE_NODATA'
    ATTR_TABLE = 'ATTR_TABLE'
    
    
    def initAlgorithm(self, config):
       
        self.search = ((self.tr('Search throughout the grid extent (search distance not used)'), '0'),
                        (self.tr('Search from grid centroid'), '1'))
        self.addParameter(QgsProcessingParameterFeatureSource(self.INPUT_POLYGONLAYER,
            self.tr('Vector polygon grid'), [QgsProcessing.TypeVectorPolygon]))
        self.addParameter(QgsProcessingParameterField(self.ID_FIELD,
            self.tr('ID field'),'', self.INPUT_POLYGONLAYER, QgsProcessingParameterField.Numeric))
        self.addParameter(QgsProcessingParameterEnum(self.SERACH_METHOD,
            self.tr('Search method'),
            options=[i[0] for i in self.search], defaultValue=0))
        self.addParameter(QgsProcessingParameterNumber(self.INPUT_DISTANCE, 
            self.tr('Search distance from grid cell centroid (meter)'),
            QgsProcessingParameterNumber.Integer,
            QVariant(200), False, minValue=0))
        self.addParameter(QgsProcessingParameterNumber(self.INPUT_INTERVAL, 
            self.tr('Wind direction search interval (degree)'), 
            QgsProcessingParameterNumber.Double,
            QVariant(5), False, minValue=0.1, maxValue=360.))
        self.addParameter(QgsProcessingParameterRasterLayer(self.INPUT_LCGRID,
            self.tr('UMEP formatted land cover grid (see help for more info)'), None, False))
        self.addParameter(QgsProcessingParameterString(self.FILE_PREFIX, 
            self.tr('File prefix')))
        self.addParameter(QgsProcessingParameterBoolean(self.IGNORE_NODATA,
            self.tr("Ignore NoData pixels"), defaultValue=False))
        self.addParameter(QgsProcessingParameterBoolean(self.ATTR_TABLE,
            self.tr("Add result to polygon grid attribute table"), defaultValue=False))
        self.addParameter(QgsProcessingParameterFolderDestination(self.OUTPUT_DIR, 
            self.tr('Output folder')))

        self.plugin_dir = os.path.dirname(__file__)
        if not (os.path.isdir(self.plugin_dir + '/data')):
            os.mkdir(self.plugin_dir + '/data')
        self.dir_poly = self.plugin_dir + '/data/poly_temp.shp'

    def processAlgorithm(self, parameters, context, feedback):
        # InputParameters
        inputPolygonlayer = self.parameterAsVectorLayer(parameters, self.INPUT_POLYGONLAYER, context)
        idField = self.parameterAsFields(parameters, self.ID_FIELD, context)
        searchMethod = self.parameterAsString(parameters, self.SERACH_METHOD, context)
        inputDistance = self.parameterAsDouble(parameters, self.INPUT_DISTANCE, context)
        inputInterval = self.parameterAsDouble(parameters, self.INPUT_INTERVAL, context)
        dsmlayer = None
        filePrefix = self.parameterAsString(parameters, self.FILE_PREFIX, context)
        attrTable = self.parameterAsBool(parameters, self.ATTR_TABLE, context)
        ignoreNodata = self.parameterAsBool(parameters, self.IGNORE_NODATA, context)
        outputDir = self.parameterAsString(parameters, self.OUTPUT_DIR, context)

        if parameters['OUTPUT_DIR'] == 'TEMPORARY_OUTPUT':
            if not (os.path.isdir(outputDir)):
                os.mkdir(outputDir)

        r = inputDistance
        
        degree = float(inputInterval)
        pre = filePrefix
        ret = 0
        imp_point = 0
        imid = int(searchMethod)
        arrmat = np.empty((1, 8))

        header = 'Wd Paved Buildings EvergreenTrees DecidiousTrees Grass Baresoil Water'
        numformat = '%3d %5.3f %5.3f %5.3f %5.3f %5.3f %5.3f %5.3f'
        header2 = 'ID Paved Buildings EvergreenTrees DecidiousTrees Grass Baresoil Water'
        numformat2 = '%3d %5.3f %5.3f %5.3f %5.3f %5.3f %5.3f %5.3f'

        # temporary fix for mac, ISSUE #15
        pf = sys.platform
        if pf == 'darwin' or pf == 'linux2' or pf == 'linux':
            if not os.path.exists(outputDir + '/' + pre):
                os.makedirs(outputDir + '/' + pre)

        poly = inputPolygonlayer
        poly_field = idField
        feedback.setProgressText("poly_field: " + str(poly_field))
        vlayer = inputPolygonlayer
        prov = vlayer.dataProvider()
        fields = prov.fields()
        idx = vlayer.fields().indexFromName(poly_field[0])
        dir_poly = self.plugin_dir + '/data/poly_temp.shp'
        nGrids = vlayer.featureCount()
        index = 1
        feedback.setProgressText("Number of grids to analyse: " + str(nGrids))

        feedback.setProgressText("idx: " + str(idx))

        for f in vlayer.getFeatures():  # looping through each grid polygon
            feedback.setProgress(int((index * 100) / nGrids))
            if feedback.isCanceled():
                feedback.setProgressText("Calculation cancelled")
                break
            
            index += 1

            attributes = f.attributes()
            geometry = f.geometry()
            feature = QgsFeature()
            feature.setAttributes(attributes)
            feature.setGeometry(geometry)

            if imid == 1:  # use center point
                r = inputDistance
                y = f.geometry().centroid().asPoint().y()
                x = f.geometry().centroid().asPoint().x()
            else:
                r = 0  # Uses as info to separate from IMP point to grid
                writer = QgsVectorFileWriter(self.dir_poly, "CP1250", fields, prov.wkbType(),
                                                prov.crs(), "ESRI shapefile")
                if writer.hasError() != QgsVectorFileWriter.NoError:
                    raise QgsProcessingException("Error when creating shapefile: ", str(writer.hasError()))
                writer.addFeature(feature)
                del writer

            dsmlayer = self.parameterAsRasterLayer(parameters, self.INPUT_LCGRID, context)
            if dsmlayer is None:
                raise QgsProcessingException("No valid building land cover raster layer is selected")

            provider = dsmlayer.dataProvider()
            filePath_dsm_build = str(provider.dataSourceUri())

            if imid == 1:
                bbox = (x - r, y + r, x + r, y - r)
            else:
                # Remove gdalwarp cuttoline with gdal.Translate. Cut to envelope of polygon feature
                VectorDriver = ogr.GetDriverByName("ESRI Shapefile")
                Vector = VectorDriver.Open(self.dir_poly, 0)  #self.dir_poly
                layer = Vector.GetLayer()
                feature = layer.GetFeature(0)
                geom = feature.GetGeometryRef()
                minX, maxX, minY, maxY = geom.GetEnvelope()
                bbox = (minX, maxY, maxX, minY)  # Reorder bbox to use with gdal_translate
                Vector.Destroy()

            # Remove gdalwarp with gdal.Translate
            bigraster = gdal.Open(filePath_dsm_build)
            gdal.Translate(self.plugin_dir + '/data/clipdsm.tif', bigraster, projWin=bbox)
            bigraster = None

            dataset = gdal.Open(self.plugin_dir + '/data/clipdsm.tif')
            lcgrid = dataset.ReadAsArray().astype(np.float)
            sizex = lcgrid.shape[0]
            sizey = lcgrid.shape[1]

            # feedback.setProgressText(str(bbox))
            geotransform = dataset.GetGeoTransform()
            scale = 1 / geotransform[1]
            nd = dataset.GetRasterBand(1).GetNoDataValue()
            nodata_test = (lcgrid == nd)
            if ignoreNodata:
                if np.sum(lcgrid) == (lcgrid.shape[0] * lcgrid.shape[1] * nd):
                    feedback.setProgressText("Grid " + str(f.attributes()[idx]) + " not calculated. Includes Only NoData Pixels")
                    cal = 0
                else:
                    lcgrid[lcgrid== nd] = np.mean(lcgrid)
                    lcgrid[lcgrid == nd] = np.mean(lcgrid)
                    cal = 1
            else:
                if nodata_test.any():
                    feedback.setProgressText("Grid " + str(f.attributes()[idx]) + " not calculated. Includes NoData Pixels")
                    cal = 0
                else:
                    cal = 1

            if cal == 1:
                landcoverresult = land.landcover_v1(lcgrid, imid, degree, feedback, imp_point)
                landcoverresult = self.resultcheck(landcoverresult)
                # immorphresult = morph.imagemorphparam_v2(dsm_array, dem_array, scale, imid, degree, feedback, imp_point)

                arr = np.concatenate((landcoverresult["deg"], landcoverresult["lc_frac"]), axis=1)
                np.savetxt(outputDir + '/' + pre + '_' + 'LCFG_anisotropic_result_' + str(f.attributes()[idx]) + '.txt', arr,
                            fmt=numformat, delimiter=' ', header=header, comments='')
                del arr
                arr2 = np.array([f.attributes()[idx], landcoverresult["lc_frac_all"][0, 0], landcoverresult["lc_frac_all"][0, 1],
                                    landcoverresult["lc_frac_all"][0, 2], landcoverresult["lc_frac_all"][0, 3], landcoverresult["lc_frac_all"][0, 4],
                                    landcoverresult["lc_frac_all"][0, 5], landcoverresult["lc_frac_all"][0, 6]])

                arrmat = np.vstack([arrmat, arr2])

            dataset = None
            dataset2 = None
            dataset3 = None

        arrmatsave = arrmat[1: arrmat.shape[0], :]
        np.savetxt(outputDir + '/' + pre + '_' + 'LCFG_isotropic.txt', arrmatsave,
                            fmt=numformat2, delimiter=' ', header=header2, comments='')

        if attrTable:
            feedback.setProgressText("Adding result to layer attribute table") 
            self.addattr(vlayer, arrmatsave, header, pre, feedback, idx)

        return {self.OUTPUT_DIR: outputDir}

   
    def addattr(self, vlayer, matdata, header, pre, feedback, idx):
        current_index_length = len(vlayer.dataProvider().attributeIndexes())
        caps = vlayer.dataProvider().capabilities()

        # feedback.setProgressText("matdata:" + str(matdata))

        if caps & QgsVectorDataProvider.AddAttributes:
            line_split = header.split()
            for x in range(1, len(line_split)):
                vlayer.dataProvider().addAttributes([QgsField(pre + '_' + line_split[x], QVariant.Double)])
                vlayer.commitChanges()
                vlayer.updateFields()
            attr_dict = {}
        else:
            raise QgsProcessingException("Vector Layer does not support adding attributes")

        features = vlayer.getFeatures()

        for f in features:
            attr_dict.clear()
            id = f.id()
            # feedback.setProgressText("id:" + str(id))
            # feedback.setProgressText("idx:" + str(f.attributes()[idx]))
            wo = np.where(f.attributes()[idx] == matdata[:, 0])
            # feedback.setProgressText("wo:" + str(wo[0]))
            if wo[0] >= 0:
                for x in range(1, matdata.shape[1]):
                    attr_dict[current_index_length + x - 1] = float(matdata[wo[0], x])
                # feedback.setProgressText("attr_dict:" + str(attr_dict))
                vlayer.dataProvider().changeAttributeValues({id: attr_dict})


    # def addattributes(self, vlayer, matdata, header, pre,feedback):
    #     current_index_length = len(vlayer.dataProvider().attributeIndexes())
    #     caps = vlayer.dataProvider().capabilities()

    #     feedback.setProgressText("current_index_length:" + str(current_index_length))

    #     if caps & QgsVectorDataProvider.AddAttributes:
    #         line_split = header.split()
    #         for x in range(1, len(line_split)):

    #             vlayer.dataProvider().addAttributes([QgsField(pre + '_' + line_split[x], QVariant.Double)])
    #             vlayer.commitChanges()
    #             vlayer.updateFields()

    #         attr_dict = {}

    #         feedback.setProgressText("matdata:" + str(matdata))

    #         for y in range(0, matdata.shape[0]):
    #             attr_dict.clear()
    #             idx = int(matdata[y, 0])
    #             feedback.setProgressText("idx:" + str(idx))
    #             for x in range(1, matdata.shape[1]):
    #                 attr_dict[current_index_length + x - 1] = float(matdata[y, x])
    #                 feedback.setProgressText("attr_dict:" + str(attr_dict))
    #             vlayer.dataProvider().changeAttributeValues({idx: attr_dict})

    #         vlayer.commitChanges()
    #         vlayer.updateFields()

    #         # self.iface.mapCanvas().refresh()
    #     else:
    #         raise QgsProcessingException("Vector Layer does not support adding attributes")

    def resultcheck(self, landcoverresult):
        total = 0.
        arr = landcoverresult["lc_frac_all"]

        for x in range(0, len(arr[0])):
            total += arr[0, x]

        if total != 1.0:
            diff = total - 1.0

            maxnumber = max(arr[0])

            for x in range(0, len(arr[0])):
                if maxnumber == arr[0, x]:
                    arr[0, x] -= diff
                    break

        landcoverresult["lc_frac_all"] = arr

        return landcoverresult
    
    def name(self):
        return 'Urban Land Cover: Land Cover Fraction (Grid)'

    def displayName(self):
        return self.tr(self.name())

    def group(self):
        return self.tr(self.groupId())

    def groupId(self):
        return 'Pre-Processor'

    def shortHelpString(self):
        return self.tr('The Land Cover Fraction (Grid) plugin calculates land cover fractions required for UMEP (see Land Cover Reclassifier) '
        'from a point location based on a land cover raster grid. A land cover grid suitable for the processor in UMEP can be derived using '
        'the Land Cover Classifier. The fraction will vary depending on what angle (wind direction) you are interested in. Thus, this plugin '
        'is able to derive the land cover fractions for different directions. It is the same as the Land Cover Fraction (Point) except that '
        'this plugin calculates the fractions for each polygon object in polygon vector layer. The polygons should preferable be squares or '
        'any other regular shape. To create such a grid, built in functions in QGIS can be used (see Vector geometry -> Create Grid in the '
        'Processing Toolbox).\n'
        '--------------\n'
        'Full manual available via the <b>Help</b>-button.')

    def helpUrl(self):
        url = "https://umep-docs.readthedocs.io/en/latest/pre-processor/Urban%20Land%20Cover%20Land%20Cover%20Fraction%20(Grid).html"
        return url

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def icon(self):
        cmd_folder = Path(os.path.split(inspect.getfile(inspect.currentframe()))[0]).parent
        icon = QIcon(str(cmd_folder) + "/icons/LandCoverFractionGridIcon.png")
        return icon

    def createInstance(self):
        return ProcessingLandCoverFractionAlgorithm()