import os

from scipy.io import loadmat

from qgis.core import (
    NULL,
    Qgis,
    QgsProcessingContext,
    QgsProcessingException,
    QgsProcessingParameterDefinition,
    QgsProcessingParameterMultipleLayers,
)
from qgis.gui import (
    QgsAbstractProcessingParameterWidgetWrapper,
    QgsCheckableComboBox,
    QgsProcessingGui,
)
from qgis.PyQt.QtCore import QCoreApplication, pyqtSignal
from qgis.PyQt.QtWidgets import QComboBox


def tr(value: str) -> str:
    """
    Translate a string using Qt translation API.
    """
    return QCoreApplication.translate("ProcessingSwanProvider", value)


class QgsProcessingMatFieldWidgetWrapper(QgsAbstractProcessingParameterWidgetWrapper):
    """
    This class wraps a QComboBox widget for selecting fields from a .mat file.
    """

    widgetValueHasChanged = pyqtSignal(QgsAbstractProcessingParameterWidgetWrapper)

    def __init__(self, param, parent, *args, **kwargs):
        super().__init__(param, parent=parent)
        self.__widget = None

    def createWidgetWrapper(
        self,
        parameter: QgsProcessingParameterDefinition,
        type: QgsProcessingGui.WidgetType = QgsProcessingGui.WidgetType.Standard,  # noqa: A002
        parent=None,
    ):
        """
        Creates the wrapped widget for the parameter.
        """
        if type == QgsProcessingGui.WidgetType.Standard:
            return QgsProcessingMatFieldWidgetWrapper(parameter, parent)
        else:
            raise QgsProcessingException(
                "Le mode de traitement {} n'est pas supporté pour ce widget.".format(type)
            )

    def createWrappedWidget(self, context: QgsProcessingContext | None = None):
        """
        Creates the wrapped widget for the parameter.
        This method is called to create the widget that will be used in the GUI.
        """
        if self.parameterDefinition().allowMultipleValues():
            self.__widget = QgsCheckableComboBox()
            self.__widget.checkedItemsChanged.connect(self.widgetValueChanged)
        else:
            self.__widget = QComboBox()
            self.__widget.currentIndexChanged.connect(self.widgetValueChanged)
        return self.__widget

    def wrappedWidget(self):
        """
        Returns the wrapped widget.
        This method is used to access the actual widget that is being wrapped.
        """
        if self.__widget is not None:
            return self.__widget
        return None

    def widgetValueChanged(self):
        """
        Emits the signal when the widget value changes.
        """
        self.widgetValueHasChanged.emit(self)

    def setWidgetValue(
        self, value: str | list[str] | None, context: QgsProcessingContext | None = None
    ):
        """
        Sets the value of the widget.
        """
        widget: QComboBox | QgsCheckableComboBox = self.wrappedWidget()
        if widget is None:
            return

        if value is None or value == NULL:
            return

        if type(widget) is QComboBox:
            # Try to find by data first
            index = widget.findData(value)
            if index != -1:
                widget.setCurrentIndex(index)
                return

            # Try to find by text
            index = widget.findText(value)
            if index != -1:
                widget.setCurrentIndex(index)
                return
        elif type(widget) is QgsCheckableComboBox:
            # For QgsCheckableComboBox, we need to set the checked items
            widget.deselectAllOptions()
            if isinstance(value, list):
                widget.setCheckedItems(value)
            else:
                widget.setCheckedItems([value])
            return

    def widgetValue(self) -> str | list[str] | None:
        """
        Returns the value of the widget.
        """
        widget = self.wrappedWidget()
        if widget is None:
            return None
        if type(widget) is QgsCheckableComboBox:
            # For QgsCheckableComboBox, return the checked items
            return widget.checkedItems()
        elif type(widget) is QComboBox:
            # For QComboBox, return the current data
            if widget.currentIndex() == -1:
                return None
            # Return the data associated with the current item
            return widget.currentData()

    def postInitialize(self, wrappers: list[QgsAbstractProcessingParameterWidgetWrapper]):
        super().postInitialize(wrappers)
        for wrapper in wrappers:
            if (
                wrapper.parameterDefinition().name()
                == self.parameterDefinition().parentMatFileParameterName()
            ):
                self.setParentMatFileWrapperValue(wrapper)
                wrapper.widgetValueHasChanged.connect(self.setParentMatFileWrapperValue)

    def setParentMatFileWrapperValue(
        self, parentWrapper: QgsAbstractProcessingParameterWidgetWrapper
    ):
        """
        Sets the value of the parent .mat file wrapper.
        """
        widget = self.wrappedWidget()
        if widget is None:
            return

        self.widgetContext().messageBar().clearWidgets()
        widget.clear()
        if type(widget) is QgsCheckableComboBox:
            widget.deselectAllOptions()

        if parentWrapper is None:
            return

        # Get the value from the parent widget
        mat_path = parentWrapper.parameterValue()
        if not self.__validate_mat_path(mat_path):
            return

        # Load data from the .mat file
        data = self.__getDataFromMat(mat_path)
        if data is None:
            # Message already shown in __getDataFromMat
            return
        # Add all non-magic keys
        for key in data.keys():
            if not key.startswith("__"):
                self.wrappedWidget().addItem(str(key), key)

        if (
            type(self.wrappedWidget()) is QComboBox
            and self.parameterDefinition().defaultValueForGui()
        ):
            self.setWidgetValue(self.parameterDefinition().defaultValueForGui())
        elif (
            type(self.wrappedWidget()) is QgsCheckableComboBox
            and self.parameterDefinition().selectAllValuesByDefault()
        ):
            # Sélectionner tous les champs sauf "Xp" et "Yp"
            fieldsToSelect = [
                self.wrappedWidget().itemData(i)
                for i in range(self.wrappedWidget().count())
                if self.wrappedWidget().itemData(i) not in ["Xp", "Yp"]
            ]
            self.setWidgetValue(fieldsToSelect)

    def __validate_mat_path(self, value):
        """
        Validates the .mat file path.
        """

        if not value or value == NULL:
            return False

        valid = False
        if isinstance(value, str) and (value.endswith(".mat") or os.path.exists(value)):
            valid = True
        elif isinstance(value, list):
            valid = all(
                isinstance(v, str) and (v.endswith(".mat") or os.path.exists(v)) for v in value
            )

        if not valid:
            self.widgetContext().messageBar().pushMessage(
                "",
                tr(
                    "One or more selected files are invalid",
                    level=Qgis.MessageLevel.Warning,
                    duration=3,
                ),
            )
        return valid

    def __getDataFromMat(self, mat_path):
        """
        Loads data from a .mat file.
        """
        if isinstance(mat_path, list):
            # If multiple files are selected, check that all have the same keys
            all_keys = []
            for i, file_path in enumerate(mat_path):
                data = loadmat(str(file_path))
                keys = [key for key in data.keys() if not key.startswith("__")]
                if i == 0:
                    all_keys = set(keys)
                else:
                    current_keys = set(keys)
                    if all_keys != current_keys:
                        if self.widgetContext().messageBar():
                            self.widgetContext().messageBar().pushMessage(
                                "",
                                tr(
                                    "All .mat files must have the same field names",
                                    level=Qgis.MessageLevel.Warning,
                                    duration=3,
                                ),
                            )
                        return None

            # Use keys from the first file since all are the same
            mat_path = mat_path[0]

        try:
            data = loadmat(str(mat_path))
            return data
        except Exception as e:
            if self.widgetContext().messageBar():
                self.widgetContext().messageBar().pushMessage(
                    "",
                    tr(
                        f"Could not load .mat file: {e}",
                        level=Qgis.MessageLevel.Warning,
                        duration=3,
                    ),
                )
            return None


class QgsProcessingParameterMatField(QgsProcessingParameterDefinition):
    """
    A parameter definition for a field from a .mat file.
    """

    def __init__(
        self,
        name,
        description,
        defaultValue=None,
        parentMatFileParameterName=None,
        allowMultipleValues=False,
        selectAllValuesByDefault=False,
        optional=False,
    ):
        super().__init__(name, description, defaultValue, optional)
        self.__parentMatFileParameterName = parentMatFileParameterName
        self.__allowMultipleValues = allowMultipleValues
        self.__selectAllValuesByDefault = selectAllValuesByDefault
        self.__optional = optional
        self.setMetadata({"widget_wrapper": {"class": QgsProcessingMatFieldWidgetWrapper}})

    def clone(self):
        """
        Returns a clone of the parameter definition.
        """
        return QgsProcessingParameterMatField(
            self.name(),
            self.description(),
            self.defaultValue(),
            self.parentMatFileParameterName(),
            self.isOptional(),
            self.allowMultipleValues(),
        )

    def isOptional(self):
        """
        Returns True if the parameter is optional.
        """
        return self.__optional

    def allowMultipleValues(self):
        """
        Returns True if the parameter allows multiple values.
        """
        return self.__allowMultipleValues

    def selectAllValuesByDefault(self):
        """
        Returns True if all values should be selected by default.
        """
        return self.__selectAllValuesByDefault

    def dependsOnOtherParameter(self):
        """
        Returns True if this parameter depends on another parameter.
        """
        return self.parentMatFileParameterName() is not None

    def type(self):  # noqa: A003
        """
        Returns the type of the parameter.
        """
        return "mat_field"

    def parentMatFileParameterName(self):
        """
        Returns the name of the parent .mat file parameter.
        """
        return self.__parentMatFileParameterName

    def setParentMatFileParameterName(self, parentMatFileParameterName):
        """
        Sets the name of the parent .mat file parameter.
        """
        self.__parentMatFileParameterName = parentMatFileParameterName

    def checkValueIsAcceptable(self, value, context=None):
        """
        Check if the value is acceptable for this parameter.
        """
        if not value or value is None or value == NULL:
            return not self.isOptional()

        # For field parameters, any non-empty string is acceptable
        # The actual validation happens when the algorithm runs
        if self.allowMultipleValues():
            return isinstance(value, list) and all(
                isinstance(v, str) and len(v.strip()) > 0 for v in value
            )
        else:
            # For single value, it should be a non-empty string
            return isinstance(value, str) and len(value.strip()) > 0

    def valueAsPythonString(self, value, context=None):
        """
        Returns the value as a Python string representation.
        """
        if value is None or value == NULL:
            return "None"
        if self.allowMultipleValues():
            return "[" + ", ".join(f"'{v}'" for v in value) + "]"
        else:
            # For single value, return the string representation
            return f"'{value}'"


class QgsProcessingParameterMultipleMatFile(QgsProcessingParameterMultipleLayers):
    def createFileFilter(self):
        return "Fichier .mat (*.mat)"

    def checkValueIsAcceptable(self, value, context=None):
        """
        Check if the value is acceptable for this parameter.
        """
        if not value or value is None or value == NULL:
            return False

        # For .mat files, check if the path ends with .mat or exists
        if isinstance(value, str) and (value.endswith(".mat") or os.path.exists(value)):
            return True
        elif isinstance(value, list):
            return all(
                isinstance(v, str) and (v.endswith(".mat") or os.path.exists(v)) for v in value
            )

    def valueAsPythonString(self, value, context=None):
        """
        Returns the value as a Python string representation.
        """
        if value is None or value == NULL:
            return "None"
        if isinstance(value, list):
            return "[" + ", ".join(f"'{v}'" for v in value) + "]"
        else:
            return f"'{value}'"
