from qgis import processing
from qgis.analysis import QgsGridFileWriter, QgsInterpolator, QgsTinInterpolator
from qgis.core import (
    Qgis,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingParameterCrs,
    QgsProcessingParameterDefinition,
    QgsProcessingParameterFile,
    QgsProcessingParameterNumber,
    QgsProcessingParameterRasterDestination,
    QgsProcessingParameterString,
    QgsVectorLayer,
)
from qgis.PyQt.QtCore import QCoreApplication

from processing_swan_provider.core.processing_parameters import QgsProcessingParameterMatField


class MatToRasterAlgorithm(QgsProcessingAlgorithm):
    """
    This algorithm converts SWAN .mat files to raster format
    for use in QGIS.
    """

    # Constants used to refer to parameters and outputs
    INPUT = "INPUT"
    OUTPUT = "OUTPUT"
    FIELD = "FIELD"
    RESOLUTION = "RESOLUTION"
    X_FIELD = "X_FIELD"
    Y_FIELD = "Y_FIELD"
    CRS = "CRS"
    CREATE_OPTION = "CREATE_OPTION"

    def createInstance(self):
        return MatToRasterAlgorithm()

    def tr(self, message: str) -> str:
        """
        Get the translation for a string using Qt translation API.
        """
        return QCoreApplication.translate(self.__class__.__name__, message)

    def name(self):
        """
        Returns the algorithm name.
        """
        return "mat_to_raster"

    def displayName(self):
        """
        Returns the translated algorithm name.
        """
        return self.tr("SWAN .mat to Raster")

    def group(self):
        """
        Returns the name of the group this algorithm belongs to.
        """
        return self.tr("SWAN Postprocessing")

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to.
        """
        return "swan_postprocessing"

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm.
        """
        return self.tr("Converts SWAN .mat files to raster format for use in QGIS.")

    def initAlgorithm(self, config=None):
        """
        Define the inputs and outputs of the algorithm.
        """
        # Add the input parameters
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT,
                self.tr("Input SWAN .mat file"),
                extension="mat",
            )
        )

        # Add the field selection parameter
        self.addParameter(
            QgsProcessingParameterMatField(
                self.FIELD,
                self.tr("Field to convert"),
                defaultValue=None,
                parentMatFileParameterName=self.INPUT,
            )
        )

        # Add resolution parameter
        self.addParameter(
            QgsProcessingParameterNumber(
                self.RESOLUTION,
                self.tr("Resolution"),
                defaultValue=1.0,
            )
        )

        # Add parameter for CRS
        self.addParameter(
            QgsProcessingParameterCrs(
                self.CRS,
                self.tr("Coordinate Reference System"),
                defaultValue="EPSG:2154",  # RGF93 / Lambert-93
            )
        )

        # Add parameters for X and Y fieldsS
        param = QgsProcessingParameterMatField(
            self.X_FIELD,
            self.tr("SWAN .mat X field"),
            defaultValue="Xp",
            parentMatFileParameterName=self.INPUT,
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced)
        self.addParameter(param)

        param = QgsProcessingParameterMatField(
            self.Y_FIELD,
            self.tr("SWAN .mat Y field"),
            defaultValue="Yp",
            parentMatFileParameterName=self.INPUT,
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced)
        self.addParameter(param)

        # Add creation option for the output raster if QGIS version >= 3.44
        if Qgis.QGIS_VERSION_INT >= 34400:
            param = QgsProcessingParameterString(
                self.CREATE_OPTION,
                self.tr("Creation options for output raster"),
                defaultValue=None,
                multiLine=False,
                optional=True,
            )
            param.setMetadata(
                {
                    "widget_wrapper": {
                        "widget_type": "rasteroptions",
                    }
                }
            )
            param.setFlags(param.flags() | QgsProcessingParameterDefinition.Flag.FlagAdvanced)
            self.addParameter(param)

        # Add the output parameter
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
                self.tr("Output raster"),
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Run the algorithm.
        """

        # Get parameters
        input_file = self.parameterAsFile(parameters, self.INPUT, context)
        field_name = self.parameterAsString(parameters, self.FIELD, context)
        resolution = self.parameterAsDouble(parameters, self.RESOLUTION, context)
        x_field = self.parameterAsString(parameters, self.X_FIELD, context)
        y_field = self.parameterAsString(parameters, self.Y_FIELD, context)
        output_file = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
        crs = self.parameterAsCrs(parameters, self.CRS, context)

        alg_params = {
            "INPUT": input_file,
            "X_FIELD": x_field,
            "Y_FIELD": y_field,
            "OUTPUT": QgsProcessing.TEMPORARY_OUTPUT,
            "CRS": crs,
        }
        vLayer: QgsVectorLayer = processing.run(
            "processing_swan_provider:mat_to_vector", alg_params
        )["OUTPUT"]

        if not vLayer.isValid():
            raise QgsProcessingException(self.tr("Failed to convert .mat file to vector layer."))

        bbox = vLayer.extent()
        cols = int((bbox.xMaximum() - bbox.xMinimum()) / resolution) + 1
        rows = int((bbox.yMaximum() - bbox.yMinimum()) / resolution) + 1

        field_index = vLayer.fields().lookupField(field_name)
        if field_index == -1:
            raise QgsProcessingException(
                self.tr(f"Field '{field_name}' not found in the .mat file.")
            )

        data = QgsInterpolator.LayerData()
        data.source = vLayer
        data.transformContext = context.transformContext()
        data.valueSource = QgsInterpolator.ValueSource.ValueAttribute
        data.interpolationAttribute = field_index
        data.sourceType = QgsInterpolator.SourceType.SourcePoints
        if Qgis.QGIS_VERSION_INT >= 34400:
            interpolationMethod = QgsTinInterpolator.TinInterpolation.Linear
        else:
            interpolationMethod = QgsTinInterpolator.Linear

        interpolator = QgsTinInterpolator([data], interpolationMethod, feedback)

        writer = QgsGridFileWriter(interpolator, output_file, vLayer.extent(), cols, rows)

        if Qgis.QGIS_VERSION_INT >= 34400:
            create_options = self.parameterAsString(parameters, self.CREATE_OPTION, context)
            if create_options:
                writer.setCreationOptions(create_options.split("|"))

        writer.writeFile(feedback)
        return {self.OUTPUT: output_file}
