# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PowerPin
                                 A QGIS plugin
 Plugin eksportuje pinezkę do serwiów tpy geoportal. Plugin export pin to popular geoportal like serives
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-03-15
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Łukasz Świątek
        email                : lukasz.swiatek1996@gmail.com
 ***************************************************************************/ 

/***************************************************************************
 *                                                                         * 
 *   This program is free software; you can redistribute it and/or modify  * 
 *   it under the terms of the GNU General Public License as published by  * 
 *   the Free Software Foundation; either version 2 of the License, or     * 
 *   (at your option) any later version.                                   * 
 *                                                                         * 
 ***************************************************************************/ 
"""
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QSize
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QApplication, QAbstractItemView, QToolButton, QMenu

from qgis.gui import QgsMapTool, QgsMapToolEmitPoint, QgsRubberBand
from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsWkbTypes, QgsMessageLog, Qgis, QgsSettings, QgsRectangle

from .resources import *
from .power_pin_dockwidget import PowerPinDockWidget

import math
import os.path
import webbrowser
import json

class PowerPin:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor."""
        self.iface = iface
        self.canvas = self.iface.mapCanvas()
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = self.tr(u'&Power Pin PL')
        self.toolbar = self.iface.addToolBar(u'PowerPin')
        self.toolbar.setObjectName(u'PowerPin')
        self.pluginIsActive = False
        self.dockwidget = None
        self.portals = []
        self.split_button = None 
        self.split_button_action = None
        
        # Load portals from JSON
        self.load_portals()
        
        # Initialize locale
        locale = QgsSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'PowerPin_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)
            
        self.clickTool = QgsMapToolEmitPoint(self.canvas)
        self.clickTool.canvasClicked.connect(self.canvasClicked)

    def load_portals(self):
        """Load portals configuration from JSON file."""
        json_path = os.path.join(self.plugin_dir, 'portals.json')
        try:
            if os.path.exists(json_path):
                with open(json_path, 'r', encoding='utf-8') as f:
                    self.portals = json.load(f)
            else:
                QgsMessageLog.logMessage(f"Portals config not found: {json_path}", "PowerPin", Qgis.Warning)
        except Exception as e:
            QgsMessageLog.logMessage(f"Error loading portals: {e}", "PowerPin", Qgis.Critical)

    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        return QCoreApplication.translate('PowerPin', message)

    def add_action(self, icon_path, text, callback, enabled_flag=True, 
                   add_to_menu=True, add_to_toolbar=True, 
                   status_tip=None, whats_this=None, parent=None):
        """Add a toolbar icon to the toolbar."""
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)
        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)
        if add_to_menu:
            self.iface.addPluginToWebMenu(self.menu, action)

        self.actions.append(action)
        return action

    def refresh_gui(self):
        """Refreshes the GUI (Toolbar and Menu) without restarting."""
        for action in self.actions:
            self.iface.removePluginWebMenu(self.tr(u'&Power Pin PL'), action)
            self.toolbar.removeAction(action)
        
        # 2. Remove split button if exists
        if self.split_button_action:
            self.toolbar.removeAction(self.split_button_action)
            self.split_button_action = None
            if self.split_button:
                self.split_button.deleteLater()
                self.split_button = None
            
        self.actions = []
        
        # 3. Re-initialize GUI
        self.initGui()

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        
        # Main button (always present)
        icon_path = ':/plugins/power_pin/icons/map32.png'
        self.main_btn = self.add_action(
            icon_path,
            text=self.tr(u'Power Pin Settings'),
            callback=self.run,
            parent=self.iface.mainWindow()
        )
        
        settings = QgsSettings()
        # Default to True for compact mode
        compact_mode = settings.value("PowerPin/compact_mode", True, type=bool)
        
        if compact_mode:
            self.setup_compact_gui(settings)
        else:
            self.setup_expanded_gui(settings)

    def setup_expanded_gui(self, settings):
        """Sets up the classic multi-button interface."""
        settings.beginGroup("PowerPin/portals")
        
        # Legacy config support (optional check)
        legacy_path = os.path.join(self.plugin_dir, "power_pin_config.txt")
        if os.path.exists(legacy_path) and not settings.allKeys():
            try:
                with open(legacy_path, 'r', encoding='utf-8') as f:
                    for line in f:
                        parts = line.strip().split("\t")
                        if len(parts) > 1:
                            settings.setValue(parts[0], parts[1] == "ON")
            except Exception as e:
                pass

        for portal in self.portals:
            p_id = portal.get('id')
            p_name = portal.get('name', p_id.capitalize())
            
            is_on = settings.value(p_id, True, type=bool)
            if is_on:
                icon_path = f':/plugins/power_pin/icons/_{p_id}.png'
                self.add_action(
                    icon_path,
                    text=self.tr(p_name),
                    callback=lambda checked, p=portal: self.put_pin(p),
                    parent=self.iface.mainWindow()
                )
        settings.endGroup()

    def setup_compact_gui(self, settings):
        """Sets up the single split-button interface."""
        self.split_button = QToolButton()
        self.split_button.setPopupMode(QToolButton.MenuButtonPopup)
        # self.split_button.setToolButtonStyle(Qt.ToolButtonTextBesideIcon) # Optional
        
        menu = QMenu(self.split_button)
        
        settings.beginGroup("PowerPin/portals")
        last_used_id = QgsSettings().value("PowerPin/last_used_portal", "google", type=str)
        default_action = None
        first_action = None
        
        for portal in self.portals:
            p_id = portal.get('id')
            p_name = portal.get('name', p_id.capitalize())
            
            is_on = settings.value(p_id, True, type=bool)
            if is_on:
                icon_path = f':/plugins/power_pin/icons/_{p_id}.png'
                icon = QIcon(icon_path)
                
                # Create raw QAction
                action = QAction(icon, self.tr(p_name), self.iface.mainWindow())
                
                # Connect: 1. Put Pin, 2. Update Button
                # Note: We use a helper function to bind variables correctly
                action.triggered.connect(lambda checked, p=portal, a=action: self.activate_portal(p, a))
                
                menu.addAction(action)
                
                if first_action is None:
                    first_action = action
                
                if p_id == last_used_id:
                    default_action = action

        settings.endGroup()

        # Fallback if last used is hidden or invalid
        if default_action is None:
            default_action = first_action

        if default_action:
            self.split_button.setDefaultAction(default_action)
        
        self.split_button.setMenu(menu)
        
        # Add to toolbar and store the ACTION, not just the widget
        self.split_button_action = self.toolbar.addWidget(self.split_button)

    def activate_portal(self, portal_config, action):
        """Handles portal activation in Compact Mode."""
        # 1. Execute logic
        self.put_pin(portal_config)
        
        # 2. Update split button (icon and default action)
        if self.split_button:
            self.split_button.setDefaultAction(action)
            
        # 3. Save as last used
        QgsSettings().setValue("PowerPin/last_used_portal", portal_config['id'])

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""
        if self.dockwidget:
            self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)
        self.pluginIsActive = False

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginWebMenu(self.tr(u'&Power Pin PL'), action)
            self.toolbar.removeAction(action)
        
        # Correctly remove the split button action
        if self.split_button_action:
            self.toolbar.removeAction(self.split_button_action)
            self.split_button_action = None
            
        del self.toolbar

    def put_pin(self, portal_config):
        p_name = portal_config.get('name', portal_config.get('id'))
        
        msg = "First point is pin location"
        if portal_config.get('requires_angle', False):
            msg += ", second (drag or click again) is direction"
            
        self.iface.messageBar().pushMessage(
            f"Select location for {p_name}:", 
            msg, 
            level=Qgis.Success, 
            duration=3
        )
        
        # Load dynamic mode setting (Default is now TRUE)
        dynamic_mode = QgsSettings().value("PowerPin/dynamic_view", True, type=bool)
        
        tool = PointTool(self.iface.mapCanvas(), portal_config, dynamic_mode)
        self.iface.mapCanvas().setMapTool(tool)

    def canvasClicked(self, point):
        coords = f"{point.y():.3f}\t{point.x():.3f}"
        actual_crs = self.canvas.mapSettings().destinationCrs().authid()
        self.canvas.unsetMapTool(self.clickTool)
        QgsMessageLog.logMessage(str(coords), 'Power Clipboard')
        
        clipboard = QApplication.clipboard()
        clipboard.setText(coords)
        
        self.iface.messageBar().pushMessage(
            "Copied to clipboard:", 
            f"{coords} // {actual_crs}", 
            level=Qgis.Success, 
            duration=2
        )

    def save_config(self):
        if not self.dockwidget:
            return
            
        settings = QgsSettings()
        
        # Save dynamic view setting
        settings.setValue("PowerPin/dynamic_view", self.dockwidget.cb_dynamic_view.isChecked())
        # Save compact mode setting
        settings.setValue("PowerPin/compact_mode", self.dockwidget.cb_compact_mode.isChecked())

        settings.beginGroup("PowerPin/portals")
        
        # Get selected items (MultiSelection)
        # First set all to False
        for portal in self.portals:
             settings.setValue(portal['id'], False)

        # Then set selected to True
        for item in self.dockwidget.listWidget.selectedItems():
            settings.setValue(item.data(Qt.UserRole), True)
            
        settings.endGroup()
        
        self.iface.messageBar().pushMessage(
            "Configuration Saved", 
            "GUI Refreshed.", 
            level=Qgis.Info, 
            duration=3
        )
        
        # Refresh GUI immediately
        self.refresh_gui()

    def run(self):
        """Run method that loads and starts the plugin"""
        if not self.pluginIsActive:
            self.pluginIsActive = True
            if self.dockwidget is None:
                self.dockwidget = PowerPinDockWidget()
                self.dockwidget.pushButton.clicked.connect(self.save_config)
                # self.dockwidget.pushButton_2.clicked.connect(lambda: webbrowser.open_new('https://www.paypal.com/donate/?hosted_button_id=2AFDC9PRMGN3Q'))
                self.dockwidget.pushButton_3.clicked.connect(lambda: webbrowser.open_new('http://github.com/Rzezimioszek/QGIS-power-pin'))
                self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            self.dockwidget.listWidget.clear()
            settings = QgsSettings()
            
            # Load dynamic setting (Default True)
            dynamic_mode = settings.value("PowerPin/dynamic_view", True, type=bool)
            self.dockwidget.cb_dynamic_view.setChecked(dynamic_mode)

            # Load compact mode setting (Default True)
            compact_mode = settings.value("PowerPin/compact_mode", True, type=bool)
            self.dockwidget.cb_compact_mode.setChecked(compact_mode)
            
            settings.beginGroup("PowerPin/portals")
            
            self.dockwidget.listWidget.setSelectionMode(QAbstractItemView.MultiSelection)
            
            for portal in self.portals:
                p_id = portal.get('id')
                p_name = portal.get('name', p_id)
                
                item = QtWidgets.QListWidgetItem(p_name)
                item.setData(Qt.UserRole, p_id)
                self.dockwidget.listWidget.addItem(item)
                
                is_on = settings.value(p_id, True, type=bool)
                item.setSelected(is_on)
            
            settings.endGroup()
            
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()


class PointTool(QgsMapTool):
    def __init__(self, canvas, portal_config, dynamic_mode=True):
        QgsMapTool.__init__(self, canvas)
        self.canvas = canvas
        self.portal_config = portal_config
        self.dynamic_mode = dynamic_mode
        self.pressed = False
        self.has_line = False
        self.rb = QgsRubberBand(canvas, QgsWkbTypes.PointGeometry)
        self.rb.setColor(Qt.red)
        self.rl = QgsRubberBand(canvas, QgsWkbTypes.LineGeometry)
        self.rl.setColor(Qt.red)
        self.point0 = None
        self.point1 = None

    def canvasPressEvent(self, event):
        if not self.pressed:
            self.pressed = True
            self.point0 = self.toMapCoordinates(event.pos())
            self.rb.addPoint(self.point0)

    def canvasMoveEvent(self, event):
        if self.pressed:
            self.point1 = self.toMapCoordinates(event.pos())
            self.rl.reset(QgsWkbTypes.LineGeometry)
            self.rl.addPoint(self.point0)
            self.rl.addPoint(self.point1)
            self.has_line = True

    def canvasReleaseEvent(self, event):
        self.pressed = False
        self.has_line = False
        
        if not self.point0:
            return

        if not self.point1:
            self.point1 = self.point0

        self.open_url()
        
        self.rb.reset()
        self.rl.reset()
        self.canvas.unsetMapTool(self)

    def _get_transform(self, target_crs_authid):
        source_crs = self.canvas.mapSettings().destinationCrs()
        dest_crs = QgsCoordinateReferenceSystem(target_crs_authid)
        return QgsCoordinateTransform(source_crs, dest_crs, QgsProject.instance())

    def _get_angle(self):
        angle = math.atan2(self.point1.x() - self.point0.x(), self.point1.y() - self.point0.y())
        return math.degrees(angle) if angle > 0 else (math.degrees(angle) + 180) + 180

    def _calculate_zoom(self):
        scale = self.canvas.scale()
        if scale <= 0: return 18
        zoom = math.log2(591657550.5 / scale)
        return int(max(0, min(21, zoom)))

    def open_url(self):
        target_crs_authid = self.portal_config.get('crs', 'EPSG:4326')
        url_template = self.portal_config.get('url')
        
        if not url_template:
            return

        transform = self._get_transform(target_crs_authid)
        pt = transform.transform(self.point0)
        angle = self._get_angle()

        zoom = 19
        
        if self.dynamic_mode:
            zoom = self._calculate_zoom()
            
            canvas_extent = self.canvas.extent()
            transformed_extent = transform.transformBoundingBox(canvas_extent)
            
            xmin = transformed_extent.xMinimum()
            ymin = transformed_extent.yMinimum()
            xmax = transformed_extent.xMaximum()
            ymax = transformed_extent.yMaximum()
            
        else:
            xmin = pt.x() - 100
            ymin = pt.y() - 100
            xmax = pt.x() + 100
            ymax = pt.y() + 100

        params = {
            'x': pt.x(),
            'y': pt.y(),
            'angle': angle,
            'zoom': zoom,
            'lat': pt.y(),
            'lon': pt.x(),
            'xmin': xmin,
            'ymin': ymin,
            'xmax': xmax,
            'ymax': ymax
        }
        
        try:
            url = url_template.format(**params)
            QgsMessageLog.logMessage(f"Opening URL: {url}", "PowerPin")
            webbrowser.open_new(url)
        except Exception as e:
            QgsMessageLog.logMessage(f"Error formatting URL for {self.portal_config.get('id')}: {e}", "PowerPin", Qgis.Warning)
