# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PowerClipboard
                                 A QGIS plugin
 Plugin give extended clipboard option to work with coords.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-04-16
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Łukasz Świątek
        email                : lukasz.swiatek1996@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, qVersion, QCoreApplication, QVariant, Qt, QPoint
from qgis.PyQt.QtGui import QIcon, QPixmap, QKeySequence
from qgis.PyQt.QtWidgets import QAction, QToolBar, QShortcut, QWidget, QLabel, QApplication, QFrame, QVBoxLayout
from qgis.gui import QgsMessageBar, QgsMapToolEmitPoint, QgsDockWidget, QgsMapTool, QgsMapCanvas
from qgis.core import Qgis, QgsVectorLayer, QgsGeometry, QgsFeature, QgsProject, QgsField, \
    QgsCoordinateReferenceSystem, QgsPoint, QgsCoordinateTransform, QgsMessageLog, QgsRectangle
# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .power_clipboard_dockwidget import PowerClipboardDockWidget
import os.path


class ZoomMagnifierTool(QgsMapTool):
    def __init__(self, canvas, callback):
        super().__init__(canvas)
        self.canvas = canvas
        self.callback = callback
        self.magnifier = None
        self.mag_canvas = None

    def create_magnifier(self):
        self.magnifier = QFrame(self.canvas.parentWidget(), Qt.Window | Qt.FramelessWindowHint | Qt.WindowStaysOnTopHint)
        self.magnifier.setFixedSize(250, 250)
        self.magnifier.setStyleSheet("border: 1px solid black;")
        
        layout = QVBoxLayout(self.magnifier)
        layout.setContentsMargins(0, 0, 0, 0)
        
        self.mag_canvas = QgsMapCanvas(self.magnifier)
        self.mag_canvas.setLayers(self.canvas.layers())
        self.mag_canvas.setDestinationCrs(self.canvas.mapSettings().destinationCrs())
        self.mag_canvas.enableAntiAliasing(True)
        
        layout.addWidget(self.mag_canvas)
        
        # Add crosshair lines (full span)
        # Center is 125 (250/2)
        
        # Horizontal line
        line_h = QFrame(self.magnifier)
        line_h.setFixedSize(250, 1) # Full width, 1px height
        line_h.setStyleSheet("background-color: black; border: none;")
        line_h.move(0, 125)
        
        # Vertical line
        line_v = QFrame(self.magnifier)
        line_v.setFixedSize(1, 250) # 1px width, full height
        line_v.setStyleSheet("background-color: black; border: none;")
        line_v.move(125, 0)
        
        self.magnifier.show()

    def canvasMoveEvent(self, event):
        point = self.toMapCoordinates(event.pos())
        
        if self.magnifier is None:
            self.create_magnifier()
        
        # Position magnifier near cursor
        pos = self.canvas.mapToGlobal(event.pos())
        self.magnifier.move(pos.x() + 20, pos.y() + 20)
        
        # Set extent for magnifier (10x zoom)
        # We want a fixed size in map units roughly.
        # Use current map scale/resolution to determine window size in map units
        # Window is 250px.
        map_units_per_pixel = self.canvas.mapUnitsPerPixel()
        
        # We want a zoomed in view, say 4x zoom compared to main canvas
        zoom_factor = 4
        view_width = (250 * map_units_per_pixel) / zoom_factor
        view_height = (250 * map_units_per_pixel) / zoom_factor
        
        rect = QgsRectangle(point.x() - view_width/2, point.y() - view_height/2, 
                            point.x() + view_width/2, point.y() + view_height/2)
        
        self.mag_canvas.setExtent(rect)
        self.mag_canvas.refresh()

    def canvasReleaseEvent(self, event):
        point = self.toMapCoordinates(event.pos())
        self.callback(point)
        self.deactivate()

    def deactivate(self):
        if self.magnifier:
            self.magnifier.close()
            self.magnifier = None
        super().deactivate()


class PowerClipboard:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'PowerClipboard_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Power Clipboard')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'PowerClipboard')
        self.toolbar.setObjectName(u'PowerClipboard')

        #print "** INITIALIZING PowerClipboard"

        ###
        self.iface = iface
        self.canvas = self.iface.mapCanvas()
        
        # out click tool will emit a QgsPoint on every click
        self.clickTool = ZoomMagnifierTool(self.canvas, self.canvasClicked)
        ###

        self.pluginIsActive = False
        self.dockwidget = None

        self.reverse = False


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('PowerClipboard', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/power_clipboard/icon.png'

        """self.add_action(
            icon_path,
            text=self.tr(u'Power Clipboard'),
            callback=self.run,
            parent=self.iface.mainWindow())"""
        

        icon_path = ':/plugins/power_clipboard/icons/copyXY.png'
        
        btn = self.add_action(
                    icon_path,
                    text=self.tr(f'Copy XY coords'),
                    #callback=lambda: self.put_pin(portal),
                    callback=lambda: self.copy_to_clip(False),
                    parent=self.iface.mainWindow() #
                )
                
        icon_path = ':/plugins/power_clipboard/icons/copyYX.png'
        
        btn = self.add_action(
                    icon_path,
                    text=self.tr(f'Copy YX coords'),
                    #callback=lambda: self.put_pin(portal),
                    callback=lambda: self.copy_to_clip(True),
                    parent=self.iface.mainWindow() #
                )
        
        icon_path = ':/plugins/power_clipboard/icons/zoomXY.png'
        
        btn = self.add_action(
                    icon_path,
                    text=self.tr(f'Zoom to XY'),
                    #callback=lambda: self.put_pin(portal),
                    callback=lambda: self.zoom_to(False),
                    parent=self.iface.mainWindow() #
                )
        
        icon_path = ':/plugins/power_clipboard/icons/zoomYX.png'
        
        btn = self.add_action(
                    icon_path,
                    text=self.tr(f'Zoom to YX'),
                    #callback=lambda: self.put_pin(portal),
                    callback=lambda: self.zoom_to(True),
                    parent=self.iface.mainWindow() #
                )
        


    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING PowerClipboard"

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD PowerClipboard"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Power Clipboard'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    #--------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""

        self.canvas.setMapTool(self.clickTool)

    def copy_to_clip(self, reverse):

        self.reverse = reverse

        self.iface.messageBar().pushMessage( "Select point", "and copy it to clipboard", level=Qgis.Success, duration=2)
        self.canvas.setMapTool(self.clickTool)



    def canvasClicked(self, point):

        # dodać przyciąganie według obecnych ustawień
        reverse = self.reverse
        self.canvas.unsetMapTool(self.clickTool)

        # QgsMessageLog.logMessage(str(reverse), 'Power Clipboard')
        
        if reverse:
            coords = f"{point.y():.3f}\t{point.x():.3f}"
        else:
            coords = f"{point.x():.3f}\t{point.y():.3f}"

        QgsMessageLog.logMessage(str(coords), 'Power Clipboard')
        QApplication.clipboard().setText(coords)
        self.iface.messageBar().pushMessage("Copied to clipboard:", f"{coords}", level=Qgis.Success, duration=2)

    def zoom_to(self, reverse):

        values = QApplication.clipboard().text()

        values = values.strip()

        if "." in values and "," in values:
           values = values.replace(",", "\t")

        values = values.replace(",", ".")
        values = values.replace(" ", "\t")
        splt= values.split("\t")

        a, b = 1, 0

        if reverse:
            a, b = 0, 1

        try:
            x = float(splt[a])
            y = float(splt[b])

            zm = 10

            extent = QgsRectangle(y-zm,x-zm,y+zm,x+zm)
            # self.canvas.setFixedSize(extent.width(), extent.height())

            self.canvas.setExtent(extent)
            self.canvas.refresh()

            self.iface.messageBar().pushMessage("Zoomed to:",
                                                 f"{x}\t{y}",
                                                   level=Qgis.Success, duration=5)
        except:
            self.iface.messageBar().pushMessage("Can't paste coords:",
                                                 f"Clipboard value is wrong coords format: '{values}'",
                                                   level=Qgis.Critical, duration=5)