# -*- coding: utf-8 -*-
# *****************************************************************************************
# *   Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier  *
# *   selon les termes de la Licence Publique Générale GNU telle que publiée par          *
# *   la Free Software Foundation ; soit la version 2 de la Licence, ou                   *
# *   (à votre choix) toute version ultérieure.                                           *
# *****************************************************************************************

import os
import psycopg2
from qgis.PyQt.QtCore import Qt, QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import (
    QAction, QDialog, QVBoxLayout, QFormLayout,
    QLabel, QPushButton, QComboBox, QLineEdit, QMessageBox
)
from qgis.core import Qgis, QgsApplication, QgsAuthMethodConfig, QgsMessageLog
from . import resources  # generated by pyrcc5


class PostgisAuthDialog(QDialog):
    """Fenêtre principale de test de connexion PostGIS"""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle(QCoreApplication.translate("PostgisAuthDialog", "PostGIS Connection Tester"))
        self.setWindowModality(Qt.ApplicationModal)
        self.resize(450, 380)

        layout = QVBoxLayout(self)
        self.setLayout(layout)

        # Retrieve PostGIS connections from QGIS settings
        settings = QSettings()
        settings.beginGroup("/PostgreSQL/connections")
        self.available_conns = settings.childGroups()
        settings.endGroup()

        if not self.available_conns:
            QMessageBox.critical(self,
                                 QCoreApplication.translate("PostgisAuthDialog", "Error"),
                                 QCoreApplication.translate("PostgisAuthDialog", "No PostGIS connections configured."))
            self.close()
            return

        self.conn_selector = QComboBox()
        self.conn_selector.addItems(self.available_conns)
        self.conn_selector.currentIndexChanged.connect(self.load_connection)
        layout.addWidget(QLabel(QCoreApplication.translate("PostgisAuthDialog", "Select a connection:")))
        layout.addWidget(self.conn_selector)

        # Connection info layout
        self.info_layout = QFormLayout()
        layout.addLayout(self.info_layout)
        self.labels = {}
        for field in ["service (conf file)", "host", "port", "database", "username", "password", "authcfg"]:
            lbl = QLabel()
            self.labels[field] = lbl
            self.info_layout.addRow(QLabel(QCoreApplication.translate("PostgisAuthDialog", "{} :").format(field.capitalize())), lbl)

        # Buttons
        self.test_button = QPushButton(QCoreApplication.translate("PostgisAuthDialog", "Test connection"))
        self.test_button.clicked.connect(self.test_connection)
        layout.addWidget(self.test_button)

        layout.addWidget(QLabel(QCoreApplication.translate("PostgisAuthDialog", "SQL query to test:")))
        self.sql_input = QLineEdit()
        self.sql_input.setPlaceholderText(QCoreApplication.translate("PostgisAuthDialog", "e.g. SELECT COUNT(*) FROM my_table;"))
        layout.addWidget(self.sql_input)

        self.sql_test_button = QPushButton(QCoreApplication.translate("PostgisAuthDialog", "Test SQL query"))
        self.sql_test_button.clicked.connect(self.test_sql)
        layout.addWidget(self.sql_test_button)

        close_button = QPushButton(QCoreApplication.translate("PostgisAuthDialog", "Close"))
        close_button.clicked.connect(self.close)
        layout.addWidget(close_button)

        self.load_connection(0)

    def load_connection(self, index):
        conn_name = self.available_conns[index]
        settings = QSettings()
        settings.beginGroup(f"/PostgreSQL/connections/{conn_name}")

        self.conn_name = conn_name
        self.params = {
            "service": settings.value("service", ""),
            "host": settings.value("host", ""),
            "port": settings.value("port", ""),
            "database": settings.value("database", ""),
            "username": settings.value("username", ""),
            "password": settings.value("password", ""),
            "authcfg": settings.value("authcfg", "")
        }
        settings.endGroup()

        # Load authentication config if defined
        if self.params["authcfg"]:
            auth_mgr = QgsApplication.authManager()
            cfg = QgsAuthMethodConfig()
            try:
                success = auth_mgr.loadAuthenticationConfig(self.params["authcfg"], cfg, True)
                if success and cfg.isValid():
                    auth_info = cfg.configMap()
                    self.params["username"] = auth_info.get("username", self.params["username"])
                    self.params["password"] = auth_info.get("password", self.params["password"])
                else:
                    QgsMessageLog.logMessage(
                        QCoreApplication.translate("PostgisAuthDialog", "⚠️ Auth config '{}' invalid for '{}'").format(
                            self.params['authcfg'], conn_name),
                        "PostGIS Auth", Qgis.Warning
                    )
            except Exception as e:
                QgsMessageLog.logMessage(
                    QCoreApplication.translate("PostgisAuthDialog", "Error loading authcfg vault '{}': {}").format(
                        self.params['authcfg'], e),
                    "PostGIS Auth", Qgis.Warning
                )
        else:
            # Plaintext credentials check
            if self.params.get("username") and self.params.get("password"):
                msg_text = (
                    "<html><body>"
                    "<p style='color:red; font-weight:bold; font-size:16pt;'>*** {0} ***</p>"
                    "<p>{1} '<b>{2}</b>' {3} </p>"
                    "<p><b>{4} :</b> {5}<br/><b>{6} :</b> {7}</p>"
                    "<p>⚠️ {8}</p>"
                    "</body></html>"
                ).format(
                    QCoreApplication.translate("PostgisAuthDialog", "WARNING!"),
                    QCoreApplication.translate("PostgisAuthDialog", "The credentials for connection"),
                    conn_name,
                    QCoreApplication.translate("PostgisAuthDialog", "are stored in plain text."),
                    QCoreApplication.translate("PostgisAuthDialog", "Username"),
                    self.params['username'],
                    QCoreApplication.translate("PostgisAuthDialog", "Password"),
                    self.params['password'],
                    QCoreApplication.translate("PostgisAuthDialog", "It is recommended to use an authentication vault (authcfg) for better security.")
                )

                QgsMessageLog.logMessage(
                    QCoreApplication.translate("PostgisAuthDialog", "Plain text credentials for '{}'").format(conn_name),
                    "PostGIS", Qgis.Warning
                )
                msg_box = QMessageBox(self)
                msg_box.setWindowTitle(QCoreApplication.translate("PostgisAuthDialog", "Plaintext credentials"))
                msg_box.setTextFormat(Qt.RichText)
                msg_box.setText(msg_text)
                msg_box.setIcon(QMessageBox.Warning)
                msg_box.setStandardButtons(QMessageBox.Ok)
                msg_box.exec_()
            else:
                msg_text = QCoreApplication.translate(
                    "PostgisAuthDialog",
                    "⚠️ No auth vault defined and no plaintext credentials for connection '{}'. The connection may fail."
                .format(conn_name))
                QgsMessageLog.logMessage(msg_text, "PostGIS", Qgis.Warning)
                QMessageBox.warning(self, QCoreApplication.translate("PostgisAuthDialog", "Connection warning"), msg_text)

        # Display connection parameters
        using_service = bool(self.params.get("service"))
        conf_file = None
        if using_service:
            possible_paths = [
                os.environ.get("PGSERVICEFILE"),
                os.path.expanduser("~/.pg_service.conf"),
                os.path.join(os.environ.get("APPDATA", ""), "postgresql", "pg_service.conf"),
                os.path.join(os.environ.get("PROGRAMDATA", ""), "postgresql", "pg_service.conf")
            ]
            conf_file = next((p for p in possible_paths if p and os.path.exists(p)), None)

        for k, lbl in self.labels.items():
            if k == "service (conf file)":
                lbl.setText(conf_file or QCoreApplication.translate("PostgisAuthDialog", "(pg_service.conf not found)")
                            if using_service else QCoreApplication.translate("PostgisAuthDialog", "Not used"))
            else:
                lbl.setText(QCoreApplication.translate("PostgisAuthDialog", "See conf file")
                            if using_service else self.params.get(k, QCoreApplication.translate("PostgisAuthDialog", "(empty)")))

    def test_connection(self):
        try:
            if self.params.get("service"):
                conn = psycopg2.connect(service=self.params["service"])
            else:
                conn = psycopg2.connect(
                    host=self.params.get("host") or None,
                    port=self.params.get("port") or None,
                    dbname=self.params.get("database"),
                    user=self.params.get("username") or None,
                    password=self.params.get("password") or None
                )
            conn.close()
            QMessageBox.information(
                self,
                QCoreApplication.translate("PostgisAuthDialog", "Success"),
                QCoreApplication.translate("PostgisAuthDialog", "Successfully connected to '{}' ✅").format(self.conn_name)
            )
        except Exception as e:
            self.show_error(
                QCoreApplication.translate("PostgisAuthDialog", "Unable to connect to database '{}':\n{}".format(0, 1))
                .format(self.params.get('database', ''), str(e))
            )

    def test_sql(self):
        query = self.sql_input.text().strip()
        if not query:
            QMessageBox.warning(self,
                                QCoreApplication.translate("PostgisAuthDialog", "Warning"),
                                QCoreApplication.translate("PostgisAuthDialog", "Please enter an SQL query."))
            return
        try:
            if self.params.get("service"):
                conn = psycopg2.connect(service=self.params["service"])
            else:
                conn = psycopg2.connect(
                    host=self.params.get("host") or None,
                    port=self.params.get("port") or None,
                    dbname=self.params.get("database"),
                    user=self.params.get("username") or None,
                    password=self.params.get("password") or None
                )
            cur = conn.cursor()
            cur.execute(query)
            results = cur.fetchall()
            cur.close()
            conn.close()
            QMessageBox.information(
                self,
                QCoreApplication.translate("PostgisAuthDialog", "SQL Result"),
                QCoreApplication.translate("PostgisAuthDialog", "Result:\n{}").format(results)
            )
        except Exception as e:
            self.show_error(QCoreApplication.translate("PostgisAuthDialog", "SQL Error: {}").format(e))

    def show_error(self, message):
        msg = QMessageBox(self)
        msg.setIcon(QMessageBox.Critical)
        msg.setWindowTitle(QCoreApplication.translate("PostgisAuthDialog", "Connection Error"))
        msg.setText(QCoreApplication.translate("PostgisAuthDialog", "Connection failed:"))
        msg.setInformativeText(message)
        msg.setStandardButtons(QMessageBox.Ok)
        msg.exec_()


class PostgisConnectionTesterPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.action = None
        self.translator = None

    def initGui(self):
        # Load translation file according to QGIS language
        locale = QSettings().value("locale/userLocale", "en")[0:2]
        locale_path = os.path.join(os.path.dirname(__file__), "i18n", f"translate_{locale}.qm")

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        icon = QIcon(":/postgis_connection_tester/icons/icon.svg")
        self.action = QAction(
            icon,
            QCoreApplication.translate("PostgisConnectionTesterPlugin", "PostGIS Connection Tester"),
            self.iface.mainWindow()
        )
        self.action.triggered.connect(self.run)
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu(
            QCoreApplication.translate("PostgisConnectionTesterPlugin", "&PostGIS Tools"),
            self.action
        )
        QgsMessageLog.logMessage(
            QCoreApplication.translate("PostgisConnectionTesterPlugin", "🔌 PostGIS Connection Tester plugin loaded"),
            "PostGIS",
            Qgis.Info
        )

    def unload(self):
        if self.action:
            self.iface.removeToolBarIcon(self.action)
            self.iface.removePluginMenu(
                QCoreApplication.translate("PostgisConnectionTesterPlugin", "&PostGIS Tools"),
                self.action
            )

    def run(self):
        dlg = PostgisAuthDialog(self.iface.mainWindow())
        dlg.exec_()
