# coding: utf-8
"""
 PolyStrip
                                 A QGIS plugin
 Polygons along lines 
                              -------------------
        begin                : 2017-07-29
        git sha              : $Format:%H$
        copyright            : (C) 2017 by Werner Macho
        email                : werner.macho@gmail.com
"""
from typing import Optional, List
from qgis.PyQt.QtCore import (
    QSettings,
    QTranslator,
    QFileInfo,
    qVersion,
    QCoreApplication
)
from qgis.PyQt.QtGui import (
    QIcon
)
from qgis.PyQt.QtWidgets import (
    QAction,
    QWidget
)
from qgis.core import (
    Qgis,
    QgsMapLayer,
    QgsWkbTypes
)
from qgis.gui import QgisInterface
# Import the code for the dialog
from .polystripdialog import PolyStripDialog

import os.path


class PolyStrip:
    """QGIS Plugin Implementation."""

    def __init__(self, iface: QgisInterface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            '{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&PolyStrip')
        self.toolbar = self.iface.addToolBar(u'PolyStrip')
        self.toolbar.setObjectName(u'PolyStrip')

    def show_warning(self, message: str) -> None:
        text = QCoreApplication.translate('Polystrip', message)
        mb = self.iface.messageBar()
        mb.pushWidget(mb.createMessage(text), Qgis.Warning, 5)
    
    # noinspection PyMethodMayBeStatic
    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.
        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('PolyStrip', message)

    def add_action(
            self,
            icon_path: str,
            text: str,
            callback,
            enabled_flag: bool = True,
            add_to_menu: bool = True,
            add_to_toolbar: bool = True,
            status_tip: Optional[str] = None,
            whats_this: Optional[str] = None,
            parent: Optional[QWidget] = None) -> QAction:
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self) -> None:
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = QFileInfo(__file__).absolutePath() + '/img/polystrip.svg'
        self.add_action(
            icon_path,
            text=self.tr(u'PolyStrip'),
            callback=self.run)

    def unload(self) -> None:
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&PolyStrip'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    def run(self) -> None:
        """Run method that performs all the real work"""
        # Check if there are any line layers
        has_line_layers = any(
            layer.type() == QgsMapLayer.VectorLayer and 
            layer.geometryType() == QgsWkbTypes.LineGeometry
            for layer in self.iface.mapCanvas().layers()
        )
        
        if not has_line_layers:
            message = self.tr(u'No layers with line features - polystrip needs a selected line feature!')
            self.show_warning(message)
            return
        
        # Validate active layer
        active_layer = self.iface.activeLayer()
        
        if not active_layer or active_layer.type() != QgsMapLayer.VectorLayer:
            message = self.tr(u'Select a Feature in a Vectorlayer')
            self.show_warning(message)
            return
            
        if active_layer.geometryType() != QgsWkbTypes.LineGeometry:
            message = self.tr(u'Select a Feature in a Linelayer')
            self.show_warning(message)
            return
            
        if active_layer.selectedFeatureCount() == 0:
            message = self.tr('No feature in active Layer selected!')
            self.show_warning(message)
            return

        # Create the dialog only when needed
        self.dlg = PolyStripDialog(iface=self.iface)
        
        # show the dialog
        self.dlg.show()

        # Run the dialog event loop
        result = self.dlg.exec()

        # See if OK was pressed
        if result:
            layer = self.iface.activeLayer()
            self.dlg.polystrip(layer)
