# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PolygonCompare
                                 A QGIS plugin
 Compares two polygon layers to find added, deleted and modified polygons by unique ID and geometry
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-06-07
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Anustup Jana
        email                : anustupjana21@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis import processing
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterField,
    QgsProcessingParameterFeatureSink,
    QgsProcessingException,
    QgsVectorLayer,
    QgsFeatureSink,
    QgsFeature,
    QgsFields,
    QgsWkbTypes,
    QgsProcessingContext,
    QgsProject,
    QgsProcessingFeatureSource,
    QgsFeatureRequest,
    QgsProcessingProvider,
    QgsApplication
)
import os.path

# Initialize Qt resources from file resources.py
from .resources import *

class ComparePolygonsAlgorithm(QgsProcessingAlgorithm):
    OLD_LAYER = 'OLD_LAYER'
    NEW_LAYER = 'NEW_LAYER'
    ID_FIELD = 'ID_FIELD'
    MODIFIED_OUTPUT = 'MODIFIED_OUTPUT'
    ADDED_OUTPUT = 'ADDED_OUTPUT'
    DELETED_OUTPUT = 'DELETED_OUTPUT'

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.OLD_LAYER,
                'Old Polygon Layer',
                [QgsProcessing.TypeVectorPolygon],
                optional=False
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.NEW_LAYER,
                'New Polygon Layer',
                [QgsProcessing.TypeVectorPolygon],
                optional=False
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.ID_FIELD,
                'Unique ID Field',
                defaultValue='id',
                type=QgsProcessingParameterField.Any,
                parentLayerParameterName=self.OLD_LAYER,
                allowMultiple=False,
                optional=False
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.MODIFIED_OUTPUT,
                'Modified Polygons',
                QgsProcessing.TypeVectorPolygon
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.ADDED_OUTPUT,
                'Added Polygons',
                QgsProcessing.TypeVectorPolygon
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.DELETED_OUTPUT,
                'Deleted Polygons',
                QgsProcessing.TypeVectorPolygon
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        try:
            old_source = self.parameterAsSource(parameters, self.OLD_LAYER, context)
            new_source = self.parameterAsSource(parameters, self.NEW_LAYER, context)
            id_field = self.parameterAsString(parameters, self.ID_FIELD, context)

            if old_source is None:
                raise QgsProcessingException("Old Polygon Layer is not provided.")
            if new_source is None:
                raise QgsProcessingException("New Polygon Layer is not provided.")

            old_layer = old_source.materialize(QgsFeatureRequest())
            new_layer = new_source.materialize(QgsFeatureRequest())

            feedback.pushInfo(f"Old Layer: {old_layer.name()}, Feature Count: {old_layer.featureCount()}, CRS: {old_layer.crs().authid()}")
            feedback.pushInfo(f"New Layer: {new_layer.name()}, Feature Count: {new_layer.featureCount()}, CRS: {new_layer.crs().authid()}")

            if not isinstance(old_layer, QgsVectorLayer):
                raise QgsProcessingException(f"Old layer is not a vector layer. Type: {type(old_layer)}")
            if old_layer.geometryType() != 2:
                raise QgsProcessingException(f"Old layer is not a polygon layer. Geometry Type: {old_layer.geometryType()}")
            if not isinstance(new_layer, QgsVectorLayer):
                raise QgsProcessingException(f"New layer is not a vector layer. Type: {type(new_layer)}")
            if new_layer.geometryType() != 2:
                raise QgsProcessingException(f"New layer is not a polygon layer. Geometry Type: {new_layer.geometryType()}")

            if id_field not in old_layer.fields().names():
                raise QgsProcessingException(f"ID field '{id_field}' not found in Old Polygon Layer. Fields: {old_layer.fields().names()}")
            if id_field not in new_layer.fields().names():
                raise QgsProcessingException(f"ID field '{id_field}' not found in New Polygon Layer. Fields: {new_layer.fields().names()}")

            if old_layer.featureCount() == 0:
                feedback.pushInfo("Old Polygon Layer is empty. No features to process.")
            if new_layer.featureCount() == 0:
                feedback.pushInfo("New Polygon Layer is empty. No features to process.")

            modified_fields = new_layer.fields()
            (modified_sink, modified_dest_id) = self.parameterAsSink(
                parameters, self.MODIFIED_OUTPUT, context,
                modified_fields, QgsWkbTypes.Polygon, new_layer.crs()
            )
            added_fields = new_layer.fields()
            (added_sink, added_dest_id) = self.parameterAsSink(
                parameters, self.ADDED_OUTPUT, context,
                added_fields, QgsWkbTypes.Polygon, new_layer.crs()
            )
            deleted_fields = old_layer.fields()
            (deleted_sink, deleted_dest_id) = self.parameterAsSink(
                parameters, self.DELETED_OUTPUT, context,
                deleted_fields, QgsWkbTypes.Polygon, old_layer.crs()
            )

            context.addLayerToLoadOnCompletion(
                modified_dest_id,
                QgsProcessingContext.LayerDetails('Modified Polygons', QgsProject.instance(), self.MODIFIED_OUTPUT)
            )
            context.addLayerToLoadOnCompletion(
                added_dest_id,
                QgsProcessingContext.LayerDetails('Added Polygons', QgsProject.instance(), self.ADDED_OUTPUT)
            )
            context.addLayerToLoadOnCompletion(
                deleted_dest_id,
                QgsProcessingContext.LayerDetails('Deleted Polygons', QgsProject.instance(), self.DELETED_OUTPUT)
            )

            old_features = {}
            invalid_old_ids = 0
            for f in old_layer.getFeatures():
                if f[id_field] is None or f[id_field] == '':
                    invalid_old_ids += 1
                    continue
                old_features[f[id_field]] = f
            new_features = {}
            invalid_new_ids = 0
            for f in new_layer.getFeatures():
                if f[id_field] is None or f[id_field] == '':
                    invalid_new_ids += 1
                    continue
                new_features[f[id_field]] = f

            if invalid_old_ids > 0:
                feedback.pushInfo(f"Skipped {invalid_old_ids} features in Old Layer with null or empty IDs.")
            if invalid_new_ids > 0:
                feedback.pushInfo(f"Skipped {invalid_new_ids} features in New Layer with null or empty IDs.")

            old_ids = set(old_features.keys())
            new_ids = set(new_features.keys())
            added_ids = new_ids - old_ids
            deleted_ids = old_ids - new_ids
            common_ids = old_ids & new_ids

            feedback.pushInfo(f"Found {len(added_ids)} added IDs, {len(deleted_ids)} deleted IDs, {len(common_ids)} common IDs.")

            total_steps = len(added_ids) + len(deleted_ids) + len(common_ids)
            if total_steps == 0:
                feedback.pushInfo("No differences found between layers. Output layers will be empty.")
                return {
                    self.MODIFIED_OUTPUT: modified_dest_id,
                    self.ADDED_OUTPUT: added_dest_id,
                    self.DELETED_OUTPUT: deleted_dest_id
                }
            current_step = 0

            added_count = 0
            for id in added_ids:
                if feedback.isCanceled():
                    raise QgsProcessingException("Processing canceled by user.")
                feature = new_features[id]
                if feature.geometry().isNull() or not feature.geometry().isGeosValid():
                    feedback.pushInfo(f"Skipped added feature ID {id} due to null or invalid geometry.")
                    continue
                added_sink.addFeature(feature, QgsFeatureSink.FastInsert)
                added_count += 1
                current_step += 1
                feedback.setProgress(100 * current_step / total_steps)

            deleted_count = 0
            for id in deleted_ids:
                if feedback.isCanceled():
                    raise QgsProcessingException("Processing canceled by user.")
                feature = old_features[id]
                if feature.geometry().isNull() or not feature.geometry().isGeosValid():
                    feedback.pushInfo(f"Skipped deleted feature ID {id} due to null or invalid geometry.")
                    continue
                deleted_sink.addFeature(feature, QgsFeatureSink.FastInsert)
                deleted_count += 1
                current_step += 1
                feedback.setProgress(100 * current_step / total_steps)

            modified_count = 0
            for id in common_ids:
                if feedback.isCanceled():
                    raise QgsProcessingException("Processing canceled by user.")
                old_feature = old_features[id]
                new_feature = new_features[id]
                if old_feature.geometry().isNull() or new_feature.geometry().isNull():
                    feedback.pushInfo(f"Skipped feature ID {id} due to null geometry.")
                    continue
                if not old_feature.geometry().isGeosValid() or not new_feature.geometry().isGeosValid():
                    feedback.pushInfo(f"Skipped feature ID {id} due to invalid geometry.")
                    continue
                if not old_feature.geometry().equals(new_feature.geometry()):
                    modified_sink.addFeature(new_feature, QgsFeatureSink.FastInsert)
                    modified_count += 1
                current_step += 1
                feedback.setProgress(100 * current_step / total_steps)

            feedback.pushInfo(f"Processed {added_count} added, {deleted_count} deleted, and {modified_count} modified polygons.")

            return {
                self.MODIFIED_OUTPUT: modified_dest_id,
                self.ADDED_OUTPUT: added_dest_id,
                self.DELETED_OUTPUT: deleted_dest_id
            }

        except Exception as e:
            raise QgsProcessingException(f"An error occurred: {str(e)}")

    def name(self):
        return 'comparepolygons'

    def displayName(self):
        return 'Compare Polygon Layers'

    def group(self):
        return 'Vector Analysis'

    def groupId(self):
        return 'vectoranalysis'

    def createInstance(self):
        return ComparePolygonsAlgorithm()

class PolygonCompareProvider(QgsProcessingProvider):
    def __init__(self):
        QgsProcessingProvider.__init__(self)

    def id(self):
        return 'polygoncompare'

    def name(self):
        return 'Polygon Compare'

    def loadAlgorithms(self):
        self.addAlgorithm(ComparePolygonsAlgorithm())

class PolygonCompare:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.provider = None
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'PolygonCompare_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        self.actions = []
        self.menu = self.tr(u'&Polygon Compare')
        self.first_start = None

    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        return QCoreApplication.translate('PolygonCompare', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar."""
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)
        return action

    def initProcessing(self):
        """Initialize the processing provider."""
        self.provider = PolygonCompareProvider()
        QgsApplication.processingRegistry().addProvider(self.provider)

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        self.add_action(
            icon_path,
            text=self.tr(u'Compare Polygon Layers'),
            callback=self.run,
            parent=self.iface.mainWindow(),
            status_tip='Compare two polygon layers to find added, deleted, and modified polygons'
        )
        self.initProcessing()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Polygon Compare'),
                action)
            self.iface.removeToolBarIcon(action)
        if self.provider:
            QgsApplication.processingRegistry().removeProvider(self.provider)

    def run(self):
        """Run method that performs all the real work."""
        processing.execAlgorithmDialog('polygoncompare:comparepolygons', {})