# -*- coding: UTF-8 -*- do QGIS -*-

from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QComboBox,
    QCheckBox, QLineEdit, QToolButton, QPushButton,
    QFileDialog, QTabWidget, QWidget, QTextEdit, QFrame,
    QApplication
)
from qgis.PyQt.QtGui import QIcon
from qgis.core import QgsProject, QgsVectorLayer

# -------------------------------
# Classe de processamento real
# -------------------------------
class PoliQuadratePlugin:

    def process_layer(self, camada, mark_centroid=False, show_lines=False,
                      table_format="CSV", output_path="", separate_quadrants=False,
                      log_callback=None):

        features = list(camada.getFeatures())
        total = len(features)

        if total == 0:
            if log_callback:
                log_callback("Nenhum polígono encontrado.")
            return

        layer_name_base = camada.name()  # Nome base da camada

        if log_callback:
            log_callback("Iniciando processamento...")

        QApplication.processEvents()

        # Marcar centroide
        if mark_centroid:
            centroid_layer_name = f"{layer_name_base} centroide"
            if log_callback:
                log_callback(f"Criando camada: {centroid_layer_name}")
            # Aqui adiciona a lógica de criação da camada de centroides

        # Mostrar linhas divisórias
        if show_lines:
            lines_layer_name = f"{layer_name_base} linhas"
            if log_callback:
                log_callback(f"Criando camada: {lines_layer_name}")
            # Aqui adiciona a lógica de criação da camada de linhas

        # Separar quadrantes
        if separate_quadrants:
            quadrants_layer_name = f"{layer_name_base} quadrantes"
            if log_callback:
                log_callback(f"Criando camada: {quadrants_layer_name}")
            # Aqui adiciona a lógica de criação da camada de quadrantes

        # Loop pelos polígonos (opcional)
        for i, feat in enumerate(features):
            geom = feat.geometry()
            if log_callback:
                log_callback(f"Processando polígono {i+1} de {total}")
            QApplication.processEvents()

        if log_callback:
            log_callback("Processamento concluído!")

        QApplication.processEvents()


# -------------------------------
# Dialog principal
# -------------------------------
class PoliQuadrateDialog(QDialog):

    def __init__(self, iface, plugin_instance, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.plugin_instance = plugin_instance
        self.setWindowTitle("PoliQuadrate")

        # -----------------
        # Abas
        # -----------------
        self.tabs = QTabWidget()
        self.tab_params = QWidget()
        self.tab_logs = QWidget()
        self.tabs.addTab(self.tab_params, "Parâmetros")
        self.tabs.addTab(self.tab_logs, "Registos")

        # -----------------
        # Aba Parâmetros
        # -----------------
        layout_params = QHBoxLayout()
        form_layout = QVBoxLayout()

        # Camadas
        hlayer = QHBoxLayout()
        hlayer.addWidget(QLabel("Camada de entrada:"))
        self.cb_layers = QComboBox()
        hlayer.addWidget(self.cb_layers)
        form_layout.addLayout(hlayer)

        # Checkboxes
        self.chk_centroid = QCheckBox("Marcar centroide visível")
        self.chk_lines = QCheckBox("Mostrar linhas divisórias")
        self.chk_separate_quadrants = QCheckBox("Gerar camadas separadas por linhas")
        form_layout.addWidget(self.chk_centroid)
        form_layout.addWidget(self.chk_lines)
        form_layout.addWidget(self.chk_separate_quadrants)

        # Tabela
        htable = QHBoxLayout()
        htable.addWidget(QLabel("Gerar tabela de resumo:"))
        self.cb_table = QComboBox()
        self.cb_table.addItems(["CSV", "Excel"])
        self.cb_table.setCurrentText("CSV")
        htable.addWidget(self.cb_table)
        form_layout.addLayout(htable)

        # Arquivo de saída
        hout = QHBoxLayout()
        hout.addWidget(QLabel("Arquivo de saída:"))
        self.le_output = QLineEdit()
        hout.addWidget(self.le_output)
        self.tb_output = QToolButton()
        self.tb_output.setText("...")
        hout.addWidget(self.tb_output)
        form_layout.addLayout(hout)

        # Botões
        hbtn = QHBoxLayout()
        self.btn_run = QPushButton("Executar")
        self.btn_cancel = QPushButton("Cancelar")
        self.btn_close = QPushButton("Fechar")
        hbtn.addWidget(self.btn_run)
        hbtn.addWidget(self.btn_cancel)
        hbtn.addWidget(self.btn_close)
        form_layout.addLayout(hbtn)

        layout_params.addLayout(form_layout)

        # Descrição à direita
        desc_frame = QFrame()
        desc_frame.setFrameShape(QFrame.StyledPanel)
        desc_layout = QVBoxLayout()
        desc_label = QLabel(
            "<h2>PoliQuadrate Plugin</h2>"
            "<p>O PoliQuadrate permite analisar polígonos, dividindo-os em quadrantes "
            "e gerando camadas separadas para cada quadrante.</p>"
            "<ul>"
            "<li><b>Marcar centroide:</b> Exibe o ponto central de cada polígono.</li>"
            "<li><b>Mostrar linhas divisórias:</b> Exibe as linhas que separam quadrantes.</li>"
            "<li><b>Gerar tabela de resumo:</b> Exporta resultados em CSV ou Excel.</li>"
            "<li><b>Separar quadrantes:</b> Cria camadas individuais para cada quadrante.</li>"
            "</ul>"
        )
        desc_label.setWordWrap(True)
        desc_layout.addWidget(desc_label)
        desc_layout.addStretch()
        desc_frame.setLayout(desc_layout)
        layout_params.addWidget(desc_frame)
        self.tab_params.setLayout(layout_params)

        # -----------------
        # Aba Registos com símbolos
        # -----------------
        log_layout = QVBoxLayout()
        self.text_logs = QTextEdit()
        self.text_logs.setReadOnly(True)
        log_layout.addWidget(self.text_logs)

        hicons = QHBoxLayout()
        hicons.addStretch()

        # 💾 Salvar registros
        btn_save = QToolButton()
        btn_save.setText("💾")
        btn_save.setToolTip("Salvar registros")
        btn_save.setStyleSheet("font-size: 18pt;")
        btn_save.clicked.connect(self.save_logs)
        hicons.addWidget(btn_save)

        # 📋 Copiar registros
        btn_copy = QToolButton()
        btn_copy.setText("📋")
        btn_copy.setToolTip("Copiar registros")
        btn_copy.setStyleSheet("font-size: 18pt;")
        btn_copy.clicked.connect(self.copy_logs)
        hicons.addWidget(btn_copy)

        # 🧹 Limpar registros
        btn_clear = QToolButton()
        btn_clear.setText("🧹")
        btn_clear.setToolTip("Limpar registros")
        btn_clear.setStyleSheet("font-size: 18pt;")
        btn_clear.clicked.connect(self.clear_logs)
        hicons.addWidget(btn_clear)

        log_layout.addLayout(hicons)
        self.tab_logs.setLayout(log_layout)

        # Layout principal
        main_layout = QVBoxLayout()
        main_layout.addWidget(self.tabs)
        self.setLayout(main_layout)

        # -----------------
        # Conectar sinais
        # -----------------
        self.tb_output.clicked.connect(self.choose_output_file)
        self.btn_run.clicked.connect(self.on_run_button_clicked)
        self.btn_cancel.clicked.connect(self.reject)
        self.btn_close.clicked.connect(self.close)

        # Carregar camadas
        self.populate_layers()

    # -----------------
    # Métodos de camada
    # -----------------
    def populate_layers(self):
        self.cb_layers.clear()
        layers = [l for l in QgsProject.instance().mapLayers().values() if isinstance(l, QgsVectorLayer)]
        for layer in layers:
            self.cb_layers.addItem(layer.name(), layer)
        ativo = self.iface.activeLayer()
        if ativo:
            index = self.cb_layers.findData(ativo)
            if index != -1:
                self.cb_layers.setCurrentIndex(index)

    def choose_output_file(self):
        filename, _ = QFileDialog.getSaveFileName(
            self,
            "Selecionar arquivo de saída",
            "",
            "CSV (*.csv);; Excel (*.xlsx);; Shapefile (*.shp)"
        )
        if filename:
            self.le_output.setText(filename)

    # -----------------
    # Executar processamento
    # -----------------
    def on_run_button_clicked(self):
        self.text_logs.clear()
        camada = self.cb_layers.currentData()
        if not camada:
            return

        self.plugin_instance.process_layer(
            camada,
            mark_centroid=self.chk_centroid.isChecked(),
            show_lines=self.chk_lines.isChecked(),
            table_format=self.cb_table.currentText(),
            output_path=self.le_output.text(),
            separate_quadrants=self.chk_separate_quadrants.isChecked(),
            log_callback=lambda msg: self._update_log(msg)
        )

    def _update_log(self, msg):
        self.text_logs.append(msg)

    # -----------------
    # Funções da aba Registos
    # -----------------
    def save_logs(self):
        filename, _ = QFileDialog.getSaveFileName(
            self, "Salvar registros", "", "Texto (*.txt);;CSV (*.csv)"
        )
        if filename:
            with open(filename, "w", encoding="utf-8") as f:
                f.write(self.text_logs.toPlainText())

    def copy_logs(self):
        QApplication.clipboard().setText(self.text_logs.toPlainText())

    def clear_logs(self):
        self.text_logs.clear()