# -*- coding: utf-8 -*-
"""
PoliQuadrate Plugin
Cria quadrantes, centroides, linhas divisórias e exporta tabela CSV.
"""

import os
import csv
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.core import (
    QgsProject,
    QgsVectorLayer,
    QgsFeature,
    QgsGeometry,
    QgsPointXY,
    QgsField
)
from .poli_quadrante_dialog import PoliQuadrateDialog


class PoliQuadrate:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = self.tr("&Poli Quadrante")

    def tr(self, message):
        return QCoreApplication.translate('PoliQuadrate', message)

    def add_action(self, icon_path, text, callback, parent=None):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        self.iface.addToolBarIcon(action)
        self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)
        return action

    def initGui(self):
        icon_path = os.path.join(self.plugin_dir, "icon.png")
        self.add_action(
            icon_path,
            text=self.tr("Poli Quadrante"),
            callback=self.run,
            parent=self.iface.mainWindow()
        )

    def unload(self):
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&Poli Quadrante"), action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        self.dlg = PoliQuadrateDialog(self.iface, self)
        self.dlg.show()
        self.dlg.exec_()

    # ==========================================================
    # ==================== PROCESSAMENTO ======================
    # ==========================================================
    def process_layer(
        self,
        layer,
        mark_centroid=False,
        show_lines=False,
        table_format="CSV",
        output_path=None,
        log_callback=None,
        separate_quadrants=False
    ):
        if not layer or not layer.isValid():
            return False

        designacoes = {1: "NO", 2: "NE", 3: "SO", 4: "SE"}

        # Camada principal de quadrantes (usada se separate_quadrants=False)
        quad_layer = QgsVectorLayer(
            f"Polygon?crs={layer.crs().authid()}",
            f"{layer.name()}_quadrantes",
            "memory"
        )
        quad_provider = quad_layer.dataProvider()
        fields = layer.fields().toList()
        fields.append(QgsField("Quadrante", QVariant.Int))
        fields.append(QgsField("Designacao", QVariant.String))
        fields.append(QgsField("Area", QVariant.Double))
        fields.append(QgsField("Percent", QVariant.Double))
        quad_provider.addAttributes(fields)
        quad_layer.updateFields()

        # Linhas / eixos
        if show_lines:
            lines_layer = QgsVectorLayer(
                f"LineString?crs={layer.crs().authid()}",
                f"{layer.name()}_linhas",
                "memory"
            )
            lines_provider = lines_layer.dataProvider()
            lines_provider.addAttributes([
                QgsField("Eixo", QVariant.String),
                QgsField("Valor", QVariant.Double)
            ])
            lines_layer.updateFields()

        # Centroides
        if mark_centroid:
            centroid_layer = QgsVectorLayer(
                f"Point?crs={layer.crs().authid()}",
                f"{layer.name()}_centroides",
                "memory"
            )
            centroid_provider = centroid_layer.dataProvider()

        # ---------------- PROCESSAMENTO ----------------
        for feat in layer.getFeatures():
            geom = feat.geometry()
            if not geom or geom.isEmpty() or geom.type() != 2:  # apenas polígonos
                continue

            centro = geom.centroid().asPoint()
            x_mid, y_mid = centro.x(), centro.y()
            bbox = geom.boundingBox()
            xmin, xmax = bbox.xMinimum(), bbox.xMaximum()
            ymin, ymax = bbox.yMinimum(), bbox.yMaximum()
            pol_area = geom.area()

            quadrantes = [
                [(xmin, y_mid), (x_mid, y_mid), (x_mid, ymax), (xmin, ymax)],  # NO
                [(x_mid, y_mid), (xmax, y_mid), (xmax, ymax), (x_mid, ymax)],  # NE
                [(xmin, ymin), (x_mid, ymin), (x_mid, y_mid), (xmin, y_mid)],  # SO
                [(x_mid, ymin), (xmax, ymin), (xmax, y_mid), (x_mid, y_mid)],  # SE
            ]

            for idx, pts in enumerate(quadrantes, start=1):
                poly_geom = QgsGeometry.fromPolygonXY([[QgsPointXY(x, y) for x, y in pts]])
                inter = geom.intersection(poly_geom)
                if inter.isEmpty():
                    continue

                # ================= SEPARATE QUADRANTS =================
                if separate_quadrants:
                    quad_sep_layer = QgsVectorLayer(
                        f"Polygon?crs={layer.crs().authid()}",
                        f"{layer.name()}_Q{idx}",
                        "memory"
                    )
                    quad_sep_provider = quad_sep_layer.dataProvider()
                    quad_sep_provider.addAttributes(quad_layer.fields())
                    quad_sep_layer.updateFields()

                    f_quad = QgsFeature()
                    f_quad.setFields(quad_sep_layer.fields())
                    f_quad.setGeometry(inter)
                    for field in layer.fields():
                        f_quad.setAttribute(field.name(), feat[field.name()])
                    f_quad.setAttribute(f_quad.fields().indexFromName("Quadrante"), idx)
                    f_quad.setAttribute(f_quad.fields().indexFromName("Designacao"), designacoes[idx])
                    f_quad.setAttribute(f_quad.fields().indexFromName("Area"), inter.area())
                    f_quad.setAttribute(
                        f_quad.fields().indexFromName("Percent"),
                        (inter.area() / pol_area * 100) if pol_area > 0 else 0
                    )
                    quad_sep_provider.addFeature(f_quad)
                    quad_sep_layer.updateExtents()
                    QgsProject.instance().addMapLayer(quad_sep_layer)

                else:
                    f_quad = QgsFeature()
                    f_quad.setFields(quad_layer.fields())
                    f_quad.setGeometry(inter)
                    for field in layer.fields():
                        f_quad.setAttribute(field.name(), feat[field.name()])
                    f_quad.setAttribute(f_quad.fields().indexFromName("Quadrante"), idx)
                    f_quad.setAttribute(f_quad.fields().indexFromName("Designacao"), designacoes[idx])
                    f_quad.setAttribute(f_quad.fields().indexFromName("Area"), inter.area())
                    f_quad.setAttribute(
                        f_quad.fields().indexFromName("Percent"),
                        (inter.area() / pol_area * 100) if pol_area > 0 else 0
                    )
                    quad_provider.addFeature(f_quad)

            # Linhas / Eixos
            if show_lines:
                # Linha horizontal
                h_line = QgsGeometry.fromPolylineXY([QgsPointXY(xmin, y_mid), QgsPointXY(xmax, y_mid)])
                h_inter = geom.intersection(h_line)
                if not h_inter.isEmpty():
                    f_h = QgsFeature()
                    f_h.setFields(lines_layer.fields())
                    f_h.setGeometry(h_inter)
                    f_h.setAttribute(f_h.fields().indexFromName("Eixo"), "horizontal")
                    f_h.setAttribute(f_h.fields().indexFromName("Valor"), y_mid)
                    lines_provider.addFeature(f_h)

                # Linha vertical
                v_line = QgsGeometry.fromPolylineXY([QgsPointXY(x_mid, ymin), QgsPointXY(x_mid, ymax)])
                v_inter = geom.intersection(v_line)
                if not v_inter.isEmpty():
                    f_v = QgsFeature()
                    f_v.setFields(lines_layer.fields())
                    f_v.setGeometry(v_inter)
                    f_v.setAttribute(f_v.fields().indexFromName("Eixo"), "vertical")
                    f_v.setAttribute(f_v.fields().indexFromName("Valor"), x_mid)
                    lines_provider.addFeature(f_v)

            # Centroides
            if mark_centroid:
                f_c = QgsFeature()
                f_c.setGeometry(geom.centroid())
                centroid_provider.addFeature(f_c)

        # Adicionar camadas
        if not separate_quadrants:
            quad_layer.updateExtents()
            QgsProject.instance().addMapLayer(quad_layer)
        if show_lines:
            lines_layer.updateExtents()
            QgsProject.instance().addMapLayer(lines_layer)
        if mark_centroid:
            centroid_layer.updateExtents()
            QgsProject.instance().addMapLayer(centroid_layer)

        # Exportar CSV
        if table_format and output_path:
            try:
                campos_csv = [field.name() for field in quad_layer.fields()]
                with open(output_path, mode='w', newline='', encoding='utf-8') as f:
                    writer = csv.writer(f)
                    writer.writerow(campos_csv)
                    for feat in quad_layer.getFeatures():
                        writer.writerow([feat[field] for field in campos_csv])
            except Exception as e:
                print(f"Erro ao gerar CSV: {e}")

        return True