# -*- coding: utf-8 -*-

"""
/***************************************************************************
 NearestNeighbor
                                 A QGIS plugin
 Calculates the distance to the nearest point
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-12-07
        copyright            : (C) 2020 by Mathias Gröbe
        email                : mathias.groebe@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Mathias Gröbe'
__date__ = '2020-12-07'
__copyright__ = '(C) 2020 by Mathias Gröbe'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterField,
                       QgsSpatialIndex,
                       QgsFeatureRequest,
                       QgsWkbTypes,
                       QgsDistanceArea)
import os
import math

class NearestNeighborAlgorithm(QgsProcessingAlgorithm):


    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'
    FIELD_FOR_DISTANCE = 'FIELD_FOR_DISTANCE'
    USE_ELLIPSOID = 'USE_ELLIPSOID'
    FIELD_FOR_ID = 'FIELD_FOR_ID'

    def initAlgorithm(self, config):

        # Here we define the inputs and output of the algorithm


        # Input point layer
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input layer'),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        # Chose field to store distance values
        self.addParameter(
            QgsProcessingParameterField(
            self.FIELD_FOR_DISTANCE,
            'Field for distance to the nearest point',
            None,
            self.INPUT,
            QgsProcessingParameterField.Numeric)
        )      

        # Chose field to ID of nearest point
        self.addParameter(
            QgsProcessingParameterField(
            self.FIELD_FOR_ID,
            'Field for ID fo the nearest point',
            None,
            self.INPUT,
            QgsProcessingParameterField.Numeric)
        )            

        # Chose to calculat distances on the ellipsoid or cartesian
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.USE_ELLIPSOID,
                self.tr('Calculate distances on elllisoid'),
                defaultValue = True
            )
        )
        
        # We add a feature sink in which to store our processed features 
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Nearst Neighbor')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):

        # Here is where the processing itself takes place.

        # Retrieve the feature source and sink.
        
        source = self.parameterAsSource(parameters, self.INPUT, context)
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, source.fields(), source.wkbType(), source.sourceCrs())
                
        if source is None:
            raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT))

        if QgsWkbTypes.isMultiType(source.wkbType()):
            raise QgsProcessingException(self.tr('Input layer is a MultiPoint layer - first convert to single points before using this algorithm.'))
                
                
        field_for_distance = self.parameterAsString(parameters, self.FIELD_FOR_DISTANCE, context)
        field_for_id = self.parameterAsString(parameters, self.FIELD_FOR_ID, context)
        use_ellipsoid = self.parameterAsString(parameters, self.USE_ELLIPSOID, context)
        
        
        # init distance measuring
        distance = QgsDistanceArea()
        distance.setSourceCrs(source.sourceCrs(), context.transformContext())
        distance.setEllipsoid(context.ellipsoid())

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / source.featureCount() if source.featureCount() else 0
        features = source.getFeatures()

        # create index
        feature_index = QgsSpatialIndex(source.getFeatures())
        
        
        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            # Get ID of nearest point from R-tree index
            nearestPointsIDs = feature_index.nearestNeighbor(feature.geometry().asPoint(), 2)
            # Get feature by ID 
            nearestPointID = nearestPointsIDs[1]
            points_iterator = source.getFeatures(QgsFeatureRequest().setFilterFid(nearestPointID))
            nearestPoint = next(points_iterator)

            # Calculate the distance to the nearest point
            if use_ellipsoid == 'true':
                a_distance = distance.measureLine(feature.geometry().asPoint(), nearestPoint.geometry().asPoint())
            else:
                a_distance = math.sqrt( (feature.geometry().asPoint().x() - nearestPoint.geometry().asPoint().x())**2 + (feature.geometry().asPoint().y() - nearestPoint.geometry().asPoint().y())**2)
            
            # Save distance and ID
            feature[field_for_distance] = a_distance 
            feature[field_for_id] = nearestPointID
            
            # Add a feature in the sink
            sink.addFeature(feature, QgsFeatureSink.FastInsert)

            # Update the progress bar
            feedback.setProgress(int(current * total))

        # Return the results of the algorithm.
        return {self.OUTPUT: dest_id}

    def name(self):
        """
        Returns the algorithm name
        """
        return 'Nearest Neighbor'

    def displayName(self):
        """
        Returns the translated algorithm name
        """
        return 'Nearest Neighbor' 

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''
        
    def icon(self):
        return QIcon(self.svgIconPath())


    def svgIconPath(self):
        return os.path.dirname(__file__) + '/icon/nearest_neighbor_icon.png'
        
    def shortHelpString(self):
        file = os.path.dirname(__file__) + '/help/nearest_neighbor.help'
        if not os.path.exists(file):
            return ''
        with open(file) as helpfile:
            help = helpfile.read()
        return help        

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return NearestNeighborAlgorithm()
