# -*- coding: utf-8 -*-
"""
Persistence centralisée — Langues & chemins
"""

from pathlib import Path
from qgis.PyQt.QtCore import QSettings
from typing import Iterable, Tuple

from .settings_keys import (
    SETTINGS_ORG,
    SETTINGS_APP,
    KEY_LANGS_SELECTED,
    KEY_LANGS_MULTI,
    KEY_LAST_PATH_PLUGIN,
    KEY_LAST_PATH_TS,
    KEY_LAST_PATH_HTML,
    KEY_LAST_PATH_HTML_DIR,
)


# ----------------------------------------------------------------------
#  Base
# ----------------------------------------------------------------------
def _settings() -> QSettings:
    return QSettings(SETTINGS_ORG, SETTINGS_APP)

# ----------------------------------------------------------------------
#  LANGUES
# ----------------------------------------------------------------------
def load_lang_state() -> tuple[list[str], bool]:
    """
    Retourne :
      (list[str], bool) -> (langs_selected, multi)
    """
    s = _settings()
    langs = s.value(KEY_LANGS_SELECTED, [], type=list)
    multi = s.value(KEY_LANGS_MULTI, False, type=bool)
    return list(langs) if langs else [], bool(multi)


def save_lang_state(langs_selected: Iterable[str], multi: bool = False, sync: bool = True):
    """
    Enregistre immédiatement l’état des langues.
    """
    s = _settings()
    s.setValue(KEY_LANGS_SELECTED, list(langs_selected))
    s.setValue(KEY_LANGS_MULTI, bool(multi))
    s.sync()
    if sync:
        s.sync()


# ----------------------------------------------------------------------
#  CHEMINS
# ----------------------------------------------------------------------
def load_last_path(kind: str) -> Path | None:
    key = {
        "plugin" : KEY_LAST_PATH_PLUGIN,
        "ts": KEY_LAST_PATH_TS,
        "html": KEY_LAST_PATH_HTML,
        "html_dir": KEY_LAST_PATH_HTML_DIR,
    }.get(kind)

    if not key:
        return None

    val = _settings().value(key, "")
    p = Path(val) if val else None
    return p if p and p.exists() else None


def save_last_path(kind: str, path: str | Path):
    key = {
        "plugin" : KEY_LAST_PATH_PLUGIN,
        "ts": KEY_LAST_PATH_TS,
        "html": KEY_LAST_PATH_HTML,
        "html_dir": KEY_LAST_PATH_HTML_DIR,
    }.get(kind)

    if not key:
        return

    p = Path(path)
    if not p.exists():
        return

    s = _settings()
    s.setValue(key, str(p))
    s.sync()
