# -*- coding: utf-8 -*-
"""
LangTree — composant commun pour sélection de langues par zones (QGIS / Qt)

- LANG_ZONES = source de vérité unique
- Affichage des langues dans leur nom NATIF (Deutsch, Español…)
- Fallback traduisible pour pylupdate / Plugin_translator
- Extraction TS automatique
- Auto-rebuild lors changement de langue QGIS
- Compatible Qt5 / Qt6
"""

from pathlib import Path
from qgis.PyQt.QtCore import Qt, QEvent, QLocale
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QTreeWidget, QTreeWidgetItem

# ----------------------------------------------
#  Emplacement des svg par langues et par zones
# ----------------------------------------------
_THIS_DIR = Path(__file__).resolve().parent
_ICONS_LANG_DIR = _THIS_DIR / "icons" / "lang"
_ICONS_ZONE_DIR = _THIS_DIR / "icons" / "zone"

# ----------------------------------------------------------------------
#  Définition des langues par zones
#  ⚠️ Clés internes NON traduites (SOURCE DE VÉRITÉ)
# ----------------------------------------------------------------------
LANG_ZONES = {
    "zone_europe_west": {
        "fr": "lang_fr",
        "en": "lang_en",
        "es": "lang_es",
        "pt": "lang_pt",
        "it": "lang_it",
        "de": "lang_de",
        "nl": "lang_nl",
        "sv": "lang_sv",
        "no": "lang_no",
        "da": "lang_da",
        "fi": "lang_fi",
    },
    "zone_europe_east": {
        "pl": "lang_pl",
        "cs": "lang_cs",
        "sk": "lang_sk",
        "sl": "lang_sl",
        "hr": "lang_hr",
        "sr": "lang_sr",
        "ro": "lang_ro",
        "hu": "lang_hu",
        "bg": "lang_bg",
        "el": "lang_el",
        "ru": "lang_ru",
        "uk": "lang_uk",
    },
    "zone_middle_east": {
        "he": "lang_he",
        "ar": "lang_ar",
        "tr": "lang_tr",
    },
    "zone_asia": {
        "zh": "lang_zh_cn",
        "zh-TW": "lang_zh_tw",
        "ja": "lang_ja",
        "ko": "lang_ko",
        "hi": "lang_hi",
    },
}

# ----------------------------------------------------------------------
#  Libellés des ZONES (traduisibles)
# ----------------------------------------------------------------------
ZONE_LABELS = {
    "zone_europe_west": "Europe de l’Ouest",
    "zone_europe_east": "Europe de l’Est",
    "zone_middle_east": "Moyen-Orient",
    "zone_asia": "Asie",
}

# ----------------------------------------------------------------------
#  Fallback traduisible (OBLIGATOIRE pour pylupdate)
# ----------------------------------------------------------------------
LANG_FALLBACK_LABELS = {
    "fr": "Français",
    "en": "Anglais",
    "es": "Espagnol",
    "pt": "Portugais",
    "it": "Italien",
    "de": "Allemand",
    "nl": "Néerlandais",
    "sv": "Suédois",
    "no": "Norvégien",
    "da": "Danois",
    "fi": "Finnois",
    "pl": "Polonais",
    "cs": "Tchèque",
    "sk": "Slovaque",
    "sl": "Slovène",
    "hr": "Croate",
    "sr": "Serbe",
    "ro": "Roumain",
    "hu": "Hongrois",
    "bg": "Bulgare",
    "el": "Grec",
    "ru": "Russe",
    "uk": "Ukrainien",
    "he": "Hébreu",
    "ar": "Arabe",
    "tr": "Turc",
    "zh": "Chinois",
    "ja": "Japonais",
    "ko": "Coréen",
    "hi": "Hindi",
}

# ----------------------------------------------------------------------
#  Fonctions utilitaires icones par langue et par zone
# ----------------------------------------------------------------------
def _icon_from_svg(path: Path) -> QIcon:
    if isinstance(path, Path) and path.exists():
        return QIcon(str(path))
    return QIcon()

def icon_for_language(code: str) -> QIcon:
    if not code:
        return _icon_from_svg(_ICONS_LANG_DIR / "default.svg")

    p = _ICONS_LANG_DIR / f"{code}.svg"
    if p.exists():
        return _icon_from_svg(p)

    short = code.split("-")[0]
    p2 = _ICONS_LANG_DIR / f"{short}.svg"
    if p2.exists():
        return _icon_from_svg(p2)

    return _icon_from_svg(_ICONS_LANG_DIR / "default.svg")

def icon_for_zone(zone_key: str) -> QIcon:
    if not zone_key:
        return _icon_from_svg(_ICONS_ZONE_DIR / "zone_default.svg")

    p = _ICONS_ZONE_DIR / f"{zone_key}.svg"
    if p.exists():
        return _icon_from_svg(p)

    return _icon_from_svg(_ICONS_ZONE_DIR / "zone_default.svg")

# ----------------------------------------------------------------------
#  🔖 Déclaration automatique des chaînes pour pylupdate / Plugin_translator
# ----------------------------------------------------------------------
def declare_langtree_strings(tr):
    """
    Déclare toutes les chaînes UI (zones + fallback langues)
    pour extraction TS.
    À appeler UNE SEULE FOIS dans le __init__ du dialogue.
    """
    for label in ZONE_LABELS.values():
        tr(label)

    for label in LANG_FALLBACK_LABELS.values():
        tr(label)

# ----------------------------------------------------------------------
#  Nom NATIF d'une langue via QLocale
# ----------------------------------------------------------------------
def native_language_name(code: str) -> str:
    """
    Renvoie le nom NATIF de la langue :
    - de  -> Deutsch
    - es  -> Español
    - pt  -> Português
    - ja  -> 日本語
    """
    if not code:
        return ""

    qt_code = code.replace("-", "_")
    locale = QLocale(qt_code)

    name = locale.nativeLanguageName()
    if name:
        return name[0].upper() + name[1:]

    return ""

# ----------------------------------------------------------------------
#  QTreeWidget auto-rebuild (LanguageChange)
# ----------------------------------------------------------------------
class LangTreeWidget(QTreeWidget):


    def __init__(self, parent=None, tr=lambda s: s):
        from qgis.PyQt.QtCore import QSize

        super().__init__(parent)
        self._saved_langs = []
        self._tr = tr
        self.setHeaderHidden(True)
        self.setIconSize(QSize(16, 16))

    def set_saved_langs(self, langs):
        self._saved_langs = list(langs)

    def rebuild(self):
        build_lang_tree(self, self._saved_langs, self._tr)

    def changeEvent(self, event):
        if event.type() == QEvent.LanguageChange:
            self.rebuild()
        super().changeEvent(event)

# ----------------------------------------------------------------------
#  Construction de l’arbre
# ----------------------------------------------------------------------
def build_lang_tree(tree, saved_langs, tr=lambda s: s):
    tree.clear()

    for zone_key, langs in LANG_ZONES.items():
        zone_item = QTreeWidgetItem([tr(ZONE_LABELS.get(zone_key, zone_key))])
        zone_item.setIcon(0, icon_for_zone(zone_key))
        zone_item.setData(0, Qt.UserRole, zone_key)
        zone_item.setFlags(zone_item.flags() | Qt.ItemIsUserCheckable)
        zone_item.setCheckState(0, Qt.Unchecked)
        tree.addTopLevelItem(zone_item)

        for code in langs.keys():
            native = native_language_name(code)
            fallback = tr(LANG_FALLBACK_LABELS.get(code, code))
            label = native or fallback

            child = QTreeWidgetItem([label])
            child.setData(0, Qt.UserRole, code)
            child.setIcon(0, icon_for_language(code))
            child.setFlags(child.flags() | Qt.ItemIsUserCheckable)
            child.setCheckState(
                0,
                Qt.Checked if code in saved_langs else Qt.Unchecked
            )
            zone_item.addChild(child)

        zone_item.setExpanded(False)

# ----------------------------------------------------------------------
#  Synchronisation Tree ← saved_langs
# ----------------------------------------------------------------------
def sync_tree_from_saved(tree, saved_langs):
    for i in range(tree.topLevelItemCount()):
        zone_item = tree.topLevelItem(i)

        checked = 0
        total = zone_item.childCount()

        for j in range(total):
            child = zone_item.child(j)
            code = child.data(0, Qt.UserRole)

            if code in saved_langs:
                child.setCheckState(0, Qt.Checked)
                checked += 1
            else:
                child.setCheckState(0, Qt.Unchecked)

        if checked == 0:
            zone_item.setCheckState(0, Qt.Unchecked)
        elif checked == total:
            zone_item.setCheckState(0, Qt.Checked)
        else:
            zone_item.setCheckState(0, Qt.PartiallyChecked)

# ----------------------------------------------------------------------
#  Synchronisation saved_langs ← Tree
# ----------------------------------------------------------------------
def collect_saved_langs(tree):
    langs = []

    for i in range(tree.topLevelItemCount()):
        zone = tree.topLevelItem(i)
        for j in range(zone.childCount()):
            child = zone.child(j)
            if child.checkState(0) == Qt.Checked:
                code = child.data(0, Qt.UserRole)
                if code:
                    langs.append(code)

    return sorted(set(langs))

# ----------------------------------------------------------------------
#  Gestion du clic hiérarchique
# ----------------------------------------------------------------------
def handle_tree_item_changed(tree, item, saved_langs):
    if item.childCount() > 0:
        state = item.checkState(0)
        for i in range(item.childCount()):
            item.child(i).setCheckState(0, state)
    else:
        parent = item.parent()
        if parent:
            states = [
                parent.child(i).checkState(0)
                for i in range(parent.childCount())
            ]
            if all(s == Qt.Checked for s in states):
                parent.setCheckState(0, Qt.Checked)
            elif all(s == Qt.Unchecked for s in states):
                parent.setCheckState(0, Qt.Unchecked)
            else:
                parent.setCheckState(0, Qt.PartiallyChecked)

    saved_langs[:] = collect_saved_langs(tree)

# ----------------------------------------------------------------------
#  Helpers UI
# ----------------------------------------------------------------------
def collapse_all(tree):
    for i in range(tree.topLevelItemCount()):
        tree.topLevelItem(i).setExpanded(False)

def expand_all(tree):
    for i in range(tree.topLevelItemCount()):
        tree.topLevelItem(i).setExpanded(True)
