# -*- coding: utf-8 -*-
"""
i18n_utils.py
----------------
Utilitaires partagés pour la gestion des traductions QGIS / Qt
(Qt5 + Qt6 compatibles)

À utiliser dans tous les plugins de l'organisation.
"""

from pathlib import Path
from typing import Optional, Tuple
from qgis.core import QgsApplication
from dataclasses import dataclass
import re
import xml.etree.ElementTree as ET
from collections import Counter
from typing import NamedTuple, Optional
from configparser import ConfigParser


# ----------------------------------------------------------------------
# Résultats
# ----------------------------------------------------------------------

@dataclass
class SourceLangResult:
    locale: str
    method: str          # "heuristic" | "ts" | "fallback"
    confidence: int      # 0–100


class SourceLangDetection(NamedTuple):
    locale: Optional[str]
    method: Optional[str]
    confidence: int


# ----------------------------------------------------------------------
# Règles heuristiques pondérées (FR / EN / DE / ES)
# ----------------------------------------------------------------------

LANG_PATTERNS = {
    # 🇫🇷 Français (très discriminant)
    "fr_FR": re.compile(
        r"\b("
        r"le|la|les|des|une|un|erreur|sélectionnez|génération|"
        r"fichier|langue|terminé|avertissement|échec|"
        r"création|mise à jour|chargement|supprimer"
        r")\b",
        re.I
    ),

    # 🇬🇧 Anglais (⚠️ volontairement plus faible)
    "en_GB": re.compile(
        r"\b("
        r"the|error|file|select|generation|failed|loading|update"
        r")\b",
        re.I
    ),

    # 🇩🇪 Allemand
    "de_DE": re.compile(
        r"\b("
        r"der|die|das|fehler|datei|sprache|auswahl|"
        r"erstellung|aktualisierung"
        r")\b",
        re.I
    ),

    # 🇪🇸 Espagnol
    "es_ES": re.compile(
        r"\b("
        r"el|la|los|error|archivo|seleccione|idioma|"
        r"generación|actualización"
        r")\b",
        re.I
    ),

    # 🇵🇹 Portugais
    "pt_PT": re.compile(
        r"\b("
        r"erro|arquivo|selecionar|idioma|geração|"
        r"atualização|falha"
        r")\b",
        re.I
    ),

    # 🇨🇳 Chinois simplifié
    "zh_CN": re.compile(r"[错误文件语言生成更新]", re.I),

    # 🇯🇵 Japonais
    "ja_JP": re.compile(r"[エラー言語生成更新選択]", re.I),
}


def detect_source_language(plugin_dir: Path) -> SourceLangResult:
    """
    Détecte automatiquement la langue source réelle d’un plugin QGIS.

    RÈGLE MÉTIER :
      - L’heuristique est TOUJOURS prioritaire
      - Les TS existants ne sont consultés QUE si l’heuristique est non concluante
    """

    # --------------------------------------------------
    # 1️⃣ Analyse heuristique (PRIORITÉ)
    # --------------------------------------------------
    samples = []

    for py in plugin_dir.rglob("*.py"):
        # 🔒 ignorer vendor
        if "vendor" in py.parts:
            continue

        try:
            text = py.read_text(encoding="utf-8", errors="ignore")
        except Exception:
            continue

        for m in re.finditer(r'\btr\(\s*"([^"]+)"', text):
            samples.append(m.group(1))
            if len(samples) >= 300:
                break

    if samples:
        scores = Counter()

        for s in samples:
            for lang, pattern in LANG_PATTERNS.items():
                if pattern.search(s):
                    scores[lang] += 1

        if scores:
            best_lang, best_hits = scores.most_common(1)[0]
            total_hits = sum(scores.values())

            # ✅ formule heuristique correcte
            confidence = int(100 * best_hits / max(1, total_hits))

            # 🔻 pénalisation douce anglais
            if best_lang == "en_GB" and confidence < 65:
                confidence = max(20, confidence - 20)

            # 🔒 seuil décisionnel heuristique
            if confidence >= 40:
                return SourceLangResult(
                    locale=best_lang,
                    method="heuristic",
                    confidence=confidence,
                )

    # --------------------------------------------------
    # 2️⃣ TS existants (SEULEMENT si heuristique non concluante)
    # --------------------------------------------------
    i18n_dir = plugin_dir / "i18n"
    if i18n_dir.exists():
        for ts in i18n_dir.glob("*.ts"):
            try:
                tree = ET.parse(ts)
                root = tree.getroot()

                # ❌ ignorer TS non traduits
                unfinished = root.findall(".//translation[@type='unfinished']")
                if unfinished:
                    continue

                src = root.attrib.get("sourcelanguage")
                if src and "_" in src:
                    return SourceLangResult(
                        locale=src,
                        method="ts",
                        confidence=60,
                    )
            except Exception:
                pass

    # --------------------------------------------------
    # 3️⃣ Fallback ultime
    # --------------------------------------------------
    return SourceLangResult(
        locale="en_GB",
        method="fallback",
        confidence=20,
    )

# ----------------------------------------------------------------------
# UI helpers
# ----------------------------------------------------------------------

def confidence_to_color(confidence: int) -> str:
    """
    Retourne une couleur CSS selon le score de confiance.
    """
    try:
        confidence = int(confidence)
    except Exception:
        confidence = 0

    if confidence >= 80:
        return "#2e7d32"   # vert
    elif confidence >= 50:
        return "#ed6c02"   # orange
    else:
        return "#c62828"   # rouge


# ----------------------------------------------------------------------
# Détection pylupdate (INCHANGÉE)
# ----------------------------------------------------------------------

def detect_pylupdate() -> Tuple[Optional[Path], Optional[Path]]:
    """
    Détection pylupdate robuste pour QGIS Windows (OSGeo4W / LTR / Qt5 / Qt6).
    """

    qgis_prefix = Path(QgsApplication.prefixPath()).resolve()
    apps_root = qgis_prefix.parent

    python_candidates = [
        apps_root / "Python312" / "python.exe",
        apps_root / "Python310" / "python.exe",
    ]

    python_exec = next((p for p in python_candidates if p.exists()), None)
    if python_exec is None:
        return None, None

    pylupdate_candidates = [
        apps_root / "Python312" / "Lib" / "site-packages" / "PyQt5" / "pylupdate_main.py",
        apps_root / "Python312" / "Lib" / "site-packages" / "PyQt6" / "pylupdate_main.py",
        apps_root / "Python312" / "Scripts" / "pylupdate6-script.py",
        apps_root / "Python312" / "Scripts" / "pylupdate5-script.py",
        apps_root / "Python312" / "Scripts" / "pylupdate6.exe",
        apps_root / "Python312" / "Scripts" / "pylupdate5.exe",
    ]

    for pylu in pylupdate_candidates:
        if pylu.exists():
            return python_exec, pylu

    return None, None

# ----------------------------------------------------------------------
# Prise en compte du metadata.txt dans la traduction
# ----------------------------------------------------------------------
def inject_metadata_into_ts(ts_path: Path, metadata_path: Path):
    """
    Injecte les champs metadata (name/description/about/tags, etc.)
    dans le TS pivot sous le contexte 'metadata'.

    - Compatible avec metadata.txt QGIS (INI).
    - Ignore les variantes [xx] si présentes (about[fr], etc.) -> on ne prend que la clé "neutre".
      (si tu veux aussi injecter about[fr], on peut l'ajouter, mais QGIS les lit directement sans Qt)
    """
    if not ts_path.exists() or not metadata_path.exists():
        return

    # --- lire metadata.txt (INI) ---
    cp = ConfigParser(interpolation=None)
    cp.optionxform = str  # conserve la casse des clés
    cp.read(metadata_path, encoding="utf-8")

    if not cp.has_section("general"):
        return

    # Champs utiles (tu peux étendre)
    keys = [
        "name",
        "description",
        "about",
        "tags",
        "author",
        "email",
        "homepage",
        "repository",
        "tracker",
    ]

    # récupérer les valeurs "neutres" uniquement (sans [xx])
    values = []
    for k in keys:
        if cp.has_option("general", k):
            v = (cp.get("general", k) or "").strip()
            if v:
                values.append(v)

    if not values:
        return

    # --- ouvrir TS ---
    tree = ET.parse(ts_path)
    root = tree.getroot()

    # trouver/créer contexte metadata
    ctx = None
    for c in root.findall("context"):
        name_el = c.find("name")
        if name_el is not None and (name_el.text or "") == "metadata":
            ctx = c
            break

    if ctx is None:
        ctx = ET.SubElement(root, "context")
        name_el = ET.SubElement(ctx, "name")
        name_el.text = "metadata"

    # index des sources déjà présentes (évite doublons)
    existing_sources = set()
    for m in ctx.findall("message"):
        src_el = m.find("source")
        if src_el is not None and src_el.text:
            existing_sources.add(src_el.text)

    # injecter messages
    for text in values:
        if text in existing_sources:
            continue

        msg = ET.SubElement(ctx, "message")
        src_el = ET.SubElement(msg, "source")
        src_el.text = text

        tr_el = ET.SubElement(msg, "translation")
        tr_el.set("type", "unfinished")
        tr_el.text = ""

    tree.write(ts_path, encoding="utf-8", xml_declaration=True)


# ----------------------------------------------------------------------
# Synchronisation du metadata.txt avec le .ts produit par la traduction
# ----------------------------------------------------------------------
METADATA_KEYS = {
    "name",
    "description",
    "about",
    "tags",
}

def sync_ts_to_metadata(
    ts_path: Path,
    metadata_path: Path,
    lang_code: str,
):
    """
    Synchronise les traductions du TS vers metadata.txt sous forme de clés [xx].

    Exemple :
      TS (fr) -> about[fr]=...
    """

    if not ts_path.exists() or not metadata_path.exists():
        return False

    # --------------------------------------------------
    # 1️⃣ Charger metadata.txt
    # --------------------------------------------------
    cfg = ConfigParser(interpolation=None)
    cfg.optionxform = str
    cfg.read(metadata_path, encoding="utf-8")

    if not cfg.has_section("general"):
        cfg.add_section("general")

    # --------------------------------------------------
    # 2️⃣ Charger TS
    # --------------------------------------------------
    tree = ET.parse(ts_path)
    root = tree.getroot()

    # trouver contexte metadata
    ctx = None
    for c in root.findall("context"):
        name = c.findtext("name")
        if name == "metadata":
            ctx = c
            break

    if ctx is None:
        return False

    # --------------------------------------------------
    # 3️⃣ Collecter traductions
    # --------------------------------------------------
    translations = {}

    for msg in ctx.findall("message"):
        src = msg.findtext("source")
        trn = msg.findtext("translation")

        if not src or not trn:
            continue

        translations[src.strip()] = trn.strip()

    if not translations:
        return False

    # --------------------------------------------------
    # 4️⃣ Appliquer aux clés metadata
    # --------------------------------------------------
    for key in METADATA_KEYS:
        if cfg.has_option("general", key):
            src_val = cfg.get("general", key).strip()
            if src_val in translations:
                cfg.set(
                    "general",
                    f"{key}[{lang_code}]",
                    translations[src_val]
                )

    # --------------------------------------------------
    # 5️⃣ Écriture metadata.txt
    # --------------------------------------------------
    with metadata_path.open("w", encoding="utf-8") as f:
        cfg.write(f)

    return True

