# -*- coding: utf-8 -*-
"""
ts_generator_core.py
---------------------
Cœur de génération universel des fichiers .ts pour un plugin QGIS.

Objectifs :
- Fonctionner avec n'importe quel plugin QGIS (structure libre).
- Détecter automatiquement les fichiers .py, .ui, .qrc.
- Créer automatiquement un fichier .pro dans un sous-dossier "_translation".
- Créer le dossier "i18n" s'il n'existe pas.
- Créer les .ts même s'il n'y a aucune chaîne traduisible.
- Toujours renvoyer des erreurs explicites (liste de messages) sans crasher.
"""

import os
import subprocess
from pathlib import Path
from typing import Callable, Iterable, List, Optional, Tuple

from qgis.core import QgsMessageLog


# ==========================================================================
# LOG QGIS
# ==========================================================================
def core_log(message: str, level: str = "INFO") -> None:
    tab = "ts_generator_core"
    lvl = {
        "ERROR": QgsMessageLog.CRITICAL,
        "WARNING": QgsMessageLog.WARNING,
    }.get(level, QgsMessageLog.INFO)
    QgsMessageLog.logMessage(str(message), tab, lvl)


# ==========================================================================
# FONCTIONS UTILITAIRES
# ==========================================================================
def _collect_sources(plugin_dir: Path) -> Tuple[list, list, list]:
    """
    Retourne (py_files, ui_files, qrc_files) sous forme de listes de chemins absolus (str).
    """
    plugin_dir = Path(plugin_dir)
    py_files = [str(p) for p in plugin_dir.rglob("*.py")]
    ui_files = [str(p) for p in plugin_dir.rglob("*.ui")]
    qrc_files = [str(p) for p in plugin_dir.rglob("*.qrc")]
    return py_files, ui_files, qrc_files


def _ensure_translation_layout(plugin_dir: Path, prefix: str, languages: Iterable[str]) -> Tuple[Path, Path, dict]:
    """
    - Crée le dossier "_translation" s'il n'existe pas.
    - Crée le dossier "i18n" à la racine du plugin s'il n'existe pas.
    - Crée un chemin de fichier .pro dans "_translation" (ex: _translation/<prefix>.pro).
    - Retourne (pro_path, i18n_dir, ts_paths_dict)

    ts_paths_dict: {lang: Path(ts_file)}
    """
    plugin_dir = Path(plugin_dir)
    trans_dir = plugin_dir / "_translation"
    trans_dir.mkdir(exist_ok=True)

    i18n_dir = plugin_dir / "i18n"
    i18n_dir.mkdir(exist_ok=True)

    pro_path = trans_dir / f"{prefix}.pro"

    ts_paths = {}
    for lang in languages:
        ts_paths[lang] = i18n_dir / f"{prefix}_{lang}.ts"

    return pro_path, i18n_dir, ts_paths


def _write_pro_file(pro_path: Path, plugin_dir: Path, py_files: list, ui_files: list, qrc_files: list, ts_paths: dict) -> None:
    """
    Écrit un .pro minimaliste dans pro_path, avec chemins relatifs à ce fichier.
    """
    plugin_dir = Path(plugin_dir)
    pro_dir = pro_path.parent

    def rel(p: str) -> str:
        return os.path.relpath(p, start=pro_dir)

    sources_rel = [rel(p) for p in py_files]
    forms_rel = [rel(p) for p in ui_files]
    qrc_rel = [rel(p) for p in qrc_files]
    ts_rel = [rel(str(p)) for p in ts_paths.values()]

    lines = []

    if sources_rel:
        lines.append("SOURCES = \\\n    " + " \\\n    ".join(sources_rel))
    if forms_rel:
        lines.append("FORMS = \\\n    " + " \\\n    ".join(forms_rel))
    if qrc_rel:
        lines.append("RESOURCES = \\\n    " + " \\\n    ".join(qrc_rel))

    if not ts_rel:
        ts_rel = []

    if ts_rel:
        lines.append("TRANSLATIONS = \\\n    " + " \\\n    ".join(ts_rel))

    if not lines:
        # Projet minimal, même si vide
        lines.append("SOURCES =")

    content = "\n\n".join(lines) + "\n"

    pro_path.write_text(content, encoding="utf-8")
    core_log(f"Fichier .pro généré : {pro_path}", "INFO")


# ==========================================================================
# FONCTION PRINCIPALE
# ==========================================================================
def generate_ts_for_plugin(
    plugin_dir,
    languages: Iterable[str],
    prefix: str,
    pylupdate_tuple: Tuple[Optional[Path], Optional[Path]],
    progress_callback: Optional[Callable[[int, int, str], None]] = None,
    process_events_callback: Optional[Callable[[], None]] = None,
) -> List[str]:
    """
    Génère les fichiers .ts d'un plugin QGIS.

    :param plugin_dir: Dossier du plugin (str ou Path).
    :param languages: iterable de codes de langue (ex: ["fr", "en"]). A / C + Google etc.
    :param prefix: préfixe des fichiers TS (ex: "valuerelation_fix").
    :param pylupdate_tuple: (pyexec, pylu) où :
        - pyexec = chemin vers python.exe ou None
        - pylu = script pylupdate5 (pylupdate5-script.py) ou exécutable pylupdate5.exe
    :param progress_callback: fonction (index:int, total:int, lang:str) ou None.
    :param process_events_callback: fonction () -> None appelée pour rafraîchir l'UI (ou None).
    :return: liste de chaînes décrivant les erreurs éventuelles.
    """
    errors: List[str] = []

    plugin_dir = Path(plugin_dir)
    if not plugin_dir.exists():
        msg = f"Dossier plugin introuvable: {plugin_dir}"
        core_log(msg, "ERROR")
        return [msg]

    # Normalisation / sécurisation des langues
    languages = [str(l).strip() for l in languages if str(l).strip()]
    if not languages:
        msg = "Aucune langue fournie à generate_ts_for_plugin."
        core_log(msg, "ERROR")
        return [msg]

    pyexec, pylu = pylupdate_tuple
    if pylu is None:
        msg = "pylupdate5 introuvable dans pylupdate_tuple."
        core_log(msg, "ERROR")
        errors.append(msg)
        return errors

    # Détecter les fichiers sources
    py_files, ui_files, qrc_files = _collect_sources(plugin_dir)
    core_log(f"Fichiers trouvés: py={len(py_files)}, ui={len(ui_files)}, qrc={len(qrc_files)}")

    # Préparer la structure de traduction (pro + i18n + chemins ts)
    pro_path, i18n_dir, ts_paths = _ensure_translation_layout(plugin_dir, prefix, languages)

    # Créer des .ts vides dès maintenant si souhaité
    for lang, ts_path in ts_paths.items():
        if not ts_path.exists():
            ts_path.touch()
            core_log(f"Fichier TS créé (vide) : {ts_path}")

    # Si aucun fichier source, on ne lance pas pylupdate,
    # mais on considère la génération comme "réussie", avec un message d'avertissement.
    if not py_files and not ui_files and not qrc_files:
        msg = f"Aucun fichier .py / .ui / .qrc trouvé dans {plugin_dir}. TS vides créés."
        core_log(msg, "WARNING")
        errors.append(msg)
        # on appelle tout de même le callback pour chaque langue
        total = len(languages)
        for idx, lang in enumerate(languages, start=1):
            if progress_callback:
                try:
                    progress_callback(idx, total, lang)
                except Exception as e:
                    core_log(f"Erreur dans progress_callback: {e}", "ERROR")
            if process_events_callback:
                try:
                    process_events_callback()
                except Exception as e:
                    core_log(f"Erreur dans process_events_callback: {e}", "ERROR")
        return errors

    # Écrire le .pro (SOURCES/FORMS/RESOURCES/TRANSLATIONS)
    try:
        _write_pro_file(pro_path, plugin_dir, py_files, ui_files, qrc_files, ts_paths)
    except Exception as e:
        msg = f"Erreur lors de l'écriture du .pro: {e}"
        core_log(msg, "ERROR")
        errors.append(msg)
        return errors

    total_langs = len(languages)
    core_log(f"Début génération TS : prefix={prefix}, langues={languages}")

    # Windows → éviter fenêtre console noire
    if os.name == "nt" and hasattr(subprocess, "CREATE_NO_WINDOW"):
        creation_flags = subprocess.CREATE_NO_WINDOW
    else:
        creation_flags = 0

    # ------------------------------------------------------------------
    # BOUCLE LANGUES : un appel pylupdate5 par langue
    # ------------------------------------------------------------------
    for index, lang in enumerate(languages, start=1):
        ts_path = ts_paths[lang]

        cmd: List[str]
        if pyexec:
            cmd = [str(pyexec), str(pylu), str(pro_path), "-ts", str(ts_path)]
        else:
            cmd = [str(pylu), str(pro_path), "-ts", str(ts_path)]

        core_log(f"Commande pylupdate5 ({lang}) : {' '.join(cmd)}")

        try:
            proc = subprocess.run(
                cmd,
                cwd=str(plugin_dir),
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                creationflags=creation_flags,
                text=True,
                encoding="utf-8",
                errors="replace",
            )
        except Exception as e:
            msg = f"Erreur lors de l'exécution de pylupdate5 pour {lang}: {e}"
            core_log(msg, "ERROR")
            errors.append(msg)
            continue

        if proc.returncode != 0:
            msg = f"pylupdate5 a retourné {proc.returncode} pour {lang}: {proc.stderr.strip()}"
            core_log(msg, "ERROR")
            errors.append(msg)
        else:
            core_log(f"pylupdate5 OK pour {lang}: {ts_path}", "INFO")

        # Callback de progression
        if progress_callback:
            try:
                progress_callback(index, total_langs, lang)
            except Exception as e:
                core_log(f"Erreur dans progress_callback: {e}", "ERROR")

        if process_events_callback:
            try:
                process_events_callback()
            except Exception as e:
                core_log(f"Erreur dans process_events_callback: {e}", "ERROR")

    core_log(f"Fin génération TS (prefix={prefix})")
    return errors
