# -*- coding: utf-8 -*-
"""
Génère automatiquement les fichiers .ts pour un plugin QGIS.
Interface PyQt5 avec :
- Choix du dossier plugin
- Choix du préfixe de nom de fichier TS
- Sélection multilingue (cases à cocher)
- Sauvegarde des choix (QSettings)
- Recherche automatique approfondie de pylupdate5
- Lancement automatique du traducteur
✅ Compatible console QGIS (sans __file__)
"""

from pathlib import Path
import importlib.util
import subprocess
import sys
import inspect

from qgis.PyQt.QtWidgets import (
    QApplication, QWidget, QVBoxLayout, QHBoxLayout, QPushButton,
    QFileDialog, QListWidget, QListWidgetItem, QLabel, QMessageBox, QLineEdit
)
from qgis.PyQt.QtCore import Qt, QSettings, QThread, pyqtSignal, QTimer, QPropertyAnimation


# === Dictionnaire des langues disponibles ===

# === Dictionnaire des langues disponibles ===
LANG_CODES = {
    "fr": "Français", "en": "Anglais", "de": "Allemand", "es": "Espagnol",
    "it": "Italien", "pt": "Portugais (Europe)", "br": "Portugais (Brésil)",
    "nl": "Néerlandais", "pl": "Polonais", "ru": "Russe", "ja": "Japonais",
    "zh": "Chinois", "dk": "Danois", "sv": "Suédois", "fi": "Finnois",
    "cs": "Tchèque", "sk": "Slovaque", "sl": "Slovène", "bg": "Bulgare",
    "ro": "Roumain", "el": "Grec", "he": "Hébreu", "tr": "Turc",
    "hu": "Hongrois", "et": "Estonien", "lt": "Lituanien", "lv": "Letton",
    "uk": "Ukrainien"
}

# === Détection automatique approfondie de pylupdate5 ===
def detect_pylupdate() -> Path:
    import shutil
    found = shutil.which("pylupdate5.exe")
    if found:
        return Path(found)

    exe = Path(sys.executable)
    search_roots = [
        exe.parent, exe.parent.parent, exe.parent.parent.parent,
        Path("C:/Program Files/QGIS 3.40.6"),
        Path("C:/Program Files/Qt"), Path("C:/OSGeo4W"),
        Path("C:/OSGeo4W64"), Path("C:/Program Files/JetBrains"),
        Path("C:/Program Files (x86)/JetBrains")
    ]
    for root in search_roots:
        if not root.exists():
            continue
        try:
            for path in root.rglob("pylupdate5.exe"):
                return path
        except PermissionError:
            continue

    return Path()


class TranslatorWorker(QThread):
    finished = pyqtSignal(str)
    error = pyqtSignal(str)

    def __init__(self, script_path, work_dir):
        super().__init__()
        self.script_path = script_path
        self.work_dir = work_dir

    def run(self):
        import runpy
        import traceback
        try:
            # Exécution du script dans le même interpréteur
            runpy.run_path(str(self.script_path), run_name="__main__")
            self.finished.emit(f"✅ Script terminé : {self.script_path.name}")
        except Exception as e:
            tb = traceback.format_exc()
            self.error.emit(f"Erreur pendant l'exécution du traducteur :\n{e}\n\n{tb}")


# === Interface principale ===
class TsGenerator(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("🧩 Générateur de fichiers .TS")
        self.setWindowFlags(Qt.WindowStaysOnTopHint)
        self.setFixedSize(480, 540)

        # Centrage de la fenêtre
        screen = QApplication.primaryScreen().availableGeometry()
        self.move(
            screen.center().x() - self.width() // 2,
            screen.center().y() - self.height() // 2
        )

        # Sauvegarde persistante utilisateur
        self.settings = QSettings("Plugin_Translator", "TS_Generator")

        layout = QVBoxLayout(self)
        layout.setContentsMargins(10, 10, 10, 10)
        layout.setSpacing(8)

        # === Sélection du dossier plugin ===
        row1 = QHBoxLayout()
        self.folder_label = QLabel(self.settings.value("last_folder", "Aucun dossier sélectionné"))
        self.folder_label.setStyleSheet("color:#444;")
        self.folder_btn = QPushButton("📂 Choisir dossier plugin")
        self.folder_btn.clicked.connect(self.select_folder)
        row1.addWidget(self.folder_label)
        row1.addWidget(self.folder_btn)
        layout.addLayout(row1)

        # === Préfixe du nom des fichiers TS ===
        layout.addWidget(QLabel("Préfixe du nom de fichier (.ts) :"))
        self.prefix_input = QLineEdit(self)
        default_prefix = self.settings.value("ts_prefix", "translate")  # valeur par défaut
        self.prefix_input.setPlaceholderText("Ex : translate")
        self.prefix_input.setText(default_prefix)
        layout.addWidget(self.prefix_input)

        # === Liste des langues ===
        layout.addWidget(QLabel("Langues à générer :"))
        self.lang_list = QListWidget()
        self.lang_list.setSelectionMode(QListWidget.NoSelection)
        layout.addWidget(self.lang_list)

        saved_langs = self.settings.value("selected_langs", [])
        if isinstance(saved_langs, str):
            saved_langs = saved_langs.split(",")

        for code, name in sorted(LANG_CODES.items()):
            item = QListWidgetItem(f"{name} ({code})")
            item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
            item.setCheckState(Qt.Checked if code in saved_langs else Qt.Unchecked)
            item.setData(Qt.UserRole, code)
            self.lang_list.addItem(item)

        # === Bouton de génération ===
        self.generate_btn = QPushButton("⚙ Générer les fichiers .ts")
        self.generate_btn.clicked.connect(self.generate_ts)
        layout.addWidget(self.generate_btn)

        # === Statut ===
        self.status_label = QLabel("Prêt.")
        layout.addWidget(self.status_label)

        # Initialisation
        self.plugin_dir = Path(self.settings.value("last_folder", "")) if self.settings.value("last_folder") else None
        self.pylupdate = detect_pylupdate()

        if not self.pylupdate or not self.pylupdate.exists():
            QMessageBox.warning(self, "Attention", "⚠ pylupdate5 introuvable.\nAjoutez Qt5/bin au PATH.")

    # --- Sélection du dossier ---
    def select_folder(self):
        folder = QFileDialog.getExistingDirectory(self, "Sélectionnez le dossier du plugin QGIS")
        if folder:
            self.plugin_dir = Path(folder)
            self.folder_label.setText(str(self.plugin_dir))
            self.settings.setValue("last_folder", str(self.plugin_dir))

    # --- Génération des fichiers TS ---
    def generate_ts(self):
        if not self.plugin_dir or not self.plugin_dir.exists():
            QMessageBox.warning(self, "Erreur", "Veuillez sélectionner un dossier valide.")
            return

        # Récupère les langues cochées
        languages = [
            self.lang_list.item(i).data(Qt.UserRole)
            for i in range(self.lang_list.count())
            if self.lang_list.item(i).checkState() == Qt.Checked
        ]
        if not languages:
            QMessageBox.warning(self, "Erreur", "Aucune langue sélectionnée.")
            return

        # Récupère et sauvegarde le préfixe
        prefix = self.prefix_input.text().strip() or "translate"
        self.settings.setValue("ts_prefix", prefix)
        self.settings.setValue("selected_langs", ",".join(languages))

        if not self.pylupdate or not self.pylupdate.exists():
            QMessageBox.critical(self, "Erreur", "❌ pylupdate5 introuvable.")
            return

        py_files = [str(p) for p in self.plugin_dir.rglob("*.py")]
        if not py_files:
            QMessageBox.critical(self, "Erreur", "Aucun fichier Python trouvé dans le dossier du plugin.")
            return

        i18n_dir = self.plugin_dir / "i18n"
        i18n_dir.mkdir(exist_ok=True)

        # Génération des fichiers .ts
        for lang in languages:
            ts_path = i18n_dir / f"{prefix}_{lang}.ts"
            cmd = [str(self.pylupdate)] + py_files + ["-ts", str(ts_path)]
            self.status_label.setText(f"Création/MàJ de {ts_path.name} ...")
            QApplication.processEvents()
            try:
                subprocess.run(cmd, check=True, shell=True)
                print(f"✅ {ts_path} mis à jour.")
            except subprocess.CalledProcessError as e:
                print(f"⚠️ Erreur {lang} : {e}")

        self.status_label.setText("🎉 Extraction terminée.")
        QMessageBox.information(self, "Terminé", "✅ Extraction des chaînes traduisibles terminée.")

        # Proposition : lancer le traducteur
        resp = QMessageBox.question(
            self,
            "Lancer la traduction ?",
            "Souhaitez-vous lancer le script de traduction maintenant ?",
            QMessageBox.Yes | QMessageBox.No
        )
        if resp == QMessageBox.Yes:
            self.launch_translator(i18n_dir)

    # --- Lancement du traducteur version DeepL---
    # def launch_translator(self, i18n_dir: Path):
    #     """Lance le traducteur dans un processus séparé, sans fermer la fenêtre."""
    #     try:
    #         translator_script = Path(__file__).parent / "translate_ts_gui_API_deepl.py"
    #     except NameError:
    #         translator_script = Path.cwd() / "translate_ts_gui_API_deepl.py"
    #
    #     if not translator_script.exists():
    #         QMessageBox.warning(self, "Introuvable", f"Script de traduction introuvable :\n{translator_script}")
    #         return
    #
    #     try:
    #         # Démarre un processus QGIS Python séparé pour la traduction
    #         subprocess.Popen([sys.executable, str(translator_script)], cwd=str(i18n_dir))
    #         self.status_label.setText("🈹 Traducteur lancé...")
    #     except Exception as e:
    #         QMessageBox.critical(self, "Erreur", f"Impossible de lancer le traducteur :\n{e}")

    # --- Lancement du traducteur version Google Translate---
    def launch_translator(self, i18n_dir: Path):
        """Exécute translate_ts_html_gui_API_google.py dans un thread Python local."""
        try:
            translator_script = Path(__file__).parent / "translate_ts_html_gui_API_google.py"
        except NameError:
            translator_script = Path.cwd() / "translate_ts_html_gui_API_google.py"

        if not translator_script.exists():
            QMessageBox.warning(self, "Introuvable", f"Script de traduction introuvable :\n{translator_script}")
            return

        # Lancer le thread
        self.status_label.setText("🈹 Lancement du traducteur Google...")
        self.thread = TranslatorWorker(translator_script, i18n_dir)

        def on_finished(msg):
            self.status_label.setText(msg)
            QMessageBox.information(self, "Terminé", msg)

        def on_error(err):
            self.status_label.setText("❌ Erreur pendant la traduction.")
            QMessageBox.critical(self, "Erreur", err)

        self.thread.finished.connect(on_finished)
        self.thread.error.connect(on_error)
        self.thread.start()
