# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Plugin_translator
                                 A QGIS plugin
 Générateur et traducteur automatique de fichiers .ts pour plugins QGIS
 -------------------
        begin                : 2025-10-29
        copyright            : (C) 2025
        author               : François THÉVAND
        email                : francois.thevand@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   Ce programme est un logiciel libre ; vous pouvez le redistribuer et   *
 *   le modifier selon les termes de la GNU GPL, publiée par la Free       *
 *   Software Foundation ; soit la version 2 de la licence, soit (à votre  *
 *   choix) toute version ultérieure.                                      *
 *                                                                         *
 ***************************************************************************/
"""
import re
import os
import tempfile
import stat
import sys
import subprocess
import platform
import shutil
from configparser import ConfigParser
import xml.etree.ElementTree as ET

from qgis.PyQt import QtWidgets, uic
from qgis.PyQt.QtCore import Qt, QFile
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QMessageBox, QApplication, QListWidgetItem, QTreeWidgetItem, QLabel, QPushButton, QHBoxLayout

from qgis.PyQt.QtGui import QIcon

from qgis.core import QgsMessageLog, Qgis

from .tools.lang_tree import (
    build_lang_tree,
    sync_tree_from_saved,
    handle_tree_item_changed,
    collapse_all,
)

from .tools.persistence import (
    load_lang_state,
    save_lang_state,
    load_last_path,
    save_last_path,
)

from .tools.i18n_utils import (
    detect_pylupdate,
    detect_source_language,
    confidence_to_color,
    inject_metadata_into_ts,
    sync_ts_to_metadata,
)

from .tools.settings_keys import SETTINGS_ORG, SETTINGS_APP
from .tools.settings_keys import KEY_LANGS_SELECTED

from . import resources   # force l’enregistrement des ressources


# --------------------------------------------------------------------------------------------
# Détection optionnelle des def tr() incorrects - Détecte : QCoreApplication.translate(X, ...)
# Affichage et arrêt du plugin en cas de détection
# Utilisation : décommenter dans le code l'appel à la fonction audit_tr_definitions()
# --------------------------------------------------------------------------------------------
TRANSLATE_CALL_RE = re.compile(
    r"QCoreApplication\.translate\(\s*([^,]+)\s*,"
)
# détecte une chaîne littérale "..." ou '...'
LITERAL_STRING_RE = re.compile(
    r"""^["'][^"']*["']$"""
)

def load_lang_state_shared():
    s = QSettings(SETTINGS_ORG, SETTINGS_APP)
    langs = s.value(KEY_LANGS_SELECTED, [], type=list)
    multi = s.value("langs_multi", False, type=bool)
    return langs, multi

def save_lang_state_shared(langs: list[str], multi: bool):
    s = QSettings(SETTINGS_ORG, SETTINGS_APP)
    s.setValue(KEY_LANGS_SELECTED, langs)
    s.setValue("langs_multi", multi)

def audit_tr_definitions(sources: list[str]) -> list[str]:
    """
    Vérifie uniquement les def tr() et détecte
    les contextes dynamiques incompatibles avec pylupdate.

    Retourne une liste de messages avec numéro de ligne.
    """
    issues: list[str] = []

    for src in sources:
        path = Path(src)

        try:
            lines = path.read_text(
                encoding="utf-8",
                errors="ignore"
            ).splitlines()
        except Exception:
            continue

        inside_tr = False
        tr_def_line = None

        for lineno, line in enumerate(lines, start=1):
            stripped = line.strip()

            # --- entrée dans def tr(...) ---
            if stripped.startswith("def tr("):
                inside_tr = True
                tr_def_line = lineno
                continue

            # --- sortie de def tr (nouvelle def ou class) ---
            if inside_tr and (
                stripped.startswith("def ")
                or stripped.startswith("class ")
            ):
                inside_tr = False
                tr_def_line = None
                continue

            if not inside_tr:
                continue

            # --- recherche de translate(...) ---
            m = TRANSLATE_CALL_RE.search(line)
            if m:
                ctx = m.group(1).strip()

                # contexte non littéral → ERREUR
                if not LITERAL_STRING_RE.match(ctx):
                    issues.append(
                        f"{path.name}:{lineno} : "
                        f"def tr() utilise un contexte dynamique → {ctx} "
                        f"(défini ligne {tr_def_line})"
                    )

                # on sort après le premier translate dans tr()
                inside_tr = False
                tr_def_line = None
    return issues

# ---------------------------------------------------------------------
# Code à placer dans le plugin pour activer la détection
# Penser à faire une importation de la fonction audit_tr_definitions()
# si elle n'est pas contenue dans le même module .py
# ----------------------------------------------------------------------
# --------------------------------------------------
# AUDIT pylupdate — def tr() uniquement
# --------------------------------------------------
# audit_issues = audit_tr_definitions(sources)
#
# if audit_issues:
#     QMessageBox.critical(
#         self,
#         self.tr("Audit pylupdate — def tr() invalide"),
#         self.tr(
#             "Des définitions def tr() utilisent un contexte dynamique "
#             "incompatible avec pylupdate :\n\n"
#         ) + "\n".join(audit_issues)
#     )
#     return
# --------------------------------------------------------------------------------------------
# Fin du code de détection optionnelle des def tr() incorrects
# --------------------------------------------------------------------------------------------

# -------------------------------------------------------------------------
# 🌍 Localisation robuste — langue réellement utilisée par QGIS
#    - charge le QM correspondant à la langue QGIS si disponible
#    - fallback anglais si présent
#    - sinon : fonctionnement en langue source du plugin
# -------------------------------------------------------------------------
from qgis.core import QgsApplication
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QTimer
from pathlib import Path

settings = QSettings()
plugin_dir = Path(__file__).resolve().parent
plugin_name = Path(__file__).resolve().parent.name
i18n_dir = plugin_dir / "i18n"

translator = QTranslator()
loaded = False

# ----------------------------------------------------------------------
#  TRADUCTION AVEC CONTEXTES — fonction unique
# ----------------------------------------------------------------------
TR_CONTEXT = "@default"

def tr(text: str) -> str:
    return QCoreApplication.translate("@default", text)

# ----------------------------------------------------------------------
# LOGGING CENTRALISÉ
# ----------------------------------------------------------------------
LOG_TAG = "PluginTranslator"

def qgis_log(msg: str, level: str = "INFO"):
    lvl = {
        "TRACE": Qgis.Info,
        "DEBUG": Qgis.Info,
        "INFO": Qgis.Info,
        "WARNING": Qgis.Warning,
        "ERROR": Qgis.Critical,
    }.get(level.upper(), Qgis.Info)

    QgsMessageLog.logMessage(msg, LOG_TAG, lvl)
# -------------------------------------------------------------------------
# 1️⃣ Langue réellement utilisée par QGIS
# -------------------------------------------------------------------------
locale_full = QgsApplication.locale() or settings.value("locale/userLocale", "")
lang = locale_full.split("_")[0].lower() if locale_full else ""

qgis_log(f"[i18n] Locale QGIS détectée : {locale_full}", "DEBUG")
qgis_log(f"[i18n] Langue QGIS détectée : {lang or 'indéterminée'}", "DEBUG")

# -------------------------------------------------------------------------
# 2️⃣ Chargement QM
# -------------------------------------------------------------------------
def load_qm(code: str) -> bool:
    qm = i18n_dir / f"{plugin_name}_{code}.qm"
    if qm.exists() and translator.load(str(qm)):
        qgis_log(f"[i18n] QM chargé : {qm.name}", "DEBUG")
        return True
    return False

# -------------------------------------------------------------------------
# 3️⃣ Logique de sélection (sans hypothèse sur la langue source)
# -------------------------------------------------------------------------
if lang and load_qm(lang):
    loaded = True
    qgis_log(f"[i18n] Traduction activée pour la langue QGIS : {lang}", "INFO")
elif load_qm("en"):
    loaded = True
    qgis_log(
        "[i18n] Traduction de la langue QGIS absente → fallback anglais",
        "WARNING"
    )
else:
    qgis_log(
        "[i18n] Aucun QM compatible trouvé → fonctionnement du plugin dans sa langue source",
        "INFO"
    )
# -------------------------------------------------------------------------
# 4️⃣ Installation du translator
# -------------------------------------------------------------------------
if loaded:
    QCoreApplication.installTranslator(translator)

# ----------------------------------------------------------------------
#  Extraction d’outils depuis les ressources
# ----------------------------------------------------------------------
def extract_resource_to_temp(path_in_qrc: str) -> Path:
    """
    Extrait un fichier des ressources vers un fichier temporaire exécutable.
    Utilisé pour distribuer lrelease/lupdate avec le plugin.
    """
    f = QFile(path_in_qrc)
    if not f.exists():
        return None
    if not f.open(QFile.ReadOnly):
        return None

    data = f.readAll()
    f.close()

    tmp = Path(tempfile.gettempdir()) / ("plugin_translator_" + Path(path_in_qrc).name)

    with tmp.open("wb") as out:
        out.write(bytes(data))

    try:
        tmp.chmod(tmp.stat().st_mode | stat.S_IEXEC)
    except Exception:
        pass

    return tmp

def detect_lrelease():
    """Charge lrelease.exe depuis les ressources du plugin si présent."""
    return extract_resource_to_temp(":/tools/lrelease.exe") or None

def detect_lupdate():
    """Prévu si tu veux ajouter lupdate plus tard (actuellement non utilisé)."""
    return extract_resource_to_temp(":/tools/lupdate.exe") or None

# ------------------------------------------------------------
# Détection de la langue du pays d'utilisation
# ------------------------------------------------------------
def get_qgis_locale() -> str:
    """
    Retourne la locale QGIS courante sous forme Qt (ex: fr_FR, en_GB).
    Fallback sûr sur en_GB.
    """
    try:
        from qgis.core import QgsApplication
        loc = QgsApplication.locale()  # ex: 'fr_FR'
        if isinstance(loc, str) and "_" in loc:
            return loc
    except Exception:
        pass
    return "en_GB"

# ------------------------------------------------------------
# Chargement UI
# ------------------------------------------------------------
FORM_CLASS, _ = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "plugin_translator_dialog_base.ui")
)

# ------------------------------------------------------------
# Mapping codes → locales Qt
# ------------------------------------------------------------
LANG_CODE_TO_QT = {
    "fr": "fr_FR",
    "en": "en_GB",
    "es": "es_ES",
    "pt": "pt_PT",
    "it": "it_IT",
    "de": "de_DE",
    "nl": "nl_NL",
    "sv": "sv_SE",
    "no": "nb_NO",
    "da": "da_DK",
    "fi": "fi_FI",
    "pl": "pl_PL",
    "cs": "cs_CZ",
    "sk": "sk_SK",
    "sl": "sl_SI",
    "hr": "hr_HR",
    "sr": "sr_RS",
    "ro": "ro_RO",
    "hu": "hu_HU",
    "bg": "bg_BG",
    "el": "el_GR",
    "ru": "ru_RU",
    "uk": "uk_UA",
    "he": "he_IL",
    "ar": "ar",
    "tr": "tr_TR",
    "zh": "zh_CN",
    "zh-TW": "zh_TW",
    "ja": "ja_JP",
    "ko": "ko_KR",
    "hi": "hi_IN",
}

SOURCE_LANG_CHOICES = {
    "fr_FR": "Français",
    "en_GB": "Anglais",
    "es_ES": "Espagnol",
    "pt_PT": "Portugais",
    "it_IT": "Italien",
    "de_DE": "Allemand",
    "nl_NL": "Néerlandais",
    "sv_SE": "Suédois",
    "nb_NO": "Norvégien",
    "da_DK": "Danois",
    "fi_FI": "Finnois",
    "pl_PL": "Polonais",
    "cs_CZ": "Tchèque",
    "sk_SK": "Slovaque",
    "sl_SI": "Slovène",
    "hr_HR": "Croate",
    "sr_RS": "Serbe",
    "ro_RO": "Roumain",
    "hu_HU": "Hongrois",
    "bg_BG": "Bulgare",
    "el_GR": "Grec",
    "ru_RU": "Russe",
    "uk_UA": "Ukrainien",
    "he_IL": "Hébreu",
    "ar": "Arabe",
    "tr_TR": "Turc",
    "zh_CN": "Chinois",
    "zh_TW": "Chinois traditionnel",
    "ja_JP": "Japonais",
    "ko_KR": "Coréen",
    "hi_IN": "Hindi",
}

# ------------------------------------------------------------
# FONCTIONS UTILITAIRES TS
# ------------------------------------------------------------
# ----------------------------------------------------------------------
#  TRADUCTION AVEC CONTEXTES — fonction unique
# ----------------------------------------------------------------------

def set_ts_header(
    ts_path: Path,
    language: str | None,
    sourcelanguage: str | None,
):
    """
    Met à jour l'entête <TS> :
      - version = 2.1
      - language (si TS cible)
      - sourcelanguage (si défini)
    """
    if not isinstance(ts_path, Path) or not ts_path.exists():
        return

    qgis_log(
        tr("Mise à jour entête TS : {0} (lang={1}, source={2})")
        .format(ts_path.name, language, sourcelanguage),
        "TRACE"
    )

    try:
        tree = ET.parse(ts_path)
        root = tree.getroot()

        # --------------------------------------------------
        # Validation stricte du format TS
        # --------------------------------------------------
        if root.tag != "TS":
            qgis_log(
                tr("Racine TS inattendue dans {0}")
                .format(ts_path),
                "WARNING"
            )
            return

        # --------------------------------------------------
        # Version TS
        # --------------------------------------------------
        root.set("version", "2.1")

        # --------------------------------------------------
        # language (TS cible uniquement)
        # --------------------------------------------------
        if isinstance(language, str) and "_" in language:
            root.set("language", language)
        else:
            # pivot TS → pas d'attribut language
            root.attrib.pop("language", None)

        # --------------------------------------------------
        # sourcelanguage
        # --------------------------------------------------
        if isinstance(sourcelanguage, str) and "_" in sourcelanguage:
            root.set("sourcelanguage", sourcelanguage)
        else:
            root.attrib.pop("sourcelanguage", None)

        # --------------------------------------------------
        # Écriture sûre (UTF-8 + déclaration XML)
        # --------------------------------------------------
        tree.write(
            ts_path,
            encoding="utf-8",
            xml_declaration=True
        )

        qgis_log(
            tr("Entête TS mis à jour : {0}")
            .format(ts_path.name),
            "DEBUG"
        )

    except ET.ParseError as e:
        qgis_log(
            tr("Erreur XML dans {0} : {1}")
            .format(ts_path.name, e),
            "ERROR"
        )

    except Exception as e:
        qgis_log(
            tr("Erreur lors de la mise à jour de l'entête TS : {0}")
            .format(e),
            "ERROR"
        )


# ------------------------------------------------------------
# DIALOGUE PRINCIPAL
# ------------------------------------------------------------
class TsGeneratorDialog(QtWidgets.QDialog, FORM_CLASS):

    """
    ATTENTION : pour une traduction correcte de la boite de dialogue fournie en ui
    il faut donner en contexte le nom de la balise class contenue dans l'ui
    (Ici, c'est TsGeneratorDialogBase, ligne dans l'ui : <class>TsGeneratorDialogBase</class>
    """
    # @staticmethod
    # def tr(text: str) -> str:
    #     return QCoreApplication.translate("TsGeneratorDialogBase", text)

    def __init__(self, parent=None, parent_plugin=None):
        super().__init__(parent)

        self.setupUi(self)
        self.retranslateUi(self)

        self.parent_plugin = parent_plugin
        self.plugin_dir = None

        # ✅ Source de vérité
        self.saved_langs = []

        # ✅ Gardes anti-boucle (resync / itemChanged)
        self._syncing_lang_tree = False

        # Titre
        self.setWindowTitle(tr("Générateur de fichier pivot .TS"))

        # Connexions (boutons)
        self.btnBrowseFolder.clicked.connect(self.select_folder)
        self.btnGenerateTS.clicked.connect(self.generate_ts)

        # ✅ Connexion arbre (lang_tree existe via le .ui)
        self.lang_tree.itemChanged.connect(self._on_lang_tree_changed)

        # ✅ Construire l’arbre (avec saved_langs vide pour l’instant)
        # (les noms affichés passent par self.tr, comme avant)
        build_lang_tree(self.lang_tree, self.saved_langs, self.tr)

        # Langue source
        self.init_source_language_combo()

        self._init_source_language_helpers()

        # ✅ Charger settings (remplit saved_langs) + resync arbre dans load_settings()
        self.load_settings()

        # ✅ Par défaut : tous les groupes repliés
        collapse_all(self.lang_tree)

    def _init_source_language_helpers(self):
        """
        Ajoute un label discret + bouton Re-détecter sous la combo langue source.
        """
        if not hasattr(self, "comboSourceLanguage"):
            return

        parent_layout = self.comboSourceLanguage.parentWidget().layout()
        if parent_layout is None:
            return

        # --- conteneur horizontal ---
        hlayout = QHBoxLayout()
        hlayout.setContentsMargins(0, 0, 0, 0)
        hlayout.setSpacing(6)

        # --- label discret ---
        self.lblAutoDetected = QLabel(
            tr("Langue source détectée automatiquement")
        )
        self.lblAutoDetected.setVisible(False)
        self.lblAutoDetected.setStyleSheet(
            "color: #777; font-size: 10px;"
        )
        self.lblAutoDetected.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)

        # --- bouton re-détecter ---
        self.btnRedetectLang = QPushButton("🔁")
        self.btnRedetectLang.setToolTip(tr("Re-détecter la langue source du plugin"))
        self.btnRedetectLang.setFixedSize(22, 22)
        self.btnRedetectLang.setVisible(False)

        self.btnRedetectLang.clicked.connect(self._redetect_source_language)

        hlayout.addWidget(self.lblAutoDetected)
        hlayout.addStretch()
        hlayout.addWidget(self.btnRedetectLang)

        parent_layout.addLayout(hlayout)

    def _on_lang_tree_changed(self, item, column):
        if column != 0 or self._syncing_lang_tree:
            return

        self._syncing_lang_tree = True
        try:
            handle_tree_item_changed(
                self.lang_tree,
                item,
                self.saved_langs
            )
            save_lang_state(self.saved_langs, multi=False)
        finally:
            self._syncing_lang_tree = False

    def _save_lang_settings(self):
        """Persistance centralisée des langues."""
        save_lang_state(self.saved_langs, multi=False)

    # --------------------------------------------------------
    def init_source_language_combo(self):
        """
        Initialise la comboBox de langue source :
        - toutes les langues sont proposées
        - tentative de détection automatique si plugin_dir connu
        - fallback sur la langue QGIS
        - fallback final sur en_GB
        """
        if not hasattr(self, "comboSourceLanguage"):
            return

        self.comboSourceLanguage.clear()

        # --------------------------------------------------
        # 1️⃣ Remplissage de la combo
        # --------------------------------------------------
        for code, label in SOURCE_LANG_CHOICES.items():
            self.comboSourceLanguage.addItem(
                f"{tr(label)} ({code})",
                code
            )

        # --------------------------------------------------
        # 2️⃣ État UI par défaut (important)
        # --------------------------------------------------
        if hasattr(self, "lblAutoDetected"):
            self.lblAutoDetected.setVisible(False)

        if hasattr(self, "btnRedetectLang"):
            self.btnRedetectLang.setVisible(False)

        detected_lang = None
        result = None

        # --------------------------------------------------
        # 3️⃣ Tentative de détection automatique
        # --------------------------------------------------
        if self.plugin_dir and self.plugin_dir.exists():
            try:
                result = detect_source_language(self.plugin_dir)
                detected_lang = result.locale

                qgis_log(
                    f"[i18n] Langue source détectée automatiquement : {detected_lang}",
                    "INFO"
                )
            except Exception as e:
                qgis_log(
                    f"[i18n] Échec détection langue source plugin : {e}",
                    "WARNING"
                )

        # --------------------------------------------------
        # 4️⃣ Sélection depuis détection
        # --------------------------------------------------
        idx = -1

        if detected_lang:
            idx = self.comboSourceLanguage.findData(detected_lang)

            if idx < 0:
                short = detected_lang.split("_")[0]
                for i in range(self.comboSourceLanguage.count()):
                    data = self.comboSourceLanguage.itemData(i)
                    if isinstance(data, str) and data.startswith(short):
                        idx = i
                        break

            if idx >= 0:
                self.comboSourceLanguage.setCurrentIndex(idx)

                # ➕ feedback UI
                if result and hasattr(self, "lblAutoDetected"):
                    method_txt = {
                        "ts": tr("détecté depuis TS existant"),
                        "heuristic": tr("analyse heuristique"),
                        "fallback": tr("valeur par défaut"),
                    }.get(result.method, "")

                    color = confidence_to_color(result.confidence)

                    self.lblAutoDetected.setText(
                        tr("Langue source détectée automatiquement")
                        + f" — {method_txt} ({result.confidence} %)"
                    )
                    self.lblAutoDetected.setStyleSheet(
                        f"color: {color}; font-size: 10pt;"
                    )
                    self.lblAutoDetected.setVisible(True)

                if hasattr(self, "btnRedetectLang"):
                    self.btnRedetectLang.setVisible(True)

                return  # ✅ FIN NORMALE

        # --------------------------------------------------
        # 5️⃣ Fallback : langue QGIS
        # --------------------------------------------------
        qgis_locale = get_qgis_locale()
        idx = self.comboSourceLanguage.findData(qgis_locale)

        if idx < 0:
            short = qgis_locale.split("_")[0]
            for i in range(self.comboSourceLanguage.count()):
                data = self.comboSourceLanguage.itemData(i)
                if isinstance(data, str) and data.startswith(short):
                    idx = i
                    break

        # --------------------------------------------------
        # 6️⃣ Fallback ultime
        # --------------------------------------------------
        if idx < 0:
            idx = self.comboSourceLanguage.findData("en_GB")

        self.comboSourceLanguage.setCurrentIndex(max(idx, 0))

    def _detect_source_language_ui(self):
        """
        Détecte la langue source du plugin et met à jour l'UI.
        """
        if not self.plugin_dir or not self.plugin_dir.exists():
            return

        try:
            result = detect_source_language(self.plugin_dir)
            detected_lang = result.locale

            if not detected_lang:
                return

            # Sélection combo
            idx = self.comboSourceLanguage.findData(detected_lang)
            if idx < 0:
                short = detected_lang.split("_")[0]
                for i in range(self.comboSourceLanguage.count()):
                    data = self.comboSourceLanguage.itemData(i)
                    if data.startswith(short):
                        idx = i
                        break

            if idx >= 0:
                self.comboSourceLanguage.setCurrentIndex(idx)

            # Label
            if hasattr(self, "lblAutoDetected"):
                method_txt = {
                    "ts": tr("détecté depuis TS existant"),
                    "heuristic": tr("analyse heuristique"),
                    "fallback": tr("valeur par défaut"),
                }.get(result.method, "")

                dot = (
                    "🟢" if result.confidence >= 80
                    else "🟠" if result.confidence >= 50
                    else "🔴"
                )

                self.lblAutoDetected.setText(
                    f"{dot} "
                    + tr("Langue source détectée automatiquement")
                    + f" — {method_txt} ({result.confidence} %)"
                )
                self.lblAutoDetected.setVisible(True)

            if hasattr(self, "btnRedetectLang"):
                self.btnRedetectLang.setVisible(True)

            qgis_log(
                f"[i18n] Langue source détectée : {detected_lang} "
                f"({result.method}, {result.confidence}%)",
                "INFO"
            )

        except Exception as e:
            qgis_log(
                f"[i18n] Échec détection langue source : {e}",
                "WARNING"
            )

    def _redetect_source_language(self):
        """
        Force une nouvelle détection de la langue source du plugin.
        """
        if not self.plugin_dir or not self.plugin_dir.exists():
            return

        try:
            result = detect_source_language(self.plugin_dir)
            detected_lang = result.locale

            if not detected_lang:
                return

            # --------------------------------------------------
            # Sélection dans la combo
            # --------------------------------------------------
            idx = self.comboSourceLanguage.findData(detected_lang)
            if idx < 0:
                short = detected_lang.split("_")[0]
                for i in range(self.comboSourceLanguage.count()):
                    data = self.comboSourceLanguage.itemData(i)
                    if isinstance(data, str) and data.startswith(short):
                        idx = i
                        break

            if idx >= 0:
                self.comboSourceLanguage.setCurrentIndex(idx)

            # --------------------------------------------------
            # 🧠 Feedback UI : méthode + confiance + couleur
            # --------------------------------------------------
            if hasattr(self, "lblAutoDetected"):
                method_txt = {
                    "ts": tr("détecté depuis TS existant"),
                    "heuristic": tr("analyse heuristique"),
                    "fallback": tr("valeur par défaut"),
                }.get(result.method, "")

                dot = (
                    "🟢" if result.confidence >= 80
                    else "🟠" if result.confidence >= 50
                    else "🔴"
                )

                self.lblAutoDetected.setText(
                    f"{dot} "
                    + tr("Langue source détectée automatiquement")
                    + f" — {method_txt} ({result.confidence} %)"
                )

                self.lblAutoDetected.setVisible(True)

            if hasattr(self, "btnRedetectLang"):
                self.btnRedetectLang.setVisible(True)

            qgis_log(
                f"[i18n] Re-détection langue source → {detected_lang} "
                f"({result.method}, {result.confidence}%)",
                "INFO"
            )

        except Exception as e:
            qgis_log(
                f"[i18n] Échec re-détection langue source : {e}",
                "WARNING"
            )

    # --------------------------------------------------------
    def load_settings(self):
        """Charge les paramètres persistants et resynchronise l’UI."""

        # --------------------------------------------------
        # 1️⃣ Langues cibles
        # --------------------------------------------------
        self.saved_langs, _ = load_lang_state_shared()

        self._syncing_lang_tree = True
        try:
            sync_tree_from_saved(self.lang_tree, self.saved_langs)
        finally:
            self._syncing_lang_tree = False

        # --------------------------------------------------
        # 2️⃣ Dossier plugin
        # --------------------------------------------------
        last_folder = load_last_path("plugin")

        if last_folder and last_folder.exists():
            self.plugin_dir = last_folder
        else:
            self.plugin_dir = Path(__file__).resolve().parent

        self.editFolder.setText(str(self.plugin_dir))
        self.editPrefix.setText(self.plugin_dir.name)

        # --------------------------------------------------
        # 3️⃣ Langue source — priorité intelligente
        # --------------------------------------------------
        if not hasattr(self, "comboSourceLanguage"):
            return

        # 🔹 tentative 1 : détection automatique
        detected_lang = None
        result = None

        try:
            result = detect_source_language(self.plugin_dir)
            detected_lang = result.locale
        except Exception:
            pass

        idx = -1

        if detected_lang:
            idx = self.comboSourceLanguage.findData(detected_lang)

            if idx < 0:
                short = detected_lang.split("_")[0]
                for i in range(self.comboSourceLanguage.count()):
                    data = self.comboSourceLanguage.itemData(i)
                    if isinstance(data, str) and data.startswith(short):
                        idx = i
                        break

            if idx >= 0:
                self.comboSourceLanguage.setCurrentIndex(idx)

                # ➕ feedback UI
                if result and hasattr(self, "lblAutoDetected"):
                    method_txt = {
                        "ts": tr("détecté depuis TS existant"),
                        "heuristic": tr("analyse heuristique"),
                        "fallback": tr("valeur par défaut"),
                    }.get(result.method, "")

                    color = confidence_to_color(result.confidence)

                    self.lblAutoDetected.setText(
                        tr("Langue source détectée automatiquement")
                        + f" — {method_txt} ({result.confidence} %)"
                    )
                    self.lblAutoDetected.setStyleSheet(
                        f"color: {color}; font-size: 10pt;"
                    )
                    self.lblAutoDetected.setVisible(True)

                if hasattr(self, "btnRedetectLang"):
                    self.btnRedetectLang.setVisible(True)

                return  # ✅ ne PAS écraser avec QGIS locale

        # --------------------------------------------------
        # 4️⃣ Fallback : langue QGIS
        # --------------------------------------------------
        qgis_locale = get_qgis_locale()
        idx = self.comboSourceLanguage.findData(qgis_locale)

        if idx < 0:
            short = qgis_locale.split("_")[0]
            for i in range(self.comboSourceLanguage.count()):
                data = self.comboSourceLanguage.itemData(i)
                if isinstance(data, str) and data.startswith(short):
                    idx = i
                    break

        # --------------------------------------------------
        # 5️⃣ Fallback ultime
        # --------------------------------------------------
        if idx < 0:
            idx = self.comboSourceLanguage.findData("en_GB")

        self.comboSourceLanguage.setCurrentIndex(max(idx, 0))

        # UI propre
        if hasattr(self, "lblAutoDetected"):
            self.lblAutoDetected.setVisible(False)
        if hasattr(self, "btnRedetectLang"):
            self.btnRedetectLang.setVisible(False)

    def _sync_lang_tree_from_saved(self):
        """
        Re-synchronise complètement l’arbre des langues à partir de self.saved_langs.
        self.saved_langs est la SOURCE DE VÉRITÉ.
        """
        if not hasattr(self, "lang_tree"):
            return

        self._syncing_lang_tree = True
        try:
            sync_tree_from_saved(self.lang_tree, self.saved_langs)
        finally:
            self._syncing_lang_tree = False

    from qgis.PyQt.QtCore import QTimer

    def select_folder(self):
        """Ouvre le sélecteur sur le dossier plugins du profil QGIS courant."""

        try:
            from qgis.core import QgsApplication
            profile_path = Path(QgsApplication.qgisSettingsDirPath())
            start_dir = profile_path / "python" / "plugins"
        except Exception:
            start_dir = Path.home()

        if not start_dir.exists():
            start_dir = Path.home()

        folder = QFileDialog.getExistingDirectory(
            self,
            tr("Choisissez un dossier de plugin QGIS"),
            str(start_dir),
            QFileDialog.ShowDirsOnly
        )

        if not folder:
            return

        folder = Path(folder)

        # UI immédiate
        self.plugin_dir = folder
        self.editFolder.setText(str(folder))
        self.editPrefix.setText(folder.name)

        self._detect_source_language_ui()

        save_last_path("plugin", folder)
        i18n_dir = folder / "i18n"
        if i18n_dir.exists():
            save_last_path("ts", i18n_dir)

    def _post_select_plugin_folder(self, folder: Path):
        """
        Traitements lourds après sélection du dossier plugin.
        Appelé après rafraîchissement de l'UI.
        """
        if not folder.exists():
            return

        # Exemple : auto-détection langues TS si nécessaire
        # (à adapter si présent dans TsGeneratorDialog)
        # self._auto_detect_langs_in_ts_folder(folder / "i18n")

        # Si tu synchronises un arbre de langues :
        self._syncing_lang_tree = True
        try:
            sync_tree_from_saved(self.lang_tree, self.saved_langs)
        finally:
            self._syncing_lang_tree = False

    # --------------------------------------------------------
    def get_source_language(self) -> str:
        """
        Renvoie la langue source sélectionnée (locale Qt, ex. 'fr_FR' ou 'en_GB').
        Si la combo n'existe pas ou est vide : 'en_GB' par défaut.
        """
        if hasattr(self, "comboSourceLanguage") and self.comboSourceLanguage.count() > 0:
            code = self.comboSourceLanguage.currentData()
            if isinstance(code, str) and code:
                return code
        return "en_GB"

    def collect_pylupdate_sources(self, plugin_dir: Path) -> list[str]:
        """
        Collecte strictement les fichiers source pour pylupdate5 :
        - .py
        - .ui
        - uniquement dans le plugin cible
        - exclusion des dossiers techniques
        """
        EXCLUDED_DIRS = {
            "__pycache__",
            "i18n",
            ".git",
            ".svn",
            ".hg",
            ".idea",
            ".vscode",
            ".pytest_cache",
            ".mypy_cache",
            "venv",
            ".venv",
            "env",
        }

        sources = []

        for root, dirs, files in os.walk(plugin_dir):
            dirs[:] = [d for d in dirs if d not in EXCLUDED_DIRS and not d.startswith(".")]

            for name in files:
                if name.endswith((".py", ".ui")):
                    sources.append(str(Path(root) / name))

        # ✅ metadata.txt uniquement à la racine du plugin
        metadata_txt = plugin_dir / "metadata.txt"
        if metadata_txt.exists():
            sources.append(str(metadata_txt))

        return sources

    def chunked(self, iterable, size):
        """Découpe une liste en blocs de taille fixe."""
        for i in range(0, len(iterable), size):
            yield iterable[i:i + size]

    # --------------------------------------------------------
    def generate_ts(self):
        """Génère un fichier pivot .ts et des copies par langue."""
        # --------------------------------------------------
        # Vérifications de base
        # --------------------------------------------------
        if not self.plugin_dir or not self.plugin_dir.exists():
            QMessageBox.warning(self, tr("Erreur"), tr("Sélectionnez un dossier valide."))
            return

        self.saved_langs, _ = load_lang_state_shared()
        languages = list(dict.fromkeys(self.saved_langs))

        if not languages:
            QMessageBox.warning(self, tr("Erreur"), tr("Aucune langue sélectionnée."))
            return

        source_locale = self.get_source_language()
        # --------------------------------------------------
        # Préparation des chemins
        # --------------------------------------------------
        plugin_name = self.plugin_dir.name
        i18n_dir = self.plugin_dir / "i18n"
        i18n_dir.mkdir(exist_ok=True)
        pivot_ts = i18n_dir / f"{plugin_name}.ts"
        # --------------------------------------------------
        # Sauvegarde des paramètres
        # --------------------------------------------------
        save_lang_state_shared(self.saved_langs, multi=False)
        # --------------------------------------------------
        # Collecte des sources
        # --------------------------------------------------
        sources = self.collect_pylupdate_sources(self.plugin_dir)

        if not sources:
            QMessageBox.warning(
                self,
                tr("Erreur"),
                tr("Aucun fichier .py ou .ui trouvé dans le plugin sélectionné.")
            )
            return

        qgis_log(f"[pylupdate] {len(sources)} fichiers source détectés", "DEBUG")
        # --------------------------------------------------
        # UI progression
        # --------------------------------------------------
        total_steps = 1 + len(languages)
        self.progressBar.setMinimum(0)
        self.progressBar.setMaximum(total_steps)
        self.progressBar.setValue(0)
        self.labelStatus.setText(tr("Création du fichier pivot…"))
        QApplication.processEvents()

        # --------------------------------------------------
        # Génération du pivot TS — MODE OFFICIEL Qt (.pro)
        # --------------------------------------------------
        pro_file = None

        try:
            pyexec, pylu = detect_pylupdate()
            if pyexec is None or pylu is None:
                raise RuntimeError("pylupdate introuvable")

            # --------------------------------------------------
            # Création du fichier .pro temporaire
            # --------------------------------------------------
            with tempfile.NamedTemporaryFile(
                    mode="w",
                    suffix=".pro",
                    delete=False,
                    encoding="utf-8"
            ) as f:
                for src in sources:
                    if src.endswith(".py"):
                        f.write(f"SOURCES += {src}\n")
                    elif src.endswith(".ui"):
                        f.write(f"FORMS += {src}\n")

                f.write(f"\nTRANSLATIONS += {pivot_ts}\n")
                pro_file = f.name

            qgis_log(f"[pylupdate] Fichier .pro généré : {pro_file}", "DEBUG")

            # --------------------------------------------------
            # Commande pylupdate (MODULE)
            # --------------------------------------------------
            cmd = [
                str(pyexec),
                "-m", "PyQt5.pylupdate_main",
                pro_file
            ]

            qgis_log(f"[pylupdate] Commande exécutée : {' '.join(cmd)}", "INFO")

            kwargs = {}
            if platform.system().lower() == "windows":
                kwargs["creationflags"] = subprocess.CREATE_NO_WINDOW

            proc = subprocess.run(
                cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
                **kwargs
            )

            if proc.returncode != 0:
                raise RuntimeError(proc.stderr or proc.stdout)

        except Exception as e:
            QMessageBox.critical(
                self,
                tr("Erreur"),
                tr("Erreur lors de la génération du fichier TS :\n{0}").format(e)
            )
            return

        finally:
            if pro_file:
                try:
                    os.remove(pro_file)
                except Exception:
                    pass

        # --------------------------------------------------
        # ✅ Injection metadata.txt → pivot TS
        # --------------------------------------------------
        metadata_txt = self.plugin_dir / "metadata.txt"
        if metadata_txt.exists():
            try:
                inject_metadata_into_ts(pivot_ts, metadata_txt)
                qgis_log(
                    f"[pylupdate] metadata injecté dans {pivot_ts.name}",
                    "INFO"
                )
            except Exception as e:
                qgis_log(
                    f"[pylupdate] échec injection metadata : {e}",
                    "WARNING"
                )

        # --------------------------------------------------
        # Mise à jour entête du pivot
        # --------------------------------------------------
        set_ts_header(
            pivot_ts,
            language=None,
            sourcelanguage=source_locale
        )

        # --------------------------------------------------
        # TS explicite langue source
        # --------------------------------------------------
        source_short = source_locale.split("_")[0]
        source_ts = i18n_dir / f"{plugin_name}_{source_short}.ts"

        try:
            shutil.copyfile(pivot_ts, source_ts)

            set_ts_header(
                source_ts,
                language=source_locale,
                sourcelanguage=source_locale
            )

            self.autofill_source_ts(source_ts)
        except Exception as e:
            qgis_log(f"Erreur TS langue source : {e}", "ERROR")
        # --------------------------------------------------
        # Progression
        # --------------------------------------------------
        self.progressBar.setValue(1)
        self.labelStatus.setText(tr("Pivot créé : {0}").format(pivot_ts.name))
        QApplication.processEvents()
        # --------------------------------------------------
        # Création des TS langues cibles
        # --------------------------------------------------
        errors = []
        step = 1

        for lang_code in languages:
            step += 1
            qt_locale = LANG_CODE_TO_QT.get(lang_code)

            if not qt_locale:
                errors.append(lang_code)
                continue

            dest_ts = i18n_dir / f"{plugin_name}_{lang_code}.ts"

            try:
                shutil.copyfile(pivot_ts, dest_ts)
                set_ts_header(
                    dest_ts,
                    language=qt_locale,
                    sourcelanguage=source_locale
                )

            except Exception as e:
                errors.append(f"{lang_code}: {e}")

            self.progressBar.setValue(step)
            self.labelStatus.setText(tr("Création {0}…").format(dest_ts.name))
            QApplication.processEvents()

        # --------------------------------------------------
        # Fin
        # --------------------------------------------------
        self.progressBar.setValue(total_steps)
        self.labelStatus.setText(tr("Terminé."))
        QApplication.processEvents()

        if errors:
            QMessageBox.warning(
                self,
                tr("Avertissement"),
                tr("Génération terminée avec avertissements :\n") + "\n".join(errors)
            )
        else:
            QMessageBox.information(
                self,
                tr("Terminé"),
                tr("Génération du fichier pivot et des langues effectuée avec succès.")
            )
        # Proposer d'ouvrir le traducteur
        resp = QMessageBox.question(
            self,
            tr("Traduction"),
            tr("Voulez-vous lancer la traduction maintenant ?"),
            QMessageBox.Yes | QMessageBox.No,
        )

        if resp == QMessageBox.Yes:
            self.close()
            if self.parent_plugin:
                self.parent_plugin.run_translator(plugin_dir=self.plugin_dir)
            else:
                QMessageBox.critical(
                    self,
                    tr("Erreur"),
                    tr("Impossible d'appeler run_translator() : parent_plugin manquant.")
                )

    def autofill_source_ts(self, ts_path: Path):
        """
        Copie <source> → <translation> pour la langue source.
        Supprime le flag unfinished.
        """
        tree = ET.parse(ts_path)
        root = tree.getroot()

        for message in root.findall(".//message"):
            src = message.find("source")
            trn = message.find("translation")

            if src is None or trn is None:
                continue

            trn.text = src.text
            trn.attrib.pop("type", None)

        tree.write(ts_path, encoding="utf-8", xml_declaration=True)

# ----------------------------------------------------------------------
#  Classe principale du plugin QGIS
# ----------------------------------------------------------------------
class TsGenerator:
    """Classe principale du plugin QGIS : gestion interface et actions."""

    # @staticmethod
    # def tr(text: str) -> str:
    #     return QCoreApplication.translate("TsGenerator", text)

    def __init__(self, iface):
        """
        Constructor.
        :param iface: instance de QgisInterface (fournie par QGIS)
        """
        self.iface = iface
        self.dlg = None
        self.action = None
        self.action_translator = None  # bouton Traducteur
        self._translator_gui = None

    # ------------------------------------------------------------------
    def initGui(self):
        """Initialise l'interface graphique du plugin dans QGIS."""

        icon_ts = QIcon(":/images/themes/default/mActionEditTable.svg")
        icon_translate = QIcon(str(Path(__file__).parent / "icon.svg"))
        # icon_translate = QIcon(":/images/themes/default/labelingRuleBased.svg")

        # Action : Générateur de fichiers TS
        self.action = QAction(
            icon_ts,
            tr("Générateur de fichiers .TS"),
            self.iface.mainWindow()
        )
        self.action.triggered.connect(self.run)

        # Action : Traducteur (Google)
        self.action_translator = QAction(
            icon_translate,
            tr("Traducteur Google"),
            self.iface.mainWindow()
        )
        self.action_translator.triggered.connect(self.run_translator)

        # Ajouts dans QGIS
        self.iface.addToolBarIcon(self.action)
        self.iface.addToolBarIcon(self.action_translator)
        self.iface.addPluginToMenu("&Plugin Translator", self.action)
        self.iface.addPluginToMenu("&Plugin Translator", self.action_translator)

    # ------------------------------------------------------------------
    def unload(self):
        """Nettoyage des actions QGIS lors de la désactivation du plugin."""
        if self.action:
            self.iface.removeToolBarIcon(self.action)
            self.iface.removePluginMenu("&Plugin Translator", self.action)
            self.action = None

        if self.action_translator:
            self.iface.removeToolBarIcon(self.action_translator)
            self.iface.removePluginMenu("&Plugin Translator", self.action_translator)
            self.action_translator = None

    # ------------------------------------------------------------------
    def run_translator(self, plugin_dir=None):
        try:
            import sip
            from .translate_ts_html_gui_API_google import StartGUI

            if getattr(self, "dlg", None) is not None:
                try:
                    if self.dlg.isVisible():
                        self.dlg.close()
                except Exception:
                    pass

            if getattr(self, "_translator_gui", None) and not sip.isdeleted(self._translator_gui):
                self._translator_gui.raise_()
                self._translator_gui.activateWindow()
                return

            self._translator_gui = StartGUI(
                parent=self.iface.mainWindow(),
                plugin_dir=plugin_dir,  # ✅ ICI
            )

            self._translator_gui.destroyed.connect(
                lambda: setattr(self, "_translator_gui", None)
            )

            self._translator_gui.setWindowFlags(Qt.Window | Qt.WindowStaysOnTopHint)
            self._translator_gui.show()
            self._translator_gui.raise_()
            self._translator_gui.activateWindow()

        except Exception as e:
            self.iface.messageBar().pushCritical(
                "Plugin Translator",
                tr("Erreur au lancement du traducteur : {}").format(e)
            )

    # ------------------------------------------------------------------
    def run(self):
        """Affiche la boîte de dialogue principale du plugin."""
        if not self.dlg:
            self.dlg = TsGeneratorDialog(self.iface.mainWindow(), parent_plugin=self)

        self.dlg.show()
        self.dlg.raise_()
        self.dlg.activateWindow()


