# -*- coding: utf-8 -*-
"""
/***************************************************************************
 TsGenerator
                                 A QGIS plugin
 Edition des fichiers .ts nécessaires à la traduction automatique d'un plugin
 ***************************************************************************/
"""

import sys
import logging
from pathlib import Path
from qgis.PyQt.QtCore import QSettings
from qgis.core import QgsMessageLog, Qgis

# ----------------------------------------------------------------------
# 🔧 LOGGING GLOBAL (INFO / TRACE / WARNING / ERROR)
# ----------------------------------------------------------------------

LOG_LEVELS = {
    "ERROR": 0,
    "WARNING": 1,
    "INFO": 2,
    "TRACE": 3,
}

settings = QSettings()
CURRENT_LOG_LEVEL = settings.value(
    "PluginTranslator/log_level",
    LOG_LEVELS["WARNING"],   # 🔕 PROD = WARNING + ERROR
    type=int
)
# 🔒 Sécurité PROD : si absent, on force WARNING
if not settings.contains("PluginTranslator/log_level"):
    settings.setValue("PluginTranslator/log_level", LOG_LEVELS["WARNING"])
    CURRENT_LOG_LEVEL = LOG_LEVELS["WARNING"]

def qgis_log(message, level="INFO", tag="PluginTranslator"):
    if LOG_LEVELS.get(level, 99) > CURRENT_LOG_LEVEL:
        return

    qgis_level = {
        "ERROR": Qgis.Critical,
        "WARNING": Qgis.Warning,
        "INFO": Qgis.Info,
        "TRACE": Qgis.Info,
    }.get(level, Qgis.Info)

    QgsMessageLog.logMessage(str(message), tag, qgis_level)

# ----------------------------------------------------------------------
# 🧼 SILENCE DES LOGS VENDOR (bs4, chardet, html5lib, etc.)
# ----------------------------------------------------------------------

def silence_vendor_logs():
    noisy = [
        "bs4",
        "chardet",
        "html5lib",
        "soupsieve",
        "urllib3",
        "charset_normalizer",
    ]
    for name in noisy:
        logger = logging.getLogger(name)
        logger.setLevel(logging.CRITICAL)
        logger.propagate = False

silence_vendor_logs()

# ----------------------------------------------------------------------
# 📦 AJOUT DES DEPENDANCES VENDOR AU PYTHONPATH
# ----------------------------------------------------------------------

_vendor = Path(__file__).parent / "vendor"
if _vendor.exists():
    vendor_paths = [
        str(_vendor),
        str(_vendor / "html5lib"),
        str(_vendor / "webencodings"),
        str(_vendor / "bs4"),
        str(_vendor / "soupsieve"),
    ]
    for p in vendor_paths:
        if p not in sys.path:
            sys.path.insert(0, p)
else:
    qgis_log("Vendor folder not found !", "ERROR")

# ----------------------------------------------------------------------
# 🧪 TEST IMPORT HTML5LIB (TRACE uniquement)
# ----------------------------------------------------------------------

try:
    import html5lib  # noqa
    qgis_log("html5lib import OK", "TRACE")
except Exception as e:
    qgis_log(f"CRITICAL — html5lib FAILED to import: {e}", "ERROR")

# ----------------------------------------------------------------------
# 🔧 DECORATEUR @debug_log (DISPONIBLE PARTOUT)
# ----------------------------------------------------------------------

import functools
import time

def debug_log(level="TRACE"):
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            qgis_log(f"▶ {func.__qualname__}()", level)
            t0 = time.time()
            try:
                return func(*args, **kwargs)
            finally:
                dt = time.time() - t0
                qgis_log(
                    f"◀ {func.__qualname__}() terminé en {dt:.3f}s",
                    level
                )
        return wrapper
    return decorator

# ----------------------------------------------------------------------
# 🚀 FACTORY QGIS
# ----------------------------------------------------------------------

def classFactory(iface):  # pylint: disable=invalid-name
    from . import resources  # noqa
    from .plugin_translator import TsGenerator
    return TsGenerator(iface)
