# -*- coding: utf-8 -*-
# Script QGIS: Traduction .ts via DeepL + compilation .qm
# - pathlib only
# - traduit uniquement type="unfinished"
# - ETA/temps restant dynamiques
# - log complet: i18n/Documents/traductions.log
# - compile tous les .ts en .qm avec lrelease.exe
# - mémorise clé API, dossier i18n, chemin lrelease

from pathlib import Path
import sys
import time
import xml.etree.ElementTree as ET
import subprocess

from PyQt5.QtCore import Qt, QThread, pyqtSignal, QTimer, QSettings, QSize
from PyQt5.QtWidgets import (
    QApplication, QWidget, QVBoxLayout, QHBoxLayout, QLabel, QLineEdit,
    QPushButton, QFileDialog, QProgressBar, QMessageBox
)
import deepl

APP_SETTINGS_ORG = "PostGIS_Auth_Tools"
APP_SETTINGS_APP = "DeepL_TS_Translator"

# Mapping suffix fichier -> code DeepL (cf. https://developers.deepl.com/)
DEEPL_CODES = {
    "fr": "FR",
    "en": "EN-GB",  # ou EN-US selon besoin
    "de": "DE",
    "es": "ES",
    "it": "IT",
    "pt": "PT-PT",  # DeepL exige PT-PT ou PT-BR
    "br": "PT-BR",  # si jamais tu utilises *_br.ts
    "nl": "NL",
    "pl": "PL",
    "ru": "RU",
    "ja": "JA",
    "zh": "ZH",     # générique (DeepL gère ZH-HANS/ZH-HANT en option)
    "cn": "ZH",     # alias fréquent
    "cs": "CS",
    "sk": "SK",
    "sl": "SL",
    "sv": "SV",
    "fi": "FI",
    "da": "DA",     # Danois
    "dk": "DA",     # alias de convenance pour fichiers *_dk.ts
    "bg": "BG",
    "el": "EL",
    "ro": "RO",
    "hu": "HU",
    "et": "ET",
    "lt": "LT",
    "lv": "LV",
    "ga": "GA",
    "he": "HE",     # Hébreu
    "il": "HE",     # alias pour *_il.ts
    "uk": "UK",     # Ukrainien
    "id": "ID",
    "tr": "TR"
}

def human_time(seconds: float) -> str:
    """Format adaptatif: HHhMMmnSSs / MMmnSSs / SSs"""
    seconds = int(max(0, round(seconds)))
    h, r = divmod(seconds, 3600)
    m, s = divmod(r, 60)
    parts = []
    if h > 0:
        parts.append(f"{h}h")
    if h > 0 or m > 0:
        parts.append(f"{m}mn")
    parts.append(f"{s}s")
    return "".join(parts)

def detect_lrelease() -> Path:
    """Recherche approfondie de lrelease.exe dans les installations QGIS, Qt ou PyCharm."""
    import shutil

    # 1️⃣ Via PATH
    found = shutil.which("lrelease.exe")
    if found:
        return Path(found)

    exe = Path(sys.executable)
    search_roots = [
        exe.parent,                  # ...\Python312
        exe.parent.parent,           # ...\apps
        exe.parent.parent.parent,    # ...\QGIS 3.40.x
    ]

    # 2️⃣ Recherche récursive (limite profondeur pour vitesse)
    for root in search_roots:
        if not root.exists():
            continue
        try:
            for path in root.rglob("lrelease.exe"):
                return path
        except PermissionError:
            pass

    # 3️⃣ Recherche dans PyCharm ou Qt éventuels
    extra_candidates = [
        Path("C:/Program Files/JetBrains"),
        Path("C:/Program Files (x86)/JetBrains"),
        Path("C:/Qt"),
        Path("C:/Program Files/QGIS 3.40.6"),
    ]
    for base in extra_candidates:
        if base.exists():
            try:
                for path in base.rglob("lrelease.exe"):
                    return path
            except Exception:
                continue

    return Path()
    
    
class TranslationWorker(QThread):
    progressFiles = pyqtSignal(int, int)         # processed_files, total_files
    progressMsgs = pyqtSignal(int, int)          # processed_msgs, total_msgs
    status = pyqtSignal(str)                     # status text
    apiError = pyqtSignal(str)                   # API error (for red display)
    finished = pyqtSignal(int, int, float)       # translated_files, compiled_files, total_seconds
    etaUpdated = pyqtSignal(float, float)        # total_estimated_sec, remaining_sec

    def __init__(self, i18n_dir: Path, api_key: str, lrelease_path: Path):
        super().__init__()
        self.i18n_dir = i18n_dir
        self.api_key = api_key
        self.lrelease = lrelease_path
        self._stop = False
        self._avg_per_msg = None  # moving average (sec/msg)
        self._start_time = None

    def stop(self):
        self._stop = True

    def _infer_target_lang(self, ts_path: Path) -> str:
        """Déduit la langue depuis le nom du fichier .ts (suffixe _xx)."""
        # Ex: postgis_auth_fr.ts -> fr
        stem = ts_path.stem  # "postgis_auth_fr"
        # Essaye les deux derniers segments séparés par _ ou -
        for sep in ("_", "-"):
            parts = stem.split(sep)
            if len(parts) >= 2:
                suffix = parts[-1].lower()
                if suffix in DEEPL_CODES:
                    return DEEPL_CODES[suffix]
        # Sinon, lire éventuellement l'attribut <TS language="..."> si dispo
        try:
            tree = ET.parse(str(ts_path))
            lang_attr = tree.getroot().attrib.get("language", "").lower()
            if lang_attr:
                # normalise zh_CN -> zh
                norm = lang_attr.replace("_", "-").split("-")[0]
                return DEEPL_CODES.get(norm, "").upper()
        except Exception:
            pass
        return ""  # non supporté -> sera logué & ignoré

    def _compile_ts(self, ts_files):
        compiled = 0
        for ts in ts_files:
            qm = ts.with_suffix(".qm")
            try:
                subprocess.run(
                    [str(self.lrelease), str(ts), "-qm", str(qm)],
                    check=True,
                    capture_output=True
                )
                compiled += 1
            except Exception as e:
                self.apiError.emit(f"[lrelease] {ts.name}: {e}")
        return compiled

    def run(self):
        self._start_time = time.time()

        log_dir = self.i18n_dir / "Documents"
        log_dir.mkdir(parents=True, exist_ok=True)
        log_file = log_dir / "traductions.log"

        ts_files = sorted([p for p in self.i18n_dir.glob("*.ts")])
        total_files = len(ts_files)
        processed_files = 0
        translated_files = 0

        # Comptage initial de tous les messages "unfinished"
        total_msgs = 0
        for ts in ts_files:
            try:
                root = ET.parse(str(ts)).getroot()
                total_msgs += sum(
                    1 for t in root.findall(".//message/translation")
                    if t is not None and t.get("type") == "unfinished"
                )
            except Exception:
                pass

        processed_msgs = 0

        with log_file.open("a", encoding="utf-8") as log:
            log.write("\n" + "=" * 70 + "\n")
            log.write(f"Début: {time.strftime('%Y-%m-%d %H:%M:%S')}\n")
            log.write(f"Dossier i18n: {self.i18n_dir}\n")
            log.write(f"lrelease: {self.lrelease if self.lrelease else '[NON TROUVÉ]'}\n")
            log.write(f"Fichiers .ts détectés: {total_files}\n")
            log.write(f"Total messages unfinished détectés: {total_msgs}\n")

            # Traduction
            if total_files == 0:
                self.status.emit("Aucun fichier .ts trouvé.")
            else:
                try:
                    translator = deepl.Translator(self.api_key)
                    translator.get_usage()  # validation rapide
                except Exception as e:
                    msg = f"[API] Échec validation clé: {e}"
                    self.apiError.emit(msg)
                    log.write(msg + "\n")
                    # Ne pas quitter pour permettre tout de même la compilation éventuelle

                for ts in ts_files:
                    if self._stop:
                        break

                    tgt = self._infer_target_lang(ts)
                    if not tgt:
                        warn = f"[SKIP] {ts.name}: Langue non supportée/inconnue (suffixe)."
                        self.status.emit(warn)
                        log.write(warn + "\n")
                        processed_files += 1
                        self.progressFiles.emit(processed_files, total_files)
                        continue

                    self.status.emit(f"Traduction: {ts.name} → {tgt}")
                    log.write(f"→ {ts.name} ({tgt})\n")

                    changed = False
                    try:
                        tree = ET.parse(str(ts))
                        root = tree.getroot()

                        for message in root.findall(".//message"):
                            if self._stop:
                                break
                            source = message.find("source")
                            translation = message.find("translation")
                            if source is None or translation is None:
                                continue

                            if translation.get("type") == "unfinished":
                                src_text = (source.text or "").strip()
                                if not src_text:
                                    continue
                                t0 = time.time()
                                try:
                                    new_text = translator.translate_text(
                                        src_text, target_lang=tgt
                                    ).text
                                    translation.text = new_text
                                    translation.attrib.pop("type", None)
                                    changed = True
                                except Exception as e:
                                    # Erreur API: affiche en rouge + log
                                    api_msg = f"[API] {ts.name} «{src_text[:30]}...» : {e}"
                                    self.apiError.emit(api_msg)
                                    log.write(api_msg + "\n")
                                finally:
                                    # Mise à jour moyenne mobile
                                    dt = time.time() - t0
                                    if dt > 0:
                                        if self._avg_per_msg is None:
                                            self._avg_per_msg = dt
                                        else:
                                            self._avg_per_msg = 0.9 * self._avg_per_msg + 0.1 * dt

                                processed_msgs += 1
                                self.progressMsgs.emit(processed_msgs, total_msgs)

                                # ETA dynamique
                                if self._avg_per_msg and total_msgs > 0:
                                    remaining_msgs = max(0, total_msgs - processed_msgs)
                                    est_total = self._avg_per_msg * total_msgs
                                    elapsed = time.time() - self._start_time
                                    remaining = max(0.0, est_total - elapsed)
                                    self.etaUpdated.emit(est_total, remaining)

                        if changed:
                            tree.write(str(ts), encoding="utf-8", xml_declaration=True)
                            translated_files += 1
                            log.write(f"   ✓ mis à jour\n")
                        else:
                            log.write(f"   – aucune entrée unfinished, non modifié\n")

                    except Exception as e:
                        err = f"[ERREUR] {ts.name}: {e}"
                        self.apiError.emit(err)
                        log.write(err + "\n")

                    processed_files += 1
                    self.progressFiles.emit(processed_files, total_files)

            # Compilation .qm (tous les .ts)
            compiled_files = 0
            if self.lrelease and self.lrelease.exists():
                self.status.emit("Compilation .qm avec lrelease…")
                try:
                    compiled_files = self._compile_ts(ts_files)
                    log.write(f"Compilation: {compiled_files}/{len(ts_files)} .qm générés\n")
                except Exception as e:
                    msg = f"[lrelease] Erreur de compilation: {e}"
                    self.apiError.emit(msg)
                    log.write(msg + "\n")
            else:
                msg = "[lrelease] Introuvable — compilation .qm ignorée"
                self.apiError.emit(msg)
                log.write(msg + "\n")

            total_sec = time.time() - self._start_time
            log.write(f"Fin: {time.strftime('%Y-%m-%d %H:%M:%S')}  (durée {human_time(total_sec)})\n")

        self.finished.emit(translated_files, compiled_files, time.time() - self._start_time)


class MiniGUI(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("DeepL → .ts → .qm")
        self.setWindowFlags(Qt.WindowStaysOnTopHint)
        self.setFixedSize(QSize(480, 200))  # compact

        # Centrage + 1/4 écran
        screen = QApplication.primaryScreen().availableGeometry()
        w = max(420, screen.width() // 4)
        h = max(180, screen.height() // 4)
        self.setFixedSize(w, h)
        self.move(
            screen.center().x() - self.width() // 2,
            screen.center().y() - self.height() // 2
        )

        self.settings = QSettings(APP_SETTINGS_ORG, APP_SETTINGS_APP)

        v = QVBoxLayout(self)
        v.setContentsMargins(10, 10, 10, 10)
        v.setSpacing(6)

        # Ligne 1: Dossier i18n + bouton
        row1 = QHBoxLayout()
        self.path_edit = QLineEdit()
        self.path_edit.setPlaceholderText("Choisir le dossier i18n…")
        last_dir = self.settings.value("last_i18n_dir", "")
        self.path_edit.setText(last_dir)
        btn_browse = QPushButton("📁")
        btn_browse.setToolTip("Choisir dossier i18n")
        btn_browse.clicked.connect(self.choose_folder)
        row1.addWidget(QLabel("Dossier i18n:"))
        row1.addWidget(self.path_edit, 1)
        row1.addWidget(btn_browse)
        v.addLayout(row1)

        # Ligne 2: API Key
        row2 = QHBoxLayout()
        self.api_edit = QLineEdit()
        self.api_edit.setPlaceholderText("Clé API DeepL")
        self.api_edit.setText(self.settings.value("deepl_api_key", ""))
        self.api_edit.setEchoMode(QLineEdit.Password)
        btn_show = QPushButton("👁")
        btn_show.setToolTip("Afficher/Masquer")
        btn_show.setCheckable(True)
        btn_show.toggled.connect(self.toggle_echo)
        row2.addWidget(QLabel("API key:"))
        row2.addWidget(self.api_edit, 1)
        row2.addWidget(btn_show)
        v.addLayout(row2)

        # Ligne 3: lrelease
        row3 = QHBoxLayout()
        self.lrelease_edit = QLineEdit()
        self.lrelease_edit.setPlaceholderText("Chemin de lrelease.exe (optionnel si trouvé auto)")
        auto_lrelease = detect_lrelease()
        saved_lrelease = Path(self.settings.value("lrelease_path", "")) if self.settings.value("lrelease_path", "") else Path()
        use_lrelease = saved_lrelease if saved_lrelease.exists() else auto_lrelease
        if use_lrelease:
            self.lrelease_edit.setText(str(use_lrelease))
        btn_lrel = QPushButton("🔎")
        btn_lrel.setToolTip("Parcourir lrelease.exe")
        btn_lrel.clicked.connect(self.choose_lrelease)
        row3.addWidget(QLabel("lrelease:"))
        row3.addWidget(self.lrelease_edit, 1)
        row3.addWidget(btn_lrel)
        v.addLayout(row3)

        # Ligne 4: Statut + temps
        self.time_label = QLabel("ETA: —  |  Restant: —")
        self.status_label = QLabel("Prêt.")
        v.addWidget(self.time_label)
        v.addWidget(self.status_label)

        # Ligne 5: Progress bar
        self.file_progress = QProgressBar()
        self.msg_progress = QProgressBar()
        v.addWidget(self.file_progress)
        v.addWidget(self.msg_progress)

        # Ligne 6: Boutons
        row6 = QHBoxLayout()
        self.btn_run = QPushButton("▶ Traduire & Compiler")
        self.btn_run.clicked.connect(self.launch)
        self.btn_open = QPushButton("Ouvrir i18n")
        self.btn_open.clicked.connect(self.open_i18n)
        self.btn_open.setEnabled(bool(self.path_edit.text()))
        row6.addWidget(self.btn_run)
        row6.addWidget(self.btn_open)
        v.addLayout(row6)

        # Timer UI pour compte à rebours fluide
        self._last_eta_total = None
        self._last_eta_remaining = None
        self._t0 = None
        self._timer = QTimer(self)
        self._timer.setInterval(200)  # 5 fois/sec
        self._timer.timeout.connect(self._tick)

        self.worker = None

    def toggle_echo(self, checked: bool):
        self.api_edit.setEchoMode(QLineEdit.Normal if checked else QLineEdit.Password)

    def choose_folder(self):
        d = QFileDialog.getExistingDirectory(self, "Choisir le dossier i18n")
        if d:
            self.path_edit.setText(d)
            self.btn_open.setEnabled(True)
            self.settings.setValue("last_i18n_dir", d)  # ✅ persistance immédiate

    def choose_lrelease(self):
        f, _ = QFileDialog.getOpenFileName(self, "Sélectionner lrelease.exe", "", "lrelease (lrelease*.exe)")
        if f:
            self.lrelease_edit.setText(f)

    def open_i18n(self):
        p = Path(self.path_edit.text().strip())
        if p and p.exists():
            try:
                if sys.platform.startswith("win"):
                    subprocess.Popen(["explorer", str(p)])
                elif sys.platform == "darwin":
                    subprocess.Popen(["open", str(p)])
                else:
                    subprocess.Popen(["xdg-open", str(p)])
            except Exception:
                pass

    def _tick(self):
        # Met à jour l'affichage du reste du temps en compte à rebours
        if self._last_eta_total is None or self._last_eta_remaining is None or self._t0 is None:
            return
        elapsed = time.time() - self._t0
        remaining = max(0.0, self._last_eta_remaining - elapsed)
        self.time_label.setText(f"ETA: {human_time(self._last_eta_total)}  |  Restant: {human_time(remaining)}")

    def launch(self):
        i18n_dir = Path(self.path_edit.text().strip())
        raw = self.path_edit.text().strip()
        if not raw:
            QMessageBox.warning(self, "Erreur", "Veuillez choisir un dossier i18n.")
            return
        i18n_dir = Path(raw)
        # Évite le cas Path("") == "."
        try:
            if i18n_dir.resolve() == Path(".").resolve():
                QMessageBox.warning(self, "Erreur", "Veuillez choisir un dossier i18n (pas le dossier courant).")
                return
        except Exception:
            pass
        if not i18n_dir.exists() or not i18n_dir.is_dir():
            QMessageBox.warning(self, "Erreur", "Veuillez choisir un dossier i18n valide.")
            return
        ts_files = list(i18n_dir.glob("*.ts"))
        if not ts_files:
            QMessageBox.warning(self, "Erreur", "Aucun fichier .ts trouvé dans ce dossier i18n.")
            return

        api_key = self.api_edit.text().strip()
        lrelease_path = Path(self.lrelease_edit.text().strip()) if self.lrelease_edit.text().strip() else detect_lrelease()

        if not i18n_dir.exists():
            QMessageBox.warning(self, "Erreur", "Veuillez choisir un dossier i18n valide.")
            return
        if not api_key:
            QMessageBox.warning(self, "Erreur", "Veuillez saisir une clé API DeepL.")
            return

        # Persist
        self.settings.setValue("last_i18n_dir", str(i18n_dir))
        self.settings.setValue("deepl_api_key", api_key)
        if lrelease_path:
            self.settings.setValue("lrelease_path", str(lrelease_path))

        # Reset UI
        self.file_progress.setValue(0)
        self.msg_progress.setValue(0)
        self.status_label.setText("Démarrage…")
        self.time_label.setText("ETA: —  |  Restant: —")

        self.worker = TranslationWorker(i18n_dir, api_key, lrelease_path)
        self.worker.progressFiles.connect(self._on_files)
        self.worker.progressMsgs.connect(self._on_msgs)
        self.worker.status.connect(self._on_status)
        self.worker.apiError.connect(self._on_api_error)
        self.worker.etaUpdated.connect(self._on_eta)
        self.worker.finished.connect(self._on_done)
        self._t0 = time.time()
        self._timer.start()
        self.btn_run.setEnabled(False)
        self.worker.start()

    def _on_files(self, done, total):
        self.file_progress.setMaximum(max(1, total))
        self.file_progress.setValue(done)

    def _on_msgs(self, done, total):
        self.msg_progress.setMaximum(max(1, total))
        self.msg_progress.setValue(done)

    def _on_status(self, txt):
        self.status_label.setStyleSheet("")  # normal
        self.status_label.setText(txt)

    def _on_api_error(self, txt):
        # rouge + message
        self.status_label.setStyleSheet("color:#b80d0d;")
        self.status_label.setText(txt)

    def _on_eta(self, total_sec, remaining_sec):
        # Mémorise la dernière estimation — le QTimer se charge d'afficher le compte à rebours
        self._last_eta_total = total_sec
        self._last_eta_remaining = remaining_sec
        self._t0 = time.time()

    def _on_done(self, translated_files, compiled_files, total_seconds):
        self._timer.stop()
        self.btn_run.setEnabled(True)
        self.time_label.setText(f"ETA: {human_time(total_seconds)}  |  Restant: 0s")
        self.status_label.setStyleSheet("")
        self.status_label.setText(f"Terminé. {translated_files} fichier(s) traduit(s), {compiled_files} .qm compilé(s).")

        msg = QMessageBox(self)
        msg.setWindowTitle("Terminé")
        msg.setText(f"✅ Traduction & compilation terminées.\n"
                    f"• Traduits : {translated_files}\n"
                    f"• Compilés : {compiled_files}\n"
                    f"• Durée : {human_time(total_seconds)}")
        msg.setIcon(QMessageBox.Information)
        msg.setStandardButtons(QMessageBox.Ok)
        ret = msg.exec_()
        if ret == QMessageBox.Ok:
            # Ferme la GUI
            self.close()

