# -*- coding: utf-8 -*-
"""
LangTree — composant commun pour sélection de langues par zones (QGIS / Qt)

- LANG_ZONES = source de vérité unique
- Construction QTreeWidget hiérarchique
- Synchronisation robuste avec une liste de langues sélectionnées
- Compatible Qt5 / Qt6
"""

from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import QTreeWidgetItem

# ----------------------------------------------------------------------
#  Définition des langues par zones
#  ⚠️ Clés internes NON traduites
# ----------------------------------------------------------------------
LANG_ZONES = {
    "Europe Ouest": {
        "fr": "Français",
        "en": "Anglais",
        "es": "Espagnol",
        "pt": "Portugais",
        "it": "Italien",
        "de": "Allemand",
        "nl": "Néerlandais",
        "sv": "Suédois",
        "no": "Norvégien",
        "da": "Danois",
        "fi": "Finnois",
    },
    "Europe Est": {
        "pl": "Polonais",
        "cs": "Tchèque",
        "sk": "Slovaque",
        "sl": "Slovène",
        "hr": "Croate",
        "sr": "Serbe",
        "ro": "Roumain",
        "hu": "Hongrois",
        "bg": "Bulgare",
        "el": "Grec",
        "ru": "Russe",
        "uk": "Ukrainien",
    },
    "Moyen-Orient": {
        "he": "Hébreu",
        "ar": "Arabe",
        "tr": "Turc",
    },
    "Asie": {
        "zh": "Chinois (simplifié)",
        "zh-TW": "Chinois (traditionnel)",
        "ja": "Japonais",
        "ko": "Coréen",
        "hi": "Hindi",
    },
}

# ----------------------------------------------------------------------
#  Construction de l’arbre
# ----------------------------------------------------------------------
def build_lang_tree(tree, saved_langs, tr=lambda s: s):
    """
    Construit un QTreeWidget hiérarchique zones → langues.

    :param tree: QTreeWidget
    :param saved_langs: list[str] — langues sélectionnées
    :param tr: fonction de traduction
    """
    tree.clear()

    for zone, langs in LANG_ZONES.items():
        zone_item = QTreeWidgetItem([tr(zone)])
        zone_item.setFlags(zone_item.flags() | Qt.ItemIsUserCheckable)
        zone_item.setCheckState(0, Qt.Unchecked)

        tree.addTopLevelItem(zone_item)

        for code, label in langs.items():
            child = QTreeWidgetItem([tr(label)])
            child.setData(0, Qt.UserRole, code)
            child.setFlags(child.flags() | Qt.ItemIsUserCheckable)
            child.setCheckState(
                0,
                Qt.Checked if code in saved_langs else Qt.Unchecked
            )
            zone_item.addChild(child)

        zone_item.setExpanded(False)

# ----------------------------------------------------------------------
#  Synchronisation Tree ← saved_langs
# ----------------------------------------------------------------------
def sync_tree_from_saved(tree, saved_langs):
    """
    Re-synchronise l’arbre depuis la liste des langues sélectionnées.
    """
    for i in range(tree.topLevelItemCount()):
        zone_item = tree.topLevelItem(i)

        checked = 0
        total = zone_item.childCount()

        for j in range(total):
            child = zone_item.child(j)
            code = child.data(0, Qt.UserRole)

            if code in saved_langs:
                child.setCheckState(0, Qt.Checked)
                checked += 1
            else:
                child.setCheckState(0, Qt.Unchecked)

        if checked == 0:
            zone_item.setCheckState(0, Qt.Unchecked)
        elif checked == total:
            zone_item.setCheckState(0, Qt.Checked)
        else:
            zone_item.setCheckState(0, Qt.PartiallyChecked)

# ----------------------------------------------------------------------
#  Synchronisation saved_langs ← Tree
# ----------------------------------------------------------------------
def collect_saved_langs(tree):
    """
    Extrait la liste des langues cochées depuis l’arbre.
    """
    langs = []

    for i in range(tree.topLevelItemCount()):
        zone = tree.topLevelItem(i)
        for j in range(zone.childCount()):
            child = zone.child(j)
            if child.checkState(0) == Qt.Checked:
                code = child.data(0, Qt.UserRole)
                if code:
                    langs.append(code)

    return sorted(set(langs))

# ----------------------------------------------------------------------
#  Gestion du clic hiérarchique
# ----------------------------------------------------------------------
def handle_tree_item_changed(tree, item, saved_langs):
    """
    Gère le comportement hiérarchique (zone ↔ langues).
    Met à jour saved_langs.
    """
    # Clic sur une zone
    if item.childCount() > 0:
        state = item.checkState(0)
        for i in range(item.childCount()):
            item.child(i).setCheckState(0, state)

    # Clic sur une langue
    else:
        parent = item.parent()
        if parent:
            states = [
                parent.child(i).checkState(0)
                for i in range(parent.childCount())
            ]
            if all(s == Qt.Checked for s in states):
                parent.setCheckState(0, Qt.Checked)
            elif all(s == Qt.Unchecked for s in states):
                parent.setCheckState(0, Qt.Unchecked)
            else:
                parent.setCheckState(0, Qt.PartiallyChecked)

    # Mise à jour de la source de vérité
    saved_langs[:] = collect_saved_langs(tree)

# ----------------------------------------------------------------------
#  Helpers UI
# ----------------------------------------------------------------------
def collapse_all(tree):
    for i in range(tree.topLevelItemCount()):
        tree.topLevelItem(i).setExpanded(False)

def expand_all(tree):
    for i in range(tree.topLevelItemCount()):
        tree.topLevelItem(i).setExpanded(True)
