# -*- coding: utf-8 -*-
"""
Plugin QGIS: Traduction via Google Translate — Modes & Multilingue

Fonctionnalités :
  1️⃣ Traduire des fichiers .ts (avec compilation .qm)
  2️⃣ Traduire un fichier HTML
  3️⃣ Traduire un dossier HTML complet (récursif) → sortie dans "translated/"

Caractéristiques :
- Multilingue (liste de langues cibles, mono ou multi-sélection).
- Pas de clé API à gérer : Google Translate endpoint gratuit.
- Utilise le réseau QGIS (QgsNetworkAccessManager) → proxy pris en compte.
- Barre 1 : "Fichiers en cours de traitement : x / total".
- Barre 2 : "Traduction de <fichier.ext> en <Nom complet langue> : nn %".
- TS : évite de retraduire les traductions manuelles (texte ≠ source et pas "unfinished").
- HTML : sortie dans translated/<base>_<suffix>.html
- Fermeture sûre : interruption propre du Worker + confirmation.
"""

from pathlib import Path
import sys
import time
import re
import tempfile
import stat
import xml.etree.ElementTree as ET
import subprocess
from html.parser import HTMLParser
from html import escape as html_escape
from datetime import datetime
from urllib.parse import quote
import json
from . import qgis_log

# -------------------------------------------------------------------
#  LOAD VENDOR PACKAGES — ordre OBLIGATOIRE
# -------------------------------------------------------------------
_vendor = Path(__file__).parent / "vendor"

if _vendor.exists():
    # 1) Racine vendor
    sys.path.insert(0, str(_vendor))

    # 2) Dépendances html5lib
    sys.path.insert(0, str(_vendor / "webencodings"))
    sys.path.insert(0, str(_vendor / "html5lib"))

    # 3) BeautifulSoup + soupsieve
    sys.path.insert(0, str(_vendor / "soupsieve"))
    sys.path.insert(0, str(_vendor / "bs4"))
else:
    from qgis.core import QgsMessageLog, Qgis
    QgsMessageLog.logMessage(
        "Vendor folder not found!", "PluginTranslator", Qgis.Critical
    )

# -------------------------------------------------------------------
#  TEST IMPORT IMMEDIAT (IMPORTANT : html5lib AVANT BeautifulSoup)
# -------------------------------------------------------------------
try:
    import html5lib # ← DOIT être importé avant BeautifulSoup
    import soupsieve
    from bs4 import BeautifulSoup
    import chardet
    import webencodings

except Exception as e:
    from qgis.core import QgsMessageLog, Qgis
    QgsMessageLog.logMessage(
        f"CRITICAL — FAILED to import: {e}",
        "PluginTranslator",
        Qgis.Critical
    )

from . import resources  # force l’enregistrement des ressources

from qgis.PyQt.QtCore import (
    Qt,
    QThread,
    QObject,
    pyqtSignal,
    QSettings,
    QSize,
    QFile,
    QEventLoop,
    QByteArray,
    QUrl,
    QCoreApplication,
    QTimer,
)
from qgis.PyQt.QtNetwork import QNetworkRequest, QNetworkReply
from qgis.core import Qgis, QgsNetworkAccessManager, QgsMessageLog

from qgis.PyQt.QtWidgets import (
    QApplication,
    QWidget,
    QVBoxLayout,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QPushButton,
    QFileDialog,
    QProgressBar,
    QMessageBox,
    QComboBox,
    QListWidget,
    QListWidgetItem,
    QCheckBox,
    QDialog,
    QTextEdit,
    QDialogButtonBox,
    QInputDialog,
    QTreeWidget,
    QTreeWidgetItem,
)

APP_SETTINGS_ORG = "Plugin_translator"
APP_SETTINGS_APP = "Google_TS_Translator"


class TranslationState:
    """
    Gestion centralisée de l'état du traducteur.
    Évite les incohérences entre :
      - lancement
      - fin normale
      - interruption (fermeture fenêtre)
      - relancement
    """

    def __init__(self, logger=qgis_log):
        self._log = logger
        self.reset()

    # --------------------------------------------------
    #  État brut
    # --------------------------------------------------
    def reset(self):
        self.is_running = False   # un worker est-il actif ?
        self.last_mode = None     # 'ts' | 'html_file' | 'html_dir'
        self.last_langs = []      # ['fr', 'pt', ...]
        self.last_html_file = None
        self.last_html_dir = None
        self.last_interrupted = False

    # --------------------------------------------------
    #  Logging interne
    # --------------------------------------------------
    def _dump(self, title):
        self._log(
            "[STATE] " + title + "\n"
            f"          is_running      = {self.is_running}\n"
            f"          last_mode       = {self.last_mode}\n"
            f"          last_langs      = {self.last_langs}\n"
            f"          last_html_file  = {self.last_html_file}\n"
            f"          last_html_dir   = {self.last_html_dir}\n"
            f"          last_interrupted= {self.last_interrupted}",
            "INFO"
        )

    # --------------------------------------------------
    #  Événements de haut niveau
    # --------------------------------------------------
    def on_launch(self, mode, langs, html_file=None, html_dir=None):
        """
        Appelé une seule fois au moment du lancement des traductions.
        """
        self.is_running = True
        self.last_mode = mode
        self.last_langs = list(langs or [])
        self.last_html_file = html_file
        self.last_html_dir = html_dir
        self.last_interrupted = False
        self._dump("WORKER START on_launch()")

    def on_worker_finished(self, interrupted=False):
        """
        Appelé quand le Worker a terminé (normalement ou interrompu).
        """
        self.is_running = False
        self.last_interrupted = interrupted
        self._dump("WORKER FINISHED on_worker_finished()")

    def on_forced_close(self):
        """
        Appelé quand la fenêtre se ferme alors qu'un worker tournait encore.
        """
        # on considère que c'est une interruption
        self.is_running = False
        self.last_interrupted = True
        self._dump("FORCED CLOSE on_forced_close()")

    def on_window_closed_idle(self):
        """
        Appelé quand la fenêtre se ferme sans worker actif.
        Permet juste de tracer proprement.
        """
        self._dump("WINDOW CLOSED IDLE on_window_closed_idle()")


# ----------------------------------------------------------------------
#  Worker GPT pour les HTML
# ----------------------------------------------------------------------

def translate_html_with_gpt(html_text: str, src_lang: str, tgt_lang: str, gpt_client):
    """
    Traduction HTML ULTRA-SAFE utilisant GPT, ligne par ligne.
    - préserve toutes les balises
    - aucune correction grammaticale destructive
    - évite le "Google multiline bug"
    """

    out_lines = []
    buf = []

    # On découpe en lignes sans interpréter l'HTML
    lines = html_text.split("\n")

    for line in lines:
        stripped = line.strip()

        # Si ligne sans texte (balises seules), on laisse tel quel
        if not stripped or stripped.startswith("<") and stripped.endswith(">"):
            out_lines.append(line)
            continue

        # Sinon : traduction GPT avec préservation du HTML
        prompt = f"""
Tu es un traducteur professionnel. Traduis uniquement le texte humain ci-dessous,
sans modifier les balises HTML, ni les espaces insécables (&nbsp;), ni les retours à la ligne :

LANGUE SOURCE : {src_lang}
LANGUE CIBLE  : {tgt_lang}

TEXTE :
{line}

RENVOIE UNIQUEMENT LA LIGNE TRADUITE, AVEC LES MÊMES BALISES HTML.
"""

        try:
            if not gpt_client:
                return html_text
            resp = gpt_client.chat.completions.create(
                model="gpt-4.1-mini",
                messages=[{"role": "user", "content": prompt}],
                temperature=0,
            )

            translated = resp.choices[0].message.content.strip()

        except Exception as e:
            # fallback : conserver la ligne d'origine
            translated = line

        out_lines.append(translated)

    return "\n".join(out_lines)

# ----------------------------------------------------------------------
#  Mini API anti-bug Google Translate
# ----------------------------------------------------------------------

PLACEHOLDER_BRACES = re.compile(r"\{.*?\}")
PLACEHOLDER_QT = re.compile(r"%\d+|%n")

def _protect_placeholders(text):
    """
    Remplace chaque placeholder par un jeton interne.
    {abc}   -> __PH_BRACES_0__
    %1      -> __PH_QT_0__
    """
    ph_map = {}

    # Curly placeholders {abc}
    def repl_br(m):
        k = f"__PH_BRACES_{len(ph_map)}__"
        ph_map[k] = m.group(0)
        return k

    text = PLACEHOLDER_BRACES.sub(repl_br, text)

    # Qt placeholders %1, %2, %n
    def repl_qt(m):
        k = f"__PH_QT_{len(ph_map)}__"
        ph_map[k] = m.group(0)
        return k

    text = PLACEHOLDER_QT.sub(repl_qt, text)

    return text, ph_map


def _restore_placeholders(text, ph_map):
    """Réinjecte les placeholders originaux."""
    for k, v in ph_map.items():
        text = text.replace(k, v)
    return text

def google_translate_line(line, target_lang):
    """Appel Google sur *une seule ligne*, avec protection placeholders."""

    if not line.strip():
        return line

    # Interruption juste après l’envoi
    if getattr(QThread.currentThread(), "_check_abort", lambda: False)():
        return line

    protected, ph_map = _protect_placeholders(line)

    url = (
        "https://translate.googleapis.com/translate_a/single?"
        f"client=gtx&sl=auto&tl={target_lang}&dt=t&q={quote(protected)}"
    )

    nam = QgsNetworkAccessManager.instance()
    req = QNetworkRequest(QUrl(url))
    reply = nam.get(req)

    loop = QEventLoop()
    timer = QTimer()
    timer.setSingleShot(True)

    timer.timeout.connect(lambda: (reply.abort(), loop.quit()))
    reply.finished.connect(loop.quit)

    t0 = time.time()
    timer.start(8000)
    loop.exec_()
    dt = time.time() - t0

    if not timer.isActive() or reply.error() != QNetworkReply.NoError:
        return line  # fallback = ligne originale

    data = reply.readAll().data()
    try:
        arr = json.loads(data)
        if isinstance(arr, list) and arr and isinstance(arr[0], list):
            translated = arr[0][0][0]
        else:
            translated = line
    except:
        translated = line

    # Réinjection des placeholders originaux
    translated = _restore_placeholders(translated, ph_map)

    return translated


def google_translate_multiline(text, target_lang):
    """
    TRADUCTION MULTILIGNE SÉCURISÉE
    - chaque ligne est traduite séparément
    - aucun risque de fusion/ suppression/ réarrangement
    - placeholders toujours préservés
    """

    lines = text.replace("\r\n", "\n").replace("\r", "\n").split("\n")

    translated_lines = [
        google_translate_line(line, target_lang)
        for line in lines
    ]

    return "\n".join(translated_lines)

# ----------------------------------------------------------------------
#  TRADUCTION AVEC CONTEXTES — fonction unique
# ----------------------------------------------------------------------
def tr(context: str, text: str) -> str:
    return QCoreApplication.translate(context, text)

# ----------------------------------------------------------------------
#  Définition des langues par zones
#  (⚠ les clés internes NE SONT PAS traduites !)
# ----------------------------------------------------------------------
LANG_ZONES = {
    "Europe Ouest": {
        "fr": "Français",
        "en": "Anglais",
        "es": "Espagnol",
        "pt": "Portugais",
        "it": "Italien",
        "de": "Allemand",
        "nl": "Néerlandais",
        "sv": "Suédois",
        "no": "Norvégien",
        "da": "Danois",
        "fi": "Finnois",
    },
    "Europe Est": {
        "pl": "Polonais",
        "cs": "Tchèque",
        "sk": "Slovaque",
        "sl": "Slovène",
        "hr": "Croate",
        "sr": "Serbe",
        "ro": "Roumain",
        "hu": "Hongrois",
        "bg": "Bulgare",
        "el": "Grec",
        "ru": "Russe",
        "uk": "Ukrainien",
    },
    "Moyen-Orient": {
        "he": "Hébreu",
        "ar": "Arabe",
        "tr": "Turc",
    },
    "Asie": {
        "zh": "Chinois (simplifié)",
        "zh-TW": "Chinois (traditionnel)",
        "ja": "Japonais",
        "ko": "Coréen",
        "hi": "Hindi",
    },
}

# ----------------------------------------------------------------------
#  LANGUES — construction cohérente pour TS et Google
# ----------------------------------------------------------------------

GOOGLE_EXCEPTIONS = {
    "zh": "zh-CN",
    "zh-TW": "zh-TW",
}

LANG_TARGETS = [
    (
        code.lower(),                                 # Code interne (FR, EN…)
        label,                                        # Nom complet ("Français")
        code.lower(),                                 # Suffixe fichier (_fr.ts)
        GOOGLE_EXCEPTIONS.get(code, code).lower()     # Code Google correct
    )
    for zone, langs in LANG_ZONES.items()
    for code, label in langs.items()
]

# Dictionnaires utilisés partout
CODE_TO_NAME   = {code: name for code, name, _, _ in LANG_TARGETS}
CODE_TO_SUFFIX = {code: suff for code, _, suff, _ in LANG_TARGETS}
CODE_TO_GOOGLE = {code: g    for code, _, _, g    in LANG_TARGETS}
SUFFIX_TO_CODE = {suff: code for code, _, suff, _ in LANG_TARGETS}

TRANSLATABLE_ATTRS = {"title", "alt", "placeholder", "aria-label"}
PLACEHOLDER_RE = re.compile(r'(%\d+|%n|%\w+|{\})')
SKIP_TAGS = {"script", "style", "meta", "title", "link", "noscript"}

# ----------------------------------------------------------------------
#  Utilitaire : reconstituer texte complet d'un élément XML Qt TS
# ----------------------------------------------------------------------
def get_full_text(elem):
    """
    Récupère *tous* les textes d’un élément TS, même si la
    première ligne est dans un nœud texte enfant (cas XML indenté).
    """
    if elem is None:
        return ""

    parts = []

    # 1) texte directement dans la balise
    if elem.text:
        parts.append(elem.text)

    # 2) textes des enfants (rare mais arrive après pretty-print)
    for child in elem:
        if child.text:
            parts.append(child.text)
        if child.tail:
            parts.append(child.tail)

    # 3) texte après la balise
    if elem.tail:
        parts.append(elem.tail)

    # Fusion propre
    return "".join(parts)

# ----------------------------------------------------------------------
#  Autres Utilitaires
# ----------------------------------------------------------------------

def extract_resource_to_temp(path_in_qrc: str):
    """Extrait un binaire (lrelease) depuis les ressources vers un fichier temporaire exécutable."""
    f = QFile(path_in_qrc)
    if not f.exists():
        return None
    if not f.open(QFile.ReadOnly):
        return None

    data = f.readAll()
    f.close()

    temp_path = Path(tempfile.gettempdir()) / Path(path_in_qrc).name
    try:
        with temp_path.open("wb") as out:
            out.write(bytes(data))
        try:
            temp_path.chmod(temp_path.stat().st_mode | stat.S_IEXEC)
        except Exception:
            pass
        return temp_path
    except Exception:
        return None

def detect_lrelease():
    """Détecte lrelease embarqué dans le plugin."""
    path = extract_resource_to_temp(":/tools/lrelease.exe")
    return path if path and path.exists() else None

def ts_base_and_suffix(stem: str):
    """Retourne (base, suffix_lang_or_'') pour TS (suffixe terminal _xx)."""
    if "_" in stem:
        parts = stem.split("_")
        if parts[-1].lower() in SUFFIX_TO_CODE:
            return "_".join(parts[:-1]), parts[-1].lower()
    return stem, ""

# Enregistrement des traductions dans un sous-dossier translated
# def html_output_name(src: Path, tgt_code: str) -> Path:
#     """
#     Construit le chemin de sortie HTML dans 'translated/' :
#     base = tout avant le premier '_' ; sortie = translated/<base>_<suffix>.<ext>
#     """
#     base = src.stem.split("_", 1)[0]
#     suffix = CODE_TO_SUFFIX.get(tgt_code.lower(), "xx")
#     out_root = (src.parent if src.is_file() else src) / "translated"
#     out_root.mkdir(parents=True, exist_ok=True)
#     return out_root / "{}_{}{}".format(base, suffix, src.suffix)

# Enregistrement des traductions dans le même dossier que la source
def html_output_name(src: Path, tgt_code: str) -> Path:
    """
    Sortie HTML dans le même dossier que le fichier source :
    <base>_<suffix>.html
    """
    base = src.stem.split("_", 1)[0]
    suffix = CODE_TO_SUFFIX.get(tgt_code.lower(), "xx")
    out_dir = src.parent  # 🔥 même dossier que le HTML source
    return out_dir / f"{base}_{suffix}{src.suffix}"

# ----------------------------------------------------------------------
#  HTML PARSERS
# ----------------------------------------------------------------------

class HTMLSegmentCounter(HTMLParser):
    """Compte les segments traduisibles dans un HTML (texte + attributs)."""

    def __init__(self, translate_func):
        super().__init__(convert_charrefs=False)
        self.translate = translate_func
        self.count = 0
        self._stack = []

    def handle_starttag(self, tag, attrs):
        lower = tag.lower()
        self._stack.append(lower)
        if lower in SKIP_TAGS:
            return
        for k, v in attrs:
            if k and v and k.lower() in TRANSLATABLE_ATTRS:
                self.count += 1
                v = self.translate(v)

    def handle_endtag(self, tag):
        if self._stack:
            self._stack.pop()

    def handle_data(self, data):
        if self._stack and self._stack[-1] in SKIP_TAGS:
            return
        if data and data.strip():
            self.count += 1


SAFE_INLINE_TAGS = {"b", "i", "u", "em", "strong", "span", "font"}

class TranslatingHTMLParser(HTMLParser):
    """
    Parser HTML optimisé :
    - Conserve la structure exacte du HTML
    - Traduit uniquement les textes visibles
    - Ignore <head>, <style>, <script>, etc.
    - Compatible Google ligne-par-ligne (pas de concatération)
    """

    def __init__(self, translate_func):
        super().__init__(convert_charrefs=True)
        self.translate = translate_func  # fonction (str)->str
        self.out = []
        self.skip_depth = 0
        self.open_tags = []  # stack for context

    # --------------------------------------------------------------
    # START TAG
    # --------------------------------------------------------------
    def handle_starttag(self, tag, attrs):
        tag_l = tag.lower()

        # skip <script> <style> <meta> <title> etc.
        if tag_l in SKIP_TAGS:
            self.skip_depth += 1

        # reconstruction brute exacte
        attr_str = ""
        if attrs:
            parts = []
            for k, v in attrs:
                if v is None:
                    parts.append(k)
                else:
                    v = v.replace('"', '&quot;')
                    parts.append(f'{k}="{v}"')
            attr_str = " " + " ".join(parts)

        self.out.append(f"<{tag}{attr_str}>")
        self.open_tags.append(tag)

    # --------------------------------------------------------------
    # END TAG
    # --------------------------------------------------------------
    def handle_endtag(self, tag):
        tag_l = tag.lower()

        if tag_l in SKIP_TAGS and self.skip_depth > 0:
            self.skip_depth -= 1

        self.out.append(f"</{tag}>")

        # retire la dernière occurrence du tag dans la pile
        for i in range(len(self.open_tags)-1, -1, -1):
            if self.open_tags[i] == tag:
                self.open_tags.pop(i)
                break

    # --------------------------------------------------------------
    # TEXT NODES
    # --------------------------------------------------------------
    def handle_data(self, data):
        if self.skip_depth > 0:
            # texte dans <style> etc.
            self.out.append(data)
            return

        if not data.strip():
            # espaces / indentation
            self.out.append(data)
            return

        # Traduction ligne par ligne
        translated = self.translate(data)

        self.out.append(translated)

    # --------------------------------------------------------------
    # RAW HTML: comments, DOCTYPE, etc.
    # --------------------------------------------------------------
    def handle_comment(self, data):
        self.out.append(f"<!--{data}-->")

    def handle_decl(self, decl):
        self.out.append(f"<!{decl}>")

    def handle_entityref(self, name):
        self.out.append(f"&{name};")

    def handle_charref(self, name):
        self.out.append(f"&#{name};")

    # --------------------------------------------------------------
    # Rendu final
    # --------------------------------------------------------------
    def get_output(self):
        return "".join(self.out)

def _clean_xml(elem):
    """
    Supprime indentations, espaces et newlines structurels ajoutés
    automatiquement par ElementTree.
    """
    if elem.text:
        elem.text = elem.text.strip("\n\r ")

    elem.tail = ""

    for child in elem:
        _clean_xml(child)

import xml.etree.ElementTree as ET

def indent(elem, level=0):
    """
    Indente proprement un TS **sans casser les <translation>**.
    - N’insère JAMAIS d'espace/retour avant le texte d’un <translation>.
    - Conserve le texte principal dans elem.text.
    - Indente uniquement tail et conteneurs.
    """

    # indentation standard pour XML
    i = "\n" + level * "    "
    j = "\n" + (level + 1) * "    "

    # --- CAS 1 : élément avec enfants --------------------------------
    if len(elem):

        # si l’élément a un texte, mais qu'il est vide ou que ce n’est que des espaces → on laisse vide
        if elem.text and elem.text.strip():
            # texte réel → on ne touche pas
            pass
        else:
            # pas de texte → indenter proprement
            elem.text = j

        # traiter les enfants
        for child in elem:
            indent(child, level + 1)

        # indenter la fin
        if not elem.tail or not elem.tail.strip():
            elem.tail = i

    # --- CAS 2 : élément sans enfants --------------------------------
    else:

        # cas critique : balise <translation>
        if elem.tag == "translation":

            # si texte vide ou absent → on met une chaîne vide, pas d’indentation
            if elem.text is None or not elem.text.strip():
                elem.text = ""

            # tail doit être l’indentation extérieure
            if not elem.tail or not elem.tail.strip():
                elem.tail = i

        else:
            # élément normal : on indente son tail
            if not elem.tail or not elem.tail.strip():
                elem.tail = i

# --------------------------------------------------------------
#  PARSER HTML OPTIMISÉ (BS4 + HTMLParser combinés)
# --------------------------------------------------------------

class OptimizedHTMLParser(HTMLParser):
    """
    Parser HTML professionnel :
    - structure 100% préservée
    - ne traduit que le texte visible
    - ignore <head>, <script>, <style>, <meta>, <link>...
    - safe avec Google (pas de modifications de structure)
    """

    def __init__(self, translate_func):
        super().__init__(convert_charrefs=True)
        self.translate = translate_func
        self.out = []
        self.skip_depth = 0
        self.stack = []

    def error(self, message):
        # Empêche HTMLParser de remonter une exception fatale
        try:
            qgis_log(f"[HTMLParser] Erreur ignorée : {message}", "WARNING")
        except:
            pass
        return

    def handle_starttag(self, tag, attrs):
        try:
            if getattr(self, "worker", None) and self.worker._check_abort():
                return
            tag_l = tag.lower()

            if tag_l in SKIP_TAGS:
                self.skip_depth += 1

            # reconstruction exacte
            txt = f"<{tag}"
            for k, v in attrs:
                if v is None:
                    txt += f" {k}"
                else:
                    v = v.replace('"', '&quot;')
                    txt += f' {k}="{v}"'
            txt += ">"

            self.out.append(txt)
            self.stack.append(tag)

        except Exception as e:
            qgis_log(f"[HTMLParser] starttag crash bloqué : {e}", "WARNING")

    def handle_endtag(self, tag):
        try:
            if getattr(self, "worker", None) and self.worker._check_abort():
                return
            tag_l = tag.lower()
            if tag_l in SKIP_TAGS and self.skip_depth > 0:
                self.skip_depth -= 1

            self.out.append(f"</{tag}>")

            # pop stack
            for i in range(len(self.stack)-1, -1, -1):
                if self.stack[i] == tag:
                    self.stack.pop(i)
                    break
        except Exception as e:
            qgis_log(f"[HTMLParser] starttag crash bloqué : {e}", "WARNING")

    def handle_data(self, data):
        try:
            if getattr(self, "worker", None) and self.worker._check_abort():
                return
            if self.skip_depth > 0:
                # texte non visible
                self.out.append(data)
                return

            if not data.strip():
                self.out.append(data)
                return

            # traduction sécurisée
            translated = self.translate(data)
            self.out.append(translated)
        except Exception as e:
            qgis_log(f"[HTMLParser] starttag crash bloqué : {e}", "WARNING")

    def handle_comment(self, data):
        try:
            self.out.append(f"<!--{data}-->")
        except Exception as e:
            qgis_log(f"[HTMLParser] starttag crash bloqué : {e}", "WARNING")

    def handle_decl(self, decl):
        try:
            self.out.append(f"<!{decl}>")
        except Exception as e:
            qgis_log(f"[HTMLParser] starttag crash bloqué : {e}", "WARNING")

    def handle_entityref(self, name):
        try:
            self.out.append(f"&{name};")
        except Exception as e:
            qgis_log(f"[HTMLParser] starttag crash bloqué : {e}", "WARNING")

    def handle_charref(self, name):
        try:
            self.out.append(f"&#{name};")
        except Exception as e:
            qgis_log(f"[HTMLParser] starttag crash bloqué : {e}", "WARNING")

    def get_html(self):
        try:
            return "".join(self.out)
        except Exception as e:
            qgis_log(f"[HTMLParser] starttag crash bloqué : {e}", "WARNING")


# ----------------------------------------------------------------------
#  WORKER THREAD : traduction TS / HTML
# ----------------------------------------------------------------------
class Worker(QThread):

    @staticmethod
    def tr(text: str) -> str:
        return QCoreApplication.translate("Worker", text)

    # ==== SIGNALS Qt ==================================================
    status = pyqtSignal(str)
    fileProgress = pyqtSignal(int, int)      # done, total
    segProgress = pyqtSignal(str, int)       # "filename|Nom langue", percent
    finished = pyqtSignal(str)

    # ==================================================================
    #  INIT
    # ==================================================================
    def __init__(
        self,
        mode,
        langs_targets,
        i18n_dir: Path = None,
        html_file: Path = None,
        html_dir: Path = None,
        lrelease_path: Path = None,
        settings_root: Path = None,
        parent=None,
    ):
        super().__init__(parent)

        self.mode = mode                   # 'ts' | 'html_file' | 'html_dir'
        self.langs_targets = list(langs_targets)

        self.i18n_dir = i18n_dir
        self.html_file = html_file
        self.html_dir = html_dir
        self.lrelease = lrelease_path
        self.settings_root = settings_root or (
            self.i18n_dir or self.html_dir or Path.cwd()
        )

        self._tm = {}
        self._tm_path = (settings_root or Path.cwd()) / "translation_memory.json"
        self._load_tm()

        self._stats = {
            "google_calls": 0,
            "tm_hits": 0,
        }

        self._all_jobs = []
        self._cur_job_index = -1
        self._interrupted = False
        self._current_log_path = None

    # ==================================================================
    #  INTERRUPT SUPPORT
    # ==================================================================
    def _check_abort(self):
        return self.isInterruptionRequested()

    def _stop_requested(self):
        return self.isInterruptionRequested()

    # ==================================================================
    #  TM LOAD/SAVE
    # ==================================================================
    def _load_tm(self):
        try:
            if self._tm_path.exists():
                self._tm = json.load(self._tm_path.open("r", encoding="utf-8"))
            else:
                self._tm = {}
        except Exception as e:
            qgis_log(f"[TM] Erreur chargement TM : {e}", "WARNING")
            self._tm = {}

    def _save_tm(self):
        try:
            with self._tm_path.open("w", encoding="utf-8") as f:
                json.dump(self._tm, f, ensure_ascii=False, indent=2)
        except Exception as e:
            qgis_log(f"[TM] Erreur sauvegarde TM : {e}", "ERROR")

    # ==================================================================
    #  LOG FILE HELPERS
    # ==================================================================
    def _logfile(self, kind: str, lang: str) -> Path:
        doc = (self.settings_root / "Documents")
        doc.mkdir(parents=True, exist_ok=True)
        name = {
            "ts": f"traductions_ts_{lang}.log",
            "html_file": f"traductions_html_{lang}.log",
            "html_dir": f"traductions_html_folder_{lang}.log",
        }[kind]
        return doc / name

    def _write_log_header(self, kind: str, lang: str):
        p = self._current_log_path = self._logfile(kind, lang)
        with p.open("a", encoding="utf-8") as log:
            log.write(
                "\n" + "=" * 70 + "\n" +
                self.tr("Début ({}) : {}").format(
                    lang,
                    datetime.now().strftime("%d/%m/%Y %H:%M:%S")
                )
                + "\n"
            )

    def _write_log_footer(self, kind: str, lang: str, files_done: int, duration_sec: float):
        with self._logfile(kind, lang).open("a", encoding="utf-8") as log:
            log.write(
                "\n" + "-" * 44 + "\n" +
                self.tr("Résumé : {} fichier(s) traduit(s) en {}").format(
                    files_done, CODE_TO_NAME.get(lang.lower(), lang)
                )
                + "\n" +
                self.tr("Durée totale : {}s").format(int(duration_sec)) +
                "\n" + "-" * 44 + "\n"
            )

    def _log_interruption(self, remaining_jobs, translated_count):
        if not self._current_log_path:
            return

        remaining_files = len(remaining_jobs)
        remaining_langs = []
        seen = set()

        for lang, _p, _k in remaining_jobs:
            name = CODE_TO_NAME.get(lang.lower(), lang)
            if name not in seen:
                remaining_langs.append(name)
                seen.add(name)

        with self._current_log_path.open("a", encoding="utf-8") as log:
            log.write(
                self.tr("[INTERRUPTION UTILISATEUR] Traduction interrompue à {}")
                .format(datetime.now().strftime("%d/%m/%Y %H:%M:%S"))
                + "\n"
            )
            log.write(
                self.tr("Traductions restantes :") + "\n"
                + self.tr(" - Fichiers non traités : {}").format(remaining_files) + "\n"
                + self.tr(" - Langues restantes : {}")
                    .format(", ".join(remaining_langs) if remaining_langs else "—")
                    + "\n"
                + self.tr("Résumé : {} fichier(s) déjà traduit(s) avant interruption.")
                    .format(translated_count)
                + "\n" + "-" * 44 + "\n"
            )

    # ==================================================================
    #  GOOGLE TRANSLATE (QT)
    # ==================================================================
    def _google_translate_qt(self, source: str, target_lang: str):
        if self._stop_requested():
            self._interrupted = True
            return source

        if not source.strip():
            return ""

        # TM
        norm = self._normalize_for_tm(source)
        key = (norm, target_lang)
        if key in self._tm:
            self._stats["tm_hits"] += 1
            return self._tm[key]

        # URL
        url = (
            "https://translate.googleapis.com/translate_a/single?"
            "client=gtx&sl=auto&tl={}&dt=t&q={}"
        ).format(target_lang, QUrl.toPercentEncoding(source).data().decode())

        nam = QgsNetworkAccessManager.instance()
        req = QNetworkRequest(QUrl(url))

        if self._stop_requested():
            self._interrupted = True
            return source

        reply = nam.get(req)

        loop = QEventLoop()
        reply.finished.connect(loop.quit)

        timer = QTimer()
        timer.timeout.connect(loop.quit)
        timer.setSingleShot(True)
        timer.start(8000)

        loop.exec_()

        if self._stop_requested():
            self._interrupted = True
            return source

        self._stats["google_calls"] += 1

        if reply.error() != QNetworkReply.NoError or not timer.isActive():
            reply.deleteLater()
            return source

        try:
            data = bytes(reply.readAll())
        except Exception:
            reply.deleteLater()
            return source

        reply.deleteLater()

        if not data or len(data) < 5:
            return source

        try:
            arr = json.loads(data)
            if isinstance(arr, list) and arr and isinstance(arr[0], list):
                raw = arr[0][0][0]
            else:
                raw = source
        except Exception:
            raw = source

        def fix_case(src, dst):
            if src and src[0].isupper() and dst and dst[0].islower():
                dst = dst[0].upper() + dst[1:]
            if src == src.upper():
                return dst.upper()
            return dst

        corrected = fix_case(source, raw)
        corrected = self._fix_placeholders(source, corrected)
        self._tm[key] = corrected

        return corrected

    # ==================================================================
    # BUILD JOB LIST
    # ==================================================================
    def _select_source_ts(self, items):
        for ts, suff in items:
            if suff == "":
                return ts
        for ts, suff in items:
            if suff == "en":
                return ts
        for ts, suff in items:
            if suff == "fr":
                return ts
        return items[0][0]

    def _build_jobs(self):
        jobs = []

        # TS --------------------------------------------------------------
        if self.mode == "ts" and self.i18n_dir:
            ts_files = sorted(self.i18n_dir.glob("*.ts"))

            pivots = []
            for ts in ts_files:
                base, suff = ts_base_and_suffix(ts.stem)
                if suff == "":
                    pivots.append((base, ts))

            if not pivots:
                qgis_log(self.tr("Aucun fichier pivot .ts trouvé."), "ERROR")
                self._all_jobs = []
                return

            for base, pivot_ts in pivots:
                pivot_text = pivot_ts.read_text(encoding="utf-8")
                for lang in self.langs_targets:
                    if not lang:
                        continue
                    suffix = CODE_TO_SUFFIX.get(lang.lower(), "")
                    if not suffix:
                        continue

                    target_ts = pivot_ts.parent / f"{base}_{suffix}.ts"

                    if not target_ts.exists():
                        try:
                            target_ts.write_text(pivot_text, encoding="utf-8")
                            qgis_log(self.tr("Création : {}").format(target_ts), "INFO")
                        except Exception as e:
                            qgis_log(
                                self.tr("TS ERROR: copie pivot → {} : {}")
                                .format(target_ts, e),
                                "ERROR"
                            )
                            continue

                    jobs.append((lang, target_ts, "ts"))

        # HTML file -------------------------------------------------------
        elif self.mode == "html_file" and self.html_file:
            for lang in self.langs_targets:
                jobs.append((lang, self.html_file, "html"))

        # HTML dir --------------------------------------------------------
        elif self.mode == "html_dir" and self.html_dir:
            html_files = list(self.html_dir.rglob("*.html")) + \
                         list(self.html_dir.rglob("*.htm"))
            for lang in self.langs_targets:
                for f in sorted(html_files):
                    jobs.append((lang, f, "html"))

        self._all_jobs = jobs

    # ==================================================================
    #  DISPATCHER (Qt5/Qt6-safe)
    # ==================================================================
    def run(self):
        # expose abort helper
        QThread.currentThread()._check_abort = self._stop_requested

        if self.mode == "ts":
            self._run_ts()
        elif self.mode == "html_file":
            self._run_html_file()
        elif self.mode == "html_dir":
            self._run_html_dir()
        else:
            self.finished.emit(self.tr("Mode inconnu"))

    # ==================================================================
    #  PUBLIC MODES
    # ==================================================================
    def _run_ts(self):
        self._run_common()

    def _run_html_file(self):
        self._run_common()

    def _run_html_dir(self):
        self._run_common()

    # ==================================================================
    #  COMMON IMPLEMENTATION (former run())
    # ==================================================================
    def _run_common(self):
        self._build_jobs()
        total_files = len(self._all_jobs)
        done_files = 0

        files_done_for_summary = {}
        start_total = time.time()

        if not total_files:
            self.finished.emit(self.tr("Aucun travail à effectuer."))
            return

        for idx, (lang, path, kind) in enumerate(self._all_jobs):
            if self._stop_requested():
                self._interrupted = True
                break

            self._cur_job_index = idx

            if kind == "ts":
                kind_key = "ts"
            elif self.mode == "html_file":
                kind_key = "html_file"
            else:
                kind_key = "html_dir"

            if files_done_for_summary.get(lang) is None:
                files_done_for_summary[lang] = 0
                self._write_log_header(kind_key, lang)

            done_files += 1
            self.fileProgress.emit(done_files, total_files)

            display_lang = CODE_TO_NAME.get(lang.lower(), lang)
            self.segProgress.emit(f"{path.name}|{display_lang}", 0)

            try:
                if kind == "ts":
                    self._process_ts_file(path, lang)
                else:
                    self._process_html_file(path, lang)

                files_done_for_summary[lang] += 1

            except Exception as e:
                qgis_log(
                    self.tr("Erreur lors du traitement de {} ({}) : {}")
                    .format(path, lang, e),
                    "ERROR"
                )
                continue

        # FOOTERS --------------------------------------------------------
        for lang, count in files_done_for_summary.items():
            if self.mode == "ts":
                kind_key = "ts"
            elif self.mode == "html_file":
                kind_key = "html_file"
            else:
                kind_key = "html_dir"

            self._write_log_footer(
                kind_key, lang, count,
                time.time() - start_total
            )

        # INTERRUPTION ----------------------------------------------------
        if self._interrupted:
            remaining = self._all_jobs[self._cur_job_index + 1:]
            translated_count = sum(files_done_for_summary.values())
            self._log_interruption(remaining, translated_count)
            self.finished.emit(self.tr("Interruption utilisateur"))
            return

        # FIN NORMALE ----------------------------------------------------
        self.finished.emit(self.tr("Terminé pour toutes les langues."))

    # ==================================================================
    #  NORMALISATION TM
    # ==================================================================
    def _normalize_for_tm(self, text: str):
        return "\n".join(
            line.rstrip()
            for line in text.strip().split("\n")
        )

    # ==================================================================
    #  PLACEHOLDER FIX
    # ==================================================================
    def _fix_placeholders(self, src: str, dst: str) -> str:
        src_ph = PLACEHOLDER_RE.findall(src)
        if not src_ph:
            return dst

        out = dst or ""

        def fix_percent(text: str):
            return re.sub(r"\b(\d+)\s*%", r"%\1", text)

        out = fix_percent(out)

        for ph in src_ph:
            if ph not in out:
                if out and not out.endswith(" "):
                    out += " "
                out += ph

        return out

    # ==================================================================
    #  MULTILINE TS TRANSLATION
    # ==================================================================
    def _translate_multiline_ts(self, text: str, tgt_google: str):
        if not text.strip():
            return text

        lines = text.split("\n")
        out = []

        for ln in lines:
            if self._stop_requested():
                self._interrupted = True
                out.append(ln)
                continue

            raw = ln.strip()
            if not raw:
                out.append(ln)
                continue

            if re.fullmatch(r"[{}%0-9\s\.:,\-]+", raw):
                out.append(ln)
                continue

            if len(raw) <= 2:
                out.append(ln)
                continue

            try:
                tr_line = google_translate_multiline(raw, tgt_google)
            except Exception:
                tr_line = raw

            prefix = ln[:len(ln) - len(ln.lstrip(" "))]
            out.append(prefix + tr_line)

        return "\n".join(out)

    # ==================================================================
    #  TS PROCESSING
    # ==================================================================
    def _apply_plural_translation(self, src: str, trans_elem: ET.Element, tgt_google: str):
        nums = trans_elem.findall("numerusform")
        if not nums:
            return False

        try:
            base_tr = self._translate_multiline_ts(src, tgt_google)
        except Exception:
            base_tr = src

        for idx, num in enumerate(nums):
            form_src = src
            hint = f" [plural_form_{idx + 1}]"
            if "%n" not in src:
                form_src = src + hint

            try:
                tr_line = self._translate_multiline_ts(form_src, tgt_google)
            except Exception:
                tr_line = base_tr

            tr_line = tr_line.replace(hint, "").strip()
            tr_line = self._fix_placeholders(src, tr_line)

            if "%n" in src and "%n" not in tr_line:
                tr_line += " %n"

            num.text = tr_line

        if "type" in trans_elem.attrib:
            del trans_elem.attrib["type"]

        return True

    def _process_ts_file(self, ts_path: Path, lang: str):
        t_file_start = time.time()
        google_before = self._stats["google_calls"]
        tm_before = self._stats["tm_hits"]

        # --------------------------------------------------
        # Parse TS
        # --------------------------------------------------
        try:
            tree = ET.parse(ts_path)
        except Exception as e:
            qgis_log(f"TS ERROR: {ts_path} : {e}", "ERROR")
            return

        root = tree.getroot()

        # --------------------------------------------------
        # Langues TS (déterministes, sans heuristique)
        # --------------------------------------------------
        src_lang = (root.attrib.get("sourcelanguage") or "").lower()
        tgt_lang = (root.attrib.get("language") or "").lower()
        tgt_google = CODE_TO_GOOGLE.get(lang.lower())

        # Pivot = pas de traduction UNIQUEMENT si suffixe vide
        if ts_path.stem == ts_path.stem.split("_")[0]:
            qgis_log(
                self.tr("Fichier ignoré : pivot détecté ({0}). "
                    "Ce fichier est un pivot et ne doit pas être traduit."
                ).format(src_lang),
                "WARNING"
            )
            return

        # --------------------------------------------------
        # Messages
        # --------------------------------------------------
        messages = [
            m
            for c in root.findall("context")
            for m in c.findall("message")
        ]

        total = max(1, len(messages))
        done = 0
        display_lang = CODE_TO_NAME.get(lang.lower(), lang)
        local_msg_count = 0

        # --------------------------------------------------
        # Boucle principale
        # --------------------------------------------------
        for context in root.findall("context"):
            if self._check_abort():
                return

            for message in context.findall("message"):
                if self._check_abort():
                    return

                local_msg_count += 1
                done += 1
                pc = int(100 * done / total)
                self.segProgress.emit(f"{ts_path.name}|{display_lang}", pc)

                source_elem = message.find("source")
                if source_elem is None:
                    continue

                src = get_full_text(source_elem).strip()
                if not src:
                    continue

                trans_elem = message.find("translation")
                if trans_elem is None:
                    trans_elem = ET.SubElement(message, "translation")
                    trans_elem.set("type", "unfinished")

                old = get_full_text(trans_elem).strip()
                # unfinished = (trans_elem.get("type") == "unfinished")
                unfinished = (
                        trans_elem.get("type") == "unfinished"
                        or not trans_elem.text
                        or trans_elem.text.strip() == ""
                )

                # --------------------------------------------------
                # DÉCISION CANONIQUE QT (sans heuristique)
                # --------------------------------------------------
                should_translate = (
                        not old  # pas de traduction
                        or unfinished  # explicitement unfinished
                        or old == src  # identique à la source
                )

                if not should_translate:
                    continue

                # --------------------------------------------------
                # Pluriels
                # --------------------------------------------------
                handled = False
                try:
                    handled = self._apply_plural_translation(
                        src, trans_elem, tgt_google
                    )
                except Exception as e:
                    qgis_log(f"[TS pluriel warn] {ts_path}: {e}", "WARNING")

                if handled:
                    continue

                # --------------------------------------------------
                # Traduction
                # --------------------------------------------------
                try:
                    new = self._translate_multiline_ts(src, tgt_google)
                except Exception:
                    new = src

                if not new.strip():
                    new = old or src

                # Nettoyage enfants hors pluriels
                for child in list(trans_elem):
                    if child.tag != "numerusform":
                        trans_elem.remove(child)

                trans_elem.text = new

                if "type" in trans_elem.attrib:
                    del trans_elem.attrib["type"]

        # --------------------------------------------------
        # Sauvegarde TS
        # --------------------------------------------------
        _clean_xml(root)
        indent(root)

        try:
            tree.write(str(ts_path), encoding="utf-8", xml_declaration=True)
            qgis_log(self.tr("TS enregistré : {}").format(ts_path))
        except Exception as e:
            qgis_log(f"[TS save error] {ts_path}: {e}", "ERROR")

        # --------------------------------------------------
        # Profilage
        # --------------------------------------------------
        google_after = self._stats["google_calls"]
        tm_after = self._stats["tm_hits"]
        dt_file = time.time() - t_file_start

        qgis_log(
            self.tr("[Profilage TS] {} : {} messages, {} requêtes Google, {} hits mémoire, {}s")
            .format(
                ts_path.name,
                local_msg_count,
                google_after - google_before,
                tm_after - tm_before,
                int(dt_file),
            ),
            "INFO"
        )

        # --------------------------------------------------
        # Compilation QM
        # --------------------------------------------------
        if self.lrelease and self.lrelease.exists():
            qm = ts_path.with_suffix(".qm")

            startupinfo = subprocess.STARTUPINFO()
            startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
            creationflags = subprocess.CREATE_NO_WINDOW

            try:
                subprocess.run(
                    [str(self.lrelease), str(ts_path), "-qm", str(qm)],
                    check=True,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.PIPE,
                    startupinfo=startupinfo,
                    creationflags=creationflags,
                )
                qgis_log(self.tr("QM généré : {}").format(qm))
            except Exception as e:
                qgis_log(f"TS ERROR lrelease {ts_path}: {e}", "ERROR")

    # ==================================================================
    # HTML PROCESS
    # ==================================================================
    def _process_html_file(self, src_file: Path, target_code: str):
        self._tm = {}

        display_lang = CODE_TO_NAME.get(target_code.lower(), target_code)
        out_file = html_output_name(src_file, target_code)

        try:
            txt = src_file.read_text(encoding="utf-8", errors="ignore")
        except Exception as e:
            qgis_log(f"[HTML] ERREUR lecture {src_file} : {e}", "ERROR")
            return

        cleaned_html = txt

        try:
            import html5lib
            parser = "html5lib"
        except Exception:
            parser = "html"

        try:
            soup = BeautifulSoup(txt, parser)
        except Exception:
            soup = BeautifulSoup(txt, "html")

        try:
            for bad in soup(["script", "style", "meta", "link", "noscript"]):
                bad.decompose()

            from .vendor.bs4 import Comment
            for c in soup.find_all(string=lambda t: isinstance(t, Comment)):
                c.extract()

            cleaned_html = str(soup)
        except Exception as e:
            qgis_log(f"[HTML][BS4] Nettoyage échoué : {e}", "WARNING")

        try:
            counter = HTMLSegmentCounter()
            counter.feed(cleaned_html)
            total_segments = max(1, counter.count)
        except Exception:
            total_segments = 1

        segments_done = 0

        def _tr(text):
            nonlocal segments_done, total_segments
            if self._stop_requested():
                self._interrupted = True
                return text

            try:
                out = google_translate_multiline(
                    text,
                    CODE_TO_GOOGLE[target_code]
                )
            except Exception:
                out = text

            segments_done += 1
            pc = int(100 * segments_done / total_segments)
            self.segProgress.emit(f"{src_file.name}|{display_lang}", pc)
            return out

        parser = OptimizedHTMLParser(_tr)
        try:
            parser.feed(cleaned_html)
            html_out = parser.get_html()
            self.segProgress.emit(f"{src_file.name}|{display_lang}", 100)
        except Exception as e:
            qgis_log(f"[HTML] ERREUR parser {src_file} : {e}", "ERROR")
            return

        try:
            out_file.write_text(html_out, encoding="utf-8")
            qgis_log(self.tr("[HTML] Enregistré : {}").format(out_file))
        except Exception as e:
            qgis_log(f"[HTML] ERREUR écriture {out_file} : {e}", "ERROR")


# ----------------------------------------------------------------------
#  WORKER & DIALOG POUR LES TESTS (TS / HTML)
# ----------------------------------------------------------------------

from .translation_tester import run_ts_test, run_html_test, add_log_listener, remove_log_listener

# ----------------------------------------------------------------------
#  TS — Détection des chaînes unfinished
# ----------------------------------------------------------------------

def detect_unfinished_translations(ts_path: Path):
    """
    Analyse un fichier TS et retourne la liste des messages
    dont la traduction est unfinished ou vide.
    """
    issues = []

    try:
        tree = ET.parse(ts_path)
        root = tree.getroot()
    except Exception as e:
        return [f"❌ Impossible d’analyser le TS : {e}"]

    for ctx in root.findall("context"):
        ctx_name = ctx.findtext("name", "@unknown")

        for msg in ctx.findall("message"):
            src = (msg.findtext("source") or "").strip()
            tr = msg.find("translation")

            if tr is None:
                issues.append(
                    f"⚠ Traduction absente — context: {ctx_name} / source: \"{src}\""
                )
                continue

            txt = (tr.text or "").strip()
            unfinished = tr.get("type") == "unfinished"

            if unfinished or not txt:
                issues.append(
                    f"⚠ Traduction unfinished — context: {ctx_name} / source: \"{src}\""
                )

    return issues


class TestWorker(QThread):
    progressChanged = pyqtSignal(int)
    logLine = pyqtSignal(str)
    finishedOk = pyqtSignal()

    def __init__(self, mode, src_path: Path, dst_path: Path, parent=None):
        super().__init__(parent)
        self.mode = mode  # 'ts' ou 'html'
        self.src_path = src_path
        self.dst_path = dst_path

    def run(self):
        def _cb_log(msg: str):
            self.logLine.emit(msg)

        add_log_listener(_cb_log)

        try:
            if self.mode == "ts":
                # Test standard existant
                run_ts_test(
                    self.src_path,
                    self.dst_path,
                    progress_cb=self.progressChanged.emit
                )

                # 🔥 NOUVEAU : contrôle unfinished
                issues = detect_unfinished_translations(self.dst_path)

                if issues:
                    self.logLine.emit(
                        f"⚠ Traduction incomplète : {len(issues)} chaîne(s) unfinished détectée(s)"
                    )
                    for line in issues:
                        self.logLine.emit(line)
                else:
                    self.logLine.emit("✓ Aucune chaîne unfinished détectée")

            else:
                run_html_test(
                    self.src_path,
                    self.dst_path,
                    progress_cb=self.progressChanged.emit
                )

        finally:
            remove_log_listener(_cb_log)

        self.finishedOk.emit()


class TestDialog(QDialog):
    """
    Fenêtre ultime d’analyse de traduction :
      - Colorisation intelligente
      - Filtrage hiérarchique
      - Recherche texte
      - Surlignage avancé
      - QThread non bloquant
    """

    @staticmethod
    def tr(text: str) -> str:
        return QCoreApplication.translate("TestDialog", text)

    COLOR_ERROR = "#ff4c4c"  # rouge
    COLOR_WARNING = "#ff9d29"  # orange
    COLOR_INFO = "#4ea3ff"  # bleu clair
    COLOR_SUCCESS = "#55c45f"  # vert
    # COLOR_DEFAULT = "#e0e0e0"  # gris clair
    COLOR_DEFAULT = "#000000"  # noir

    def __init__(self, mode, src_path: Path, dst_path: Path, parent=None):
        super().__init__(parent)
        self.setWindowTitle(self.tr("Analyse avancée des traductions"))
        self.resize(800, 400)

        self.mode = mode
        self.all_lines = []
        self.filtered_lines = []  # pour re-render

        layout = QVBoxLayout(self)

        # ----------------------------------------------
        # Bandeau titre
        # ----------------------------------------------
        title = QLabel(
            f"<b>{self.tr('Analyse en cours…')}</b><br>"
            f"{self.tr('Mode')} : <i>{mode.upper()}</i><br>"
            f"{self.tr('Source')} : <code>{src_path}</code><br>"
            f"{self.tr('Cible')} : <code>{dst_path}</code>"
        )
        title.setWordWrap(True)
        layout.addWidget(title)

        # ----------------------------------------------
        # Barre de progression
        # ----------------------------------------------
        self.progress = QProgressBar()
        self.progress.setRange(0, 100)
        layout.addWidget(self.progress)

        # ----------------------------------------------
        # Barre d’outils (filtres)
        # ----------------------------------------------
        toolbar = QHBoxLayout()

        self.btn_errors = QPushButton("❗ " + self.tr("Erreurs"))
        self.btn_errors.setCheckable(True)
        self.btn_errors.clicked.connect(self._apply_filters)
        toolbar.addWidget(self.btn_errors)

        self.chk_keywords = QCheckBox("🎯 " + self.tr("Surligner éléments TS"))
        self.chk_keywords.setToolTip(self.tr("Placeholders, pluriels, %n, %1, etc."))
        self.chk_keywords.stateChanged.connect(self._apply_filters)
        toolbar.addWidget(self.chk_keywords)

        self.chk_html = QCheckBox("🔍 " + self.tr("Surligner différences HTML"))
        self.chk_html.stateChanged.connect(self._apply_filters)
        toolbar.addWidget(self.chk_html)

        toolbar.addWidget(QLabel(self.tr("Filtre texte :")))
        self.search_edit = QLineEdit()
        self.search_edit.setPlaceholderText(self.tr("Rechercher…"))
        self.search_edit.textChanged.connect(self._apply_filters)
        toolbar.addWidget(self.search_edit)

        btn_reset = QPushButton(self.tr("Réinitialiser"))
        btn_reset.clicked.connect(self._reset_filters)
        toolbar.addWidget(btn_reset)

        layout.addLayout(toolbar)

        # ----------------------------------------------
        # Zone texte (HTML riche)
        # ----------------------------------------------
        self.text = QTextEdit()
        self.text.setReadOnly(True)
        layout.addWidget(self.text, 1)

        # ----------------------------------------------
        # Bouton fermer
        # ----------------------------------------------
        self.btn_close = QDialogButtonBox(QDialogButtonBox.Close)
        self.btn_close.button(QDialogButtonBox.Close).setEnabled(False)
        self.btn_close.rejected.connect(self.reject)
        layout.addWidget(self.btn_close)

        # ----------------------------------------------
        # Worker (asynchrone)
        # ----------------------------------------------
        self.worker = TestWorker(mode, src_path, dst_path, self)
        self.worker.progressChanged.connect(self.progress.setValue)
        self.worker.logLine.connect(self._receive_log)
        self.worker.finishedOk.connect(self._end_of_test)
        self.worker.start()

    # ==================================================================
    # RÉCEPTION DES LOGS EN DIRECT
    # ==================================================================

    def _receive_log(self, line: str):
        self.all_lines.append(line)
        self._apply_filters(live=True)

    # ==================================================================
    # COLORISATION
    # ==================================================================

    def _colorize(self, line: str) -> str:
        """Retourne la ligne HTML colorisée selon le contenu."""
        l = line.lower()

        # Erreurs
        if "erreur" in l or "❌" in l:
            color = self.COLOR_ERROR

        # Avertissements
        elif "⚠" in l or "warning" in l:
            color = self.COLOR_WARNING

        # Succès
        elif "✓" in l or "👍" in l or "aucun problème" in l:
            color = self.COLOR_SUCCESS

        # Infos importantes
        elif any(k in l for k in ["placeholder", "%1", "%2", "%n", "plur", "diff"]):
            color = self.COLOR_INFO

        else:
            color = self.COLOR_DEFAULT

        # Surlignage TS avancé
        if self.chk_keywords.isChecked():
            line = (
                line.replace("%n", "<b style='color:#bb22ff;'>%n</b>")
                    .replace("%1", "<b style='color:#bb22ff;'>%1</b>")
                    .replace("%2", "<b style='color:#bb22ff;'>%2</b>")
                    .replace("pluriel", "<b style='color:#aa00aa;'>pluriel</b>")
            )

        # Surlignage HTML avancé
        if self.chk_html.isChecked() and "diff" in l:
            line = f"<span style='background:#ffeeaa;'>{line}</span>"

        return f"<span style='color:{color};'>{line}</span>"

    # ==================================================================
    # FILTRAGE
    # ==================================================================

    def _apply_filters(self, live=False):
        txt = self.search_edit.text().lower()
        only_errors = self.btn_errors.isChecked()

        result = []
        for line in self.all_lines:
            l = line.lower()

            if only_errors and not ("erreur" in l or "❌" in l or "⚠" in l):
                continue

            if txt and txt not in l:
                continue

            result.append(line)

        # stockage
        self.filtered_lines = result

        # mise à jour UI
        if live:
            # append la dernière ligne filtrée
            if result:
                html = self._colorize(result[-1])
                self.text.append(html)
            return

        # ré-écriture complète
        self.text.clear()
        for line in result:
            self.text.append(self._colorize(line))

    def _reset_filters(self):
        self.btn_errors.setChecked(False)
        self.chk_keywords.setChecked(False)
        self.chk_html.setChecked(False)
        self.search_edit.clear()
        self._apply_filters()

    # ==================================================================
    # FIN DU TEST
    # ==================================================================

    def _end_of_test(self):
        self.btn_close.button(QDialogButtonBox.Close).setEnabled(True)
        self.progress.setValue(100)
        self.setWindowTitle(self.tr("Analyse terminée"))


def debug_state(gui, label):
    """Log lisible de l'état interne du plugin."""
    try:
        msg = (
            f"[STATE] {label}\n"
            f"   is_running     = {gui.state.is_running}\n"
            f"   last_mode       = {gui.last_mode}\n"
            f"   last_langs      = {gui.last_langs}\n"
            f"   last_html_file  = {gui.last_html_file}\n"
            f"   last_html_dir   = {gui.last_html_dir}"
        )
        qgis_log(msg, "DEBUG")
    except Exception as e:
        qgis_log(f"[STATE] logging failed: {e}", "WARNING")

def _log_state(msg):
    QgsMessageLog.logMessage(msg, "PluginTranslator", Qgis.Info)

class DummyWorker(QObject):
    """
    Worker factice utilisé pour que le testeur fonctionne même lorsque
    le worker réel a été supprimé après la traduction.

    Inclus :
    - signaux Qt pour compatibilité totale
    - vérification automatique de l’intégrité de l’état
    - logging détaillé
    """

    # Signaux Qt (nécessaires pour PyCharm et Qt)
    finished = pyqtSignal()
    progress = pyqtSignal(int)
    log = pyqtSignal(str)

    def __init__(self, mode, langs_targets,
                 html_file=None, html_dir=None, parent=None):
        super().__init__(parent)

        self.mode = mode
        self.langs_targets = langs_targets or []
        self.html_file = html_file
        self.html_dir = html_dir

        # Flag pour debug
        self._dummy = True

        # ---- AUTO-DEBUG ----
        self._self_check()

    # -----------------------------------------------------------
    # AUTO-VALIDATION & LOGGING
    # -----------------------------------------------------------
    def _self_check(self):
        """Analyse l’état et loggue les incohérences éventuelles."""

        _log_state("=== DummyWorker créé ===")
        _log_state(f" mode = {self.mode}")
        _log_state(f" langs_targets = {self.langs_targets}")
        _log_state(f" html_file = {self.html_file}")
        _log_state(f" html_dir = {self.html_dir}")

        # Vérifications intelligentes par mode
        if self.mode == "ts":
            # Aucun fichier attendu, pas d'avertissement
            return

        if self.mode == "html_file":
            if self.html_file is None:
                _log_state("[DummyWorker WARNING] html_file attendu mais None reçu")
            else:
                if not self.html_file.exists():
                    _log_state(f"[DummyWorker WARNING] html_file inexistant : {self.html_file}")

        if self.mode == "html_dir":
            if self.html_dir is None:
                _log_state("[DummyWorker WARNING] html_dir attendu mais None reçu")
            else:
                if not self.html_dir.exists():
                    _log_state(f"[DummyWorker WARNING] html_dir inexistant : {self.html_dir}")

        # Vérification langues
        if not self.langs_targets:
            _log_state("[DummyWorker WARNING] Aucune langue détectée dans langs_targets")


# ----------------------------------------------------------------------
#  UI PRINCIPALE
# ----------------------------------------------------------------------

class StartGUI(QDialog):

    @staticmethod
    def tr(text: str) -> str:
        return QCoreApplication.translate("StartGUI", text)

    def __init__(self, parent=None, html_enabled=False, **kwargs):
        super().__init__(parent)

        self.setAttribute(Qt.WA_DeleteOnClose)
        # StateManager centralisé
        self.state = TranslationState()

        # chemins de lrelease fournis par le plugin OU détectés localement
        self.lrelease_path = kwargs.get("lrelease_path") or detect_lrelease()

        self.setWindowTitle(self.tr("Google Traduction .ts / HTML (multilingue)"))
        self.setWindowModality(Qt.ApplicationModal)
        self.setWindowFlags(Qt.Window | Qt.WindowStaysOnTopHint)
        self.setFixedSize(QSize(820, 520))
        self.settings = QSettings(APP_SETTINGS_ORG, APP_SETTINGS_APP)

        # Combo des modes
        self.mode_combo = QComboBox()
        self.mode_combo.addItems(
            [
                self.tr("Traduire des fichiers .ts"),
                self.tr("Traduire un fichier HTML"),
                self.tr("Traduire un dossier HTML complet"),
            ]
        )
        self.mode_combo.currentIndexChanged.connect(self._on_mode_changed)

        # Création du layout
        v = QVBoxLayout(self)
        v.setContentsMargins(10, 10, 10, 10)
        v.setSpacing(6)
        v.addWidget(self.mode_combo)

        # Chemin
        self.path_edit = QLineEdit()
        self.path_edit.setPlaceholderText(
            self.tr("Dossier i18n / fichier HTML / dossier HTML…")
        )

        btn_browse = QPushButton(self.tr("📁 Parcourir"))
        btn_browse.clicked.connect(self.choose_path)
        h_path = QHBoxLayout()
        h_path.addWidget(self.path_edit, 1)
        h_path.addWidget(btn_browse)
        v.addLayout(h_path)

        # Label affichant le dossier parent de i18n
        self.parent_folder_label = QLabel(self.tr("Traduction fichier .ts"))
        v.addWidget(self.parent_folder_label)

        self._load_last_path_for_mode(self.mode_combo.currentIndex())

        # Multilingue + liste de langues cibles (cases)
        self.cb_multi = QCheckBox(self.tr("Activer la traduction multilingue"))
        self.cb_multi.stateChanged.connect(self._multi_changed)
        v.addWidget(self.cb_multi)

        # ------------------------------------------------------------------
        #  Sélecteur de langues — Groupes + checkbox persistants
        # ------------------------------------------------------------------

        self._init_lang_tree = True
        self._syncing_lang_tree = False

        self.lang_tree = QTreeWidget()
        self.lang_tree.setHeaderHidden(True)
        self.lang_tree.setRootIsDecorated(True)
        self.lang_tree.setAlternatingRowColors(True)
        self.lang_tree.setSelectionMode(QTreeWidget.NoSelection)

        for zone, langs in LANG_ZONES.items():
            zone_item = QTreeWidgetItem(self.lang_tree)
            zone_item.setText(0, self.tr(zone))
            zone_item.setFlags(
                Qt.ItemIsEnabled |
                Qt.ItemIsUserCheckable |
                Qt.ItemIsAutoTristate
            )
            zone_item.setExpanded(False)

            for code, label in langs.items():
                lang_item = QTreeWidgetItem(zone_item)
                lang_item.setText(0, f"{self.tr(label)} ({code})")
                lang_item.setData(0, Qt.UserRole, code.lower())
                lang_item.setFlags(Qt.ItemIsEnabled | Qt.ItemIsUserCheckable)
                lang_item.setCheckState(0, Qt.Unchecked)

            zone_item.setCheckState(0, Qt.Unchecked)

        v.addWidget(self.lang_tree)

        # ------------------------------------------------------------------
        #  🔕 Restauration silencieuse (QSettings)
        # ------------------------------------------------------------------
        self._restore_lang_settings()

        # ------------------------------------------------------------------
        #  🔗 CONNEXIONS DES SIGNAUX (ICI ET SEULEMENT ICI)
        # ------------------------------------------------------------------
        self.lang_tree.itemPressed.connect(self._on_lang_tree_pressed)
        self.lang_tree.itemChanged.connect(self._on_lang_tree_changed)

        # ------------------------------------------------------------------
        #  🔓 Fin d'initialisation
        # ------------------------------------------------------------------
        self._init_lang_tree = False

        # Auto-détection de langues TS si un dossier est déjà connu
        initial_path = self.path_edit.text().strip()
        if initial_path and self.mode_combo.currentIndex() == 0 and Path(initial_path).exists():
            self._auto_detect_langs_in_ts_folder(Path(initial_path))

        if html_enabled:
            self.mode_combo.setCurrentIndex(1)

        # Barres de progression
        self.file_progress = QProgressBar()
        self.file_progress.setTextVisible(True)
        self.seg_progress = QProgressBar()
        self.seg_progress.setTextVisible(True)
        self._reset_bars()
        v.addWidget(self.file_progress)
        v.addWidget(self.seg_progress)

        # Statut simple
        self.status_label = QLabel(self.tr("Prêt."))
        v.addWidget(self.status_label)

        # Lancer + Quitter
        row_btns = QHBoxLayout()
        self.run_btn = QPushButton(self.tr("▶ Lancer"))
        self.run_btn.clicked.connect(self.launch)
        row_btns.addWidget(self.run_btn)
        self.quit_btn = QPushButton(self.tr("Quitter"))
        self.quit_btn.clicked.connect(self.on_quit_clicked)
        row_btns.addWidget(self.quit_btn)
        v.addLayout(row_btns)

        self.worker = None
        self._closing_safely = False

    def _restore_lang_settings(self):
        saved_langs = set(self.settings.value("langs_selected", [], type=list))
        multi = self.settings.value("langs_multi", False, type=bool)

        # état multi
        self.cb_multi.setChecked(multi)

        root = self.lang_tree.invisibleRootItem()
        for i in range(root.childCount()):
            zone = root.child(i)

            # zone toujours décochée au départ
            zone.setCheckState(0, Qt.Unchecked)

            for j in range(zone.childCount()):
                it = zone.child(j)
                code = it.data(0, Qt.UserRole)
                if code in saved_langs:
                    it.setCheckState(0, Qt.Checked)
                else:
                    it.setCheckState(0, Qt.Unchecked)

    #         # synchronisation manuelle zone
    #         self._sync_zone_state(zone)
    #
    # def _sync_zone_state(self, zone_item):
    #     checked = 0
    #     total = zone_item.childCount()
    #
    #     for i in range(total):
    #         if zone_item.child(i).checkState(0) == Qt.Checked:
    #             checked += 1
    #
    #     if checked == 0:
    #         zone_item.setCheckState(0, Qt.Unchecked)
    #
    #     elif checked == total:
    #         # 🔥 CRUCIAL : ne JAMAIS laisser Checked
    #         zone_item.setCheckState(0, Qt.PartiallyChecked)
    #
    #     else:
    #         zone_item.setCheckState(0, Qt.PartiallyChecked)

    # ------------------------------------------------------------------
    #  Helpers d’affichage
    # ------------------------------------------------------------------

    def _update_parent_folder_label(self):
        """Met à jour le label affichant le dossier parent du dossier i18n."""
        path = self.path_edit.text().strip()
        if not path:
            self.parent_folder_label.setText(self.tr("Traduction fichier —"))
            return

        p = Path(path)

        # Mode TS : dossier i18n
        if self.mode_combo.currentIndex() == 0:
            if p.exists() and p.is_dir():
                parent = p.parent
                self.parent_folder_label.setText(
                    self.tr("Traduction fichier {}.ts").format(parent.name)
                )
            else:
                self.parent_folder_label.setText(self.tr("Traduction fichier —"))
        else:
            # Modes HTML
            self.parent_folder_label.setText(self.tr("Traduction fichier html"))

    def _default_browse_directory(self):
        """
        Dossier initial pour QFileDialog :
        1. Chemin déjà saisi
        2. Dossier i18n
        3. Dossier plugins profil QGIS
        4. Home
        """
        ptxt = self.path_edit.text().strip()
        if ptxt:
            p = Path(ptxt)
            if p.exists():
                return str(p.parent if p.is_file() else p)

        if ptxt:
            p = Path(ptxt)
            if p.exists() and p.is_dir():
                if (p / "i18n").exists():
                    return str(p / "i18n")

        try:
            from qgis.core import QgsApplication
            profile_path = Path(QgsApplication.qgisSettingsDirPath())
            plugin_dir = profile_path / "python" / "plugins"
            if plugin_dir.exists():
                return str(plugin_dir)
        except Exception:
            pass

        return str(Path.home())

    def _reset_bars(self):
        self.file_progress.setMaximum(1)
        self.file_progress.setValue(0)
        self.file_progress.setFormat(
            self.tr("Fichiers en cours de traitement : {} / {}").format(0, 0)
        )

        self.seg_progress.setMaximum(100)
        self.seg_progress.setValue(0)
        self.seg_progress.setFormat(
            self.tr("Traduction de {} en {} : %p%").format("—", "—")
        )

    def _multi_changed(self, _state):
        # 🔕 pendant init : ne rien faire
        if getattr(self, "_init_lang_tree", False):
            return

        if self.cb_multi.isChecked():
            self._save_lang_settings()
            return

        # Mode mono-langue → garder UNE seule langue cochée
        first_checked = None
        root = self.lang_tree.invisibleRootItem()

        for i in range(root.childCount()):
            zone = root.child(i)
            for j in range(zone.childCount()):
                it = zone.child(j)

                if it.checkState(0) == Qt.Checked:
                    if first_checked is None:
                        first_checked = it
                    else:
                        it.setCheckState(0, Qt.Unchecked)

            # # synchroniser la zone après nettoyage
            # self._sync_zone_state(zone)

        self._save_lang_settings()

    def _enforce_single_language(self, changed_item):
        root = self.lang_tree.invisibleRootItem()

        for i in range(root.childCount()):
            zone = root.child(i)
            for j in range(zone.childCount()):
                it = zone.child(j)
                if it is not changed_item and it.checkState(0) == Qt.Checked:
                    it.setCheckState(0, Qt.Unchecked)
                    # self._sync_zone_state(zone)

    def _on_mode_changed(self, idx: int):
        self._load_last_path_for_mode(idx)
        self._update_parent_folder_label()

    def _load_last_path_for_mode(self, idx: int):
        keys = {
            0: "last_path_ts",
            1: "last_path_html",
            2: "last_path_html_folder",
        }
        key = keys.get(idx, "last_path_html")
        self.path_edit.setText(self.settings.value(key, ""))
        self._update_parent_folder_label()

    def _save_last_path_for_mode(self, idx: int, path: str):
        keys = {
            0: "last_path_ts",
            1: "last_path_html",
            2: "last_path_html_folder",
        }
        key = keys.get(idx, "last_path_html")
        self.settings.setValue(key, path)

    # ------------------------------------------------------------------
    #  Choix de chemin
    # ------------------------------------------------------------------

    def choose_path(self):
        start_dir = self._default_browse_directory()
        idx = self.mode_combo.currentIndex()

        screen = QApplication.primaryScreen().availableGeometry()
        W = screen.width() // 2
        H = screen.height() // 2

        # MODE 0 : TS = dossier i18n
        if idx == 0:
            dlg = QFileDialog(self, self.tr("Choisir dossier i18n"))
            dlg.setFileMode(QFileDialog.Directory)
            dlg.setOption(QFileDialog.ShowDirsOnly, True)
            dlg.setOption(QFileDialog.DontUseNativeDialog, True)
            dlg.setDirectory(start_dir)

            dlg.resize(W, H)
            dlg.move(
                screen.center().x() - dlg.width() // 2,
                screen.center().y() - dlg.height() // 2,
            )

            if dlg.exec_():
                d = dlg.selectedFiles()[0]
                self.path_edit.setText(d)
                self._save_last_path_for_mode(idx, d)
                self._auto_detect_langs_in_ts_folder(Path(d))
                self._update_parent_folder_label()
            return

        # MODE 1 : fichier HTML
        if idx == 1:
            dlg = QFileDialog(self, self.tr("Choisir fichier HTML"))
            dlg.setFileMode(QFileDialog.ExistingFile)
            dlg.setNameFilter(self.tr("HTML (*.html *.htm)"))
            dlg.setOption(QFileDialog.DontUseNativeDialog, True)
            dlg.setDirectory(start_dir)

            dlg.resize(W, H)
            dlg.move(
                screen.center().x() - dlg.width() // 2,
                screen.center().y() - dlg.height() // 2,
            )

            if dlg.exec_():
                f = dlg.selectedFiles()[0]
                self.path_edit.setText(f)
                self._save_last_path_for_mode(idx, f)
            return

        # MODE 2 : dossier HTML
        dlg = QFileDialog(self, self.tr("Choisir dossier HTML"))
        dlg.setFileMode(QFileDialog.Directory)
        dlg.setOption(QFileDialog.ShowDirsOnly, True)
        dlg.setOption(QFileDialog.DontUseNativeDialog, True)
        dlg.setDirectory(start_dir)

        dlg.resize(W, H)
        dlg.move(
            screen.center().x() - dlg.width() // 2,
            screen.center().y() - dlg.height() // 2,
        )

        if dlg.exec_():
            d = dlg.selectedFiles()[0]
            self.path_edit.setText(d)
            self._save_last_path_for_mode(idx, d)
            self._update_parent_folder_label()

    # ------------------------------------------------------------------
    #  Détection/validation langues TS
    # ------------------------------------------------------------------

    def _auto_detect_langs_in_ts_folder(self, folder: Path):
        ts_files = sorted(folder.glob("*_??.ts"))
        if not ts_files:
            return

        detected = set(
            re.search(r"_([a-z]{2})\.ts$", ts.name, re.IGNORECASE).group(1).lower()
            for ts in ts_files
            if re.search(r"_([a-z]{2})\.ts$", ts.name, re.IGNORECASE)
        )

        self.cb_multi.setChecked(len(detected) > 1)

        root = self.lang_tree.invisibleRootItem()
        for i in range(root.childCount()):
            zone = root.child(i)
            for j in range(zone.childCount()):
                it = zone.child(j)
                code = it.data(0, Qt.UserRole)
                suff = CODE_TO_SUFFIX.get(code, "")
                it.setCheckState(
                    0,
                    Qt.Checked if suff in detected else Qt.Unchecked
                )

    def _on_lang_tree_pressed(self, item, column):
        if column != 0:
            return

        if self._init_lang_tree or self._syncing_lang_tree:
            return

        # Seulement les ZONES
        if item.childCount() == 0:
            return

        # On mémorise l'intention, SANS agir
        item._zone_toggle_requested = True

    def _on_lang_tree_changed(self, item, column):
        if column != 0:
            return

        if self._init_lang_tree or self._syncing_lang_tree:
            return

        self._syncing_lang_tree = True
        try:
            # ===========================
            # ZONE (action différée)
            # ===========================
            if item.childCount() > 0:
                if not getattr(item, "_zone_toggle_requested", False):
                    return

                del item._zone_toggle_requested

                children = [item.child(i) for i in range(item.childCount())]
                should_check_all = any(c.checkState(0) != Qt.Checked for c in children)
                new_state = Qt.Checked if should_check_all else Qt.Unchecked

                for c in children:
                    c.setCheckState(0, new_state)

                self._save_lang_settings()
                return

            # ===========================
            # LANGUE (libre)
            # ===========================
            if not self.cb_multi.isChecked() and item.checkState(0) == Qt.Checked:
                self._enforce_single_language(item)

            self._save_lang_settings()

        finally:
            self._syncing_lang_tree = False

    def _save_lang_settings(self):
        self.settings.setValue("langs_selected", self._selected_langs())
        self.settings.setValue("langs_multi", self.cb_multi.isChecked())

    # ------------------------------------------------------------------
    #  Lancement des traductions
    # ------------------------------------------------------------------
    def _selected_langs(self):
        langs = []
        root = self.lang_tree.invisibleRootItem()

        for i in range(root.childCount()):
            zone = root.child(i)
            for j in range(zone.childCount()):
                it = zone.child(j)
                if it.checkState(0) == Qt.Checked:
                    langs.append(it.data(0, Qt.UserRole))

        return langs

    def launch(self):
        """
        Lance une traduction en mode :
            0 → TS
            1 → HTML (fichier)
            2 → HTML (dossier)

        Version ultra-robuste pour relancement illimité.
        """

        # ------------------------------------------------------------------
        # 0) Protection anti-lancement multiple
        # ------------------------------------------------------------------
        if self.worker is not None:
            QMessageBox.warning(
                self,
                self.tr("Traduction en cours"),
                self.tr("Une traduction est déjà en cours. Veuillez attendre la fin ou annuler."),
            )
            return

        # ------------------------------------------------------------------
        # 1) Récupération et validation du chemin
        # ------------------------------------------------------------------
        path = self.path_edit.text().strip()
        if not path:
            QMessageBox.warning(self, self.tr("Erreur"), self.tr("Veuillez choisir un dossier/fichier."))
            return

        QApplication.processEvents()

        # ------------------------------------------------------------------
        # 2) Langues sélectionnées
        # ------------------------------------------------------------------
        langs = self._selected_langs()

        if not langs:
            QMessageBox.warning(self, self.tr("Erreur"), self.tr("Veuillez cocher au moins une langue cible."))
            return

        if not self.cb_multi.isChecked() and len(langs) > 1:
            langs = langs[:1]

        # ------------------------------------------------------------------
        # 3) Détection du mode TS / HTML fichier / HTML dossier
        # ------------------------------------------------------------------
        idx = self.mode_combo.currentIndex()
        mode_map = {0: "ts", 1: "html_file", 2: "html_dir"}
        mode = mode_map.get(idx)
        if not mode:
            QMessageBox.critical(self, self.tr("Erreur"), self.tr("Mode invalide."))
            return

        # On mémorise pour le testeur
        self.last_mode = mode
        self.last_langs = langs[:]
        self.last_html_file = None
        self.last_html_dir = None

        # ------------------------------------------------------------------
        # 4) Construction du Worker selon le mode
        # ------------------------------------------------------------------

        # MODE TS ----------------------------------------------------------
        if mode == "ts":
            i18n_dir = Path(path)
            if not i18n_dir.exists():
                QMessageBox.warning(self, self.tr("Erreur"), self.tr("Dossier i18n invalide."))
                return

            # Vérification pivot obligatoire
            ts_files = list(i18n_dir.glob("*.ts"))
            has_pivot = any(ts_base_and_suffix(ts.stem)[1] == "" for ts in ts_files)

            if not has_pivot:
                QMessageBox.critical(
                    self,
                    self.tr("Pivot manquant"),
                    self.tr("Aucun fichier pivot '<base>.ts' trouvé dans ce dossier."),
                )
                return

            # Vérification du lrelease
            if not self.lrelease_path or not self.lrelease_path.exists():
                QMessageBox.critical(
                    self,
                    self.tr("lrelease manquant"),
                    self.tr("Le fichier lrelease.exe est introuvable."),
                )
                return

            # Contexte testeur
            self.last_html_file = None
            self.last_html_dir = None

            self.worker = Worker(
                mode="ts",
                langs_targets=langs,
                i18n_dir=i18n_dir,
                lrelease_path=self.lrelease_path,
                parent=self,
                settings_root=i18n_dir,
            )

        # MODE HTML FICHIER ------------------------------------------------
        elif mode == "html_file":
            html_file = Path(path)
            if not html_file.exists():
                QMessageBox.warning(self, self.tr("Erreur"), self.tr("Fichier HTML introuvable."))
                return

            self.last_html_file = html_file
            self.last_html_dir = None

            self.worker = Worker(
                mode="html_file",
                langs_targets=langs,
                html_file=html_file,
                parent=self,
                settings_root=html_file.parent,
            )

        # MODE HTML DOSSIER ------------------------------------------------
        else:  # html_dir
            html_dir = Path(path)
            if not html_dir.exists():
                QMessageBox.warning(self, self.tr("Erreur"), self.tr("Dossier HTML introuvable."))
                return

            self.last_html_file = None
            self.last_html_dir = html_dir

            self.worker = Worker(
                mode="html_dir",
                langs_targets=langs,
                html_dir=html_dir,
                parent=self,
                settings_root=html_dir,
            )

        # ------------------------------------------------------------------
        # 5) Connexion des signaux Worker → UI
        # ------------------------------------------------------------------
        self.worker.fileProgress.connect(self.on_file_progress)
        self.worker.segProgress.connect(self.on_seg_progress)
        self.worker.finished.connect(self.on_done)

        # ------------------------------------------------------------------
        # 6) Démarrage
        # ------------------------------------------------------------------
        try:
            self.run_btn.setEnabled(False)
            self.status_label.setText(self.tr("Traduction en cours…"))
            self.seg_progress.setValue(0)
            self.file_progress.setValue(0)
        except Exception:
            pass

        self.worker.start()

    # ------------------------------------------------------------------
    #  Callbacks worker
    # ------------------------------------------------------------------

    def on_file_progress(self, done, total):
        self.file_progress.setMaximum(max(1, total))
        self.file_progress.setValue(done)
        self.file_progress.setFormat(
            self.tr("Fichiers en cours de traitement : {} / {}").format(done, total)
        )

    def on_seg_progress(self, payload, percent):
        try:
            fn, langname = payload.split("|", 1)
        except ValueError:
            fn, langname = payload, "—"
        self.seg_progress.setValue(percent)
        self.seg_progress.setFormat(
            self.tr("Traduction de {} en {} : %p%").format(fn, langname)
        )

    def _reset_visual(self):
        self._reset_bars()
        self.status_label.setText(self.tr("Prêt."))
        # <<< AJOUT NÉCESSAIRE >>>
        self.run_btn.setEnabled(True)  # réactivation bouton
        self.quit_btn.setEnabled(True)  # au cas où

    # ------------------------------------------------------------------
    #  Sortie avec question testeur
    # ------------------------------------------------------------------
    def on_done(self, msg: str):
        """
        Fin de traduction (Worker.finished)
        → Réactive l’UI
        → Affiche le message final
        → Pose la question pour lancer le testeur
        → Lance le testeur selon last_mode
        """
        # -------------------------------------------------------------
        # 0) Sécurité absolue : empêcher tout worker zombie
        # -------------------------------------------------------------
        w = self.worker
        self.worker = None

        # -------------------------------------------------------------
        # 1) Réactivation de l’UI
        # -------------------------------------------------------------
        try:
            self.run_btn.setEnabled(True)
            self.status_label.setText(self.tr("Prêt."))
            self.seg_progress.setValue(0)
            self.file_progress.setValue(0)
        except Exception:
            pass  # l’UI n’existe plus (fermeture QGIS)

        # -------------------------------------------------------------
        # 2) Message final (toujours affiché)
        # -------------------------------------------------------------
        try:
            QMessageBox.information(self, self.tr("Terminé"), msg)
        except Exception:
            return  # fenêtre fermée → pas de testeur

        # -------------------------------------------------------------
        # 3) Si aucun contexte → pas de question testeur
        # -------------------------------------------------------------
        if not self.last_mode:
            return

        # -------------------------------------------------------------
        # 4) Question pour lancer le testeur
        # -------------------------------------------------------------
        ret = QMessageBox.question(
            self,
            self.tr("Testeur de traduction"),
            self.tr("Voulez-vous lancer le testeur de conformité pour cette traduction ?"),
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No,
        )

        if ret != QMessageBox.Yes:
            return

        # -------------------------------------------------------------
        # 5) Lancement du testeur selon last_mode
        # -------------------------------------------------------------
        try:
            if self.last_mode == "ts":
                self._run_ts_tester()

            elif self.last_mode == "html_file":
                self._run_html_file_tester()

            elif self.last_mode == "html_dir":
                self._run_html_dir_tester()

            else:
                QMessageBox.warning(
                    self,
                    self.tr("Testeur"),
                    self.tr("Mode inconnu : {}").format(self.last_mode),
                )

        except Exception as e:
            QMessageBox.warning(
                self,
                self.tr("Testeur"),
                self.tr("Erreur lors du lancement du testeur : {}").format(e),
            )

        finally:
            # ---------------------------------------------------------
            # 6) Reset total du contexte (préparation au relancement)
            # ---------------------------------------------------------
            self.last_mode = None
            self.last_langs = []
            self.last_html_file = None
            self.last_html_dir = None

    # ------------------------------------------------------------------
    #  Fonctions dédiées au testeur
    # ------------------------------------------------------------------
    # Version 2

    def _run_ts_tester(self):
        """
        Analyse TS pivot/traduit après traduction.
        Ouvre une fenêtre avec logs + progression.
        """
        i18n_dir = Path(self.path_edit.text()).resolve()
        if not i18n_dir.exists():
            QMessageBox.warning(self, self.tr("Test TS"), self.tr("Dossier i18n invalide."))
            return

        ts_files = sorted(i18n_dir.glob("*.ts"))
        if not ts_files:
            QMessageBox.warning(self, self.tr("Test TS"), self.tr("Aucun fichier .ts trouvé dans ce dossier."))
            return

        items = []
        mapping = {}  # label -> (pivot, translated)

        for ts in ts_files:
            base, suff = ts_base_and_suffix(ts.stem)
            if not suff:
                continue  # c'est un pivot

            pivot = i18n_dir / f"{base}.ts"
            if not pivot.exists():
                continue

            nice = CODE_TO_NAME.get(suff.lower(), suff)
            label = f"{nice} ({suff}) — {ts.name}"
            items.append(label)
            mapping[label] = (pivot, ts)

        if not items:
            QMessageBox.warning(self, self.tr("Test TS"), self.tr("Aucun couple pivot/traduction détecté."))
            return

        label, ok = QInputDialog.getItem(
            self,
            self.tr("Test TS"),
            self.tr("Choisir la langue à tester :"),
            items,
            0,
            False,
        )
        if not ok or label not in mapping:
            return

        pivot, translated = mapping[label]

        dlg = TestDialog("ts", pivot, translated, self)
        dlg.exec_()

    def _run_html_file_tester(self):
        """
        Analyse HTML pour un fichier unique.
        Ouvre une fenêtre avec logs + progression.
        Ne dépend PAS de self.worker.
        """
        # On prend d'abord le dernier fichier utilisé, sinon le champ texte
        src_file = self.last_html_file
        if src_file is None:
            txt = self.path_edit.text().strip()
            if txt:
                src_file = Path(txt).resolve()

        if not src_file or not src_file.exists():
            QMessageBox.warning(self, self.tr("Test HTML"), self.tr("Fichier HTML source introuvable."))
            return

        langs = self.last_langs or []
        if not langs:
            QMessageBox.warning(
                self,
                self.tr("Test HTML"),
                self.tr("Aucune langue n'a été mémorisée pour cette traduction."),
            )
            return

        items = []
        mapping = {}  # label -> (src, translated)

        for lang in langs:
            out_file = html_output_name(src_file, lang)
            if out_file.exists():
                nice = CODE_TO_NAME.get(lang.lower(), lang)
                label = f"{nice} ({lang}) → {out_file.name}"
                items.append(label)
                mapping[label] = (src_file, out_file)

        if not items:
            QMessageBox.warning(self, self.tr("Test HTML"), self.tr("Aucun fichier HTML traduit détecté."))
            return

        label, ok = QInputDialog.getItem(
            self,
            self.tr("Test HTML"),
            self.tr("Choisir la langue à tester :"),
            items,
            0,
            False,
        )
        if not ok or label not in mapping:
            return

        src, translated = mapping[label]

        dlg = TestDialog("html", src, translated, self)
        dlg.exec_()

    def _run_html_dir_tester(self):
        """
        Analyse HTML pour un dossier (un seul couple fichier/langue choisi).
        Ouvre une fenêtre avec logs + progression.
        Ne dépend PAS de self.worker.
        """
        html_dir = self.last_html_dir
        if html_dir is None:
            txt = self.path_edit.text().strip()
            if txt:
                html_dir = Path(txt).resolve()

        if not html_dir or not html_dir.exists():
            QMessageBox.warning(self, self.tr("Test HTML"), self.tr("Dossier HTML introuvable."))
            return

        langs = self.last_langs or []
        if not langs:
            QMessageBox.warning(
                self,
                self.tr("Test HTML"),
                self.tr("Aucune langue n'a été mémorisée pour cette traduction."),
            )
            return

        html_files = list(html_dir.rglob("*.html")) + list(html_dir.rglob("*.htm"))

        items = []
        mapping = {}  # label -> (src, translated)

        for src_file in sorted(html_files):
            # ignorer les fichiers déjà dans 'translated'
            if "translated" in src_file.parts:
                continue

            for lang in langs:
                out_file = html_output_name(src_file, lang)
                if out_file.exists():
                    nice = CODE_TO_NAME.get(lang.lower(), lang)
                    label = f"{src_file.name} → {nice} ({lang})"
                    items.append(label)
                    mapping[label] = (src_file, out_file)

        if not items:
            QMessageBox.warning(
                self,
                self.tr("Test HTML"),
                self.tr("Aucun couple HTML source/traduit détecté dans ce dossier."),
            )
            return

        label, ok = QInputDialog.getItem(
            self,
            self.tr("Test HTML dossier"),
            self.tr("Choisir le fichier et la langue à tester :"),
            items,
            0,
            False,
        )
        if not ok or label not in mapping:
            return

        src, translated = mapping[label]

        dlg = TestDialog("html", src, translated, self)
        dlg.exec_()

    # ------------------------------------------------------------------
    #  Fermeture sécurisée
    # ------------------------------------------------------------------

    def on_quit_clicked(self):
        """
        Fermeture ULTRA-SÛRE, compatible Qt5 / Qt6 :
        - confirmation utilisateur
        - arrêt propre du worker (interruption + wait)
        - purge totale de l'état interne
        - réactivation de l'UI
        - garantie 100% relançable
        """

        # -------------------------------------------------------
        # 1) Confirmation
        # -------------------------------------------------------
        worker_running = (
                self.worker is not None
                and hasattr(self.worker, "isRunning")
                and self.worker.isRunning()
        )

        if worker_running:
            text = self.tr("Une traduction est en cours.\n"
                      "Voulez-vous vraiment interrompre et quitter ?")
        else:
            text = self.tr("Voulez-vous vraiment quitter ?")

        ret = QMessageBox.question(
            self,
            self.tr("Quitter"),
            text,
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No,
        )

        if ret != QMessageBox.Yes:
            return

        # -------------------------------------------------------
        # 2) Arrêt du worker (Qt5/Qt6-safe)
        # -------------------------------------------------------
        if self.worker is not None:
            try:
                if hasattr(self.worker, "requestInterruption"):
                    self.worker.requestInterruption()
            except Exception:
                pass

            try:
                if hasattr(self.worker, "wait"):
                    # 1 seconde max : évite blocages Qt5/Qt6
                    self.worker.wait(1000)
            except Exception:
                pass

            try:
                if hasattr(self.worker, "deleteLater"):
                    self.worker.deleteLater()
            except Exception:
                pass

            self.worker = None

        # -------------------------------------------------------
        # 3) Reset état interne (Qt5/Qt6 compatible)
        # -------------------------------------------------------
        try:
            self.state.reset()
        except Exception:
            pass

        self.last_mode = None
        self.last_langs = []
        self.last_html_file = None
        self.last_html_dir = None

        # -------------------------------------------------------
        # 4) Réinitialise l'UI (Qt5/Qt6 safe)
        # -------------------------------------------------------
        try:
            self.run_btn.setEnabled(True)
            self.quit_btn.setEnabled(True)
        except Exception:
            pass

        try:
            self.status_label.setText(self.tr("Prêt."))
        except Exception:
            pass

        # Progress bars
        try:
            self.file_progress.reset()
            self.seg_progress.reset()
        except Exception:
            # fallback Qt5-only
            try:
                self.file_progress.setValue(0)
                self.seg_progress.setValue(0)
            except Exception:
                pass

        # -------------------------------------------------------
        # 5) Fermeture finale
        # -------------------------------------------------------
        self.close()

    def closeEvent(self, event):
        """
        Même logique que le bouton Quitter.
        """
        if self.worker is None or not isinstance(self.worker, QThread):
            event.accept()
            return

        ret = QMessageBox.question(
            self,
            self.tr("Arrêter la traduction"),
            self.tr("Une traduction est en cours. Voulez-vous l'interrompre ?"),
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No
        )

        if ret != QMessageBox.Yes:
            event.ignore()
            return

        try:
            self.worker.requestInterruption()
        except:
            pass

        self.worker.wait(2000)

        try:
            self.worker.fileProgress.disconnect()
        except:
            pass
        try:
            self.worker.segProgress.disconnect()
        except:
            pass
        try:
            self.worker.finished.disconnect()
        except:
            pass

        try:
            self.worker.deleteLater()
        except:
            pass

        self.worker = None

        self.run_btn.setEnabled(True)
        self.status_label.setText(self.tr("Prêt"))
        self.file_progress.setValue(0)
        self.seg_progress.setValue(0)

        event.accept()

    def _stop_worker_safely(self):
        """
        Arrête le worker de manière sûre :
        - envoie une interruption
        - attend la fin du thread
        - déconnecte les signaux
        """

        # état logique
        self.state.on_worker_finished(interrupted=True)

        if not isinstance(self.worker, QThread):
            # rien à arrêter (ou worker factice : on l'oublie)
            self.worker = None
            return

        try:
            if self.worker.isRunning():
                self.status_label.setText(self.tr("Arrêt du worker en cours…"))
                QApplication.processEvents()

                self.worker.requestInterruption()
                self.worker.wait()

            # Déconnexion des signaux
            try:
                self.worker.fileProgress.disconnect(self.on_file_progress)
            except Exception:
                pass
            try:
                self.worker.segProgress.disconnect(self.on_seg_progress)
            except Exception:
                pass
            try:
                self.worker.finished.disconnect(self.on_done)
            except Exception:
                pass

            self.worker.deleteLater()

        except Exception as e:
            qgis_log(f"[StartGUI] Erreur lors de l'arrêt du worker : {e}", "WARNING")

        self.worker = None

    def _request_close(self):
        """
        Logique de fermeture :
        - si worker en cours : confirmation + arrêt propre + reset UI
        - sinon : confirmation simple
        Retourne True si fermeture autorisée.
        """

        # CAS 1 : worker en cours (vrai QThread uniquement)
        if isinstance(self.worker, QThread) and self.worker.isRunning():
            ret = QMessageBox.question(
                self,
                self.tr("Confirmer la fermeture"),
                self.tr("⚠️ Une traduction est en cours.\n"
                        "Voulez-vous vraiment l'interrompre et quitter ?"
                ),
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No,
            )
            if ret != QMessageBox.Yes:
                return False

            # Arrêt complet
            self._stop_worker_safely()
            self._reset_visual()
            self._closing_safely = True
            return True

        # CAS 2 : aucun worker actif → simple confirmation
        ret = QMessageBox.question(
            self,
            self.tr("Quitter"),
            self.tr("Voulez-vous vraiment quitter ?"),
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No,
        )
        if ret == QMessageBox.Yes:
            self._closing_safely = True
            # Ici, on peut tracer proprement
            self.state.on_window_closed_idle()
            return True

        return False

# ----------------------------------------------------------------------
#  MAIN (standalone)
# ----------------------------------------------------------------------

def main():
    app = QApplication.instance() or QApplication([])
    w = StartGUI()
    w.show()
    if not QApplication.instance().property("_in_qgis_loop"):
        app.exec_()

if __name__ == "__main__":
    main()



