# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Plugin_translator
                                 A QGIS plugin
 Générateur et traducteur automatique de fichiers .ts pour plugins QGIS
 -------------------
        begin                : 2025-10-29
        copyright            : (C) 2025
        author               : François THÉVAND
        email                : francois.thevand@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   Ce programme est un logiciel libre ; vous pouvez le redistribuer et   *
 *   le modifier selon les termes de la GNU GPL, publiée par la Free       *
 *   Software Foundation ; soit la version 2 de la licence, soit (à votre  *
 *   choix) toute version ultérieure.                                      *
 *                                                                         *
 ***************************************************************************/
"""
import os
import tempfile
import stat
import sys
import subprocess
import platform
import shutil
import xml.etree.ElementTree as ET

from qgis.PyQt import QtWidgets, uic
from qgis.PyQt.QtCore import Qt, QFile
from qgis.PyQt.QtWidgets import QFileDialog, QMessageBox, QApplication, QListWidgetItem

from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
from qgis.core import QgsMessageLog, Qgis

from . import resources   # force l’enregistrement des ressources

# -------------------------------------------------------------------------
# 🌍 Localisation robuste — langue réellement utilisée par QGIS
#    - charge le QM correspondant à la langue QGIS si disponible
#    - fallback anglais si présent
#    - sinon : fonctionnement en langue source du plugin
# -------------------------------------------------------------------------
from qgis.core import QgsApplication
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from pathlib import Path

settings = QSettings()
plugin_dir = Path(__file__).resolve().parent
plugin_name = Path(__file__).resolve().parent.name
i18n_dir = plugin_dir / "i18n"

translator = QTranslator()
loaded = False

# ----------------------------------------------------------------------
#  TRADUCTION AVEC CONTEXTES — fonction unique
# ----------------------------------------------------------------------
TR_CONTEXT = "@default"

def tr(text: str) -> str:
    """
    Fonction de traduction globale SAFE pour pylupdate5.
    """
    return QCoreApplication.translate(TR_CONTEXT, text)

# ----------------------------------------------------------------------
# LOGGING CENTRALISÉ
# ----------------------------------------------------------------------
LOG_TAG = "PluginTranslator"

def qgis_log(msg: str, level: str = "INFO"):
    lvl = {
        "TRACE": Qgis.Info,
        "DEBUG": Qgis.Info,
        "INFO": Qgis.Info,
        "WARNING": Qgis.Warning,
        "ERROR": Qgis.Critical,
    }.get(level.upper(), Qgis.Info)

    QgsMessageLog.logMessage(msg, LOG_TAG, lvl)
# -------------------------------------------------------------------------
# 1️⃣ Langue réellement utilisée par QGIS
# -------------------------------------------------------------------------
locale_full = QgsApplication.locale() or settings.value("locale/userLocale", "")
lang = locale_full.split("_")[0].lower() if locale_full else ""

qgis_log(f"[i18n] Locale QGIS détectée : {locale_full}", "DEBUG")
qgis_log(f"[i18n] Langue QGIS détectée : {lang or 'indéterminée'}", "DEBUG")

# -------------------------------------------------------------------------
# 2️⃣ Chargement QM
# -------------------------------------------------------------------------
def load_qm(code: str) -> bool:
    qm = i18n_dir / f"{plugin_name}_{code}.qm"
    if qm.exists() and translator.load(str(qm)):
        qgis_log(f"[i18n] QM chargé : {qm.name}", "DEBUG")
        return True
    return False

# -------------------------------------------------------------------------
# 3️⃣ Logique de sélection (sans hypothèse sur la langue source)
# -------------------------------------------------------------------------
if lang and load_qm(lang):
    loaded = True
    qgis_log(f"[i18n] Traduction activée pour la langue QGIS : {lang}", "INFO")
elif load_qm("en"):
    loaded = True
    qgis_log(
        "[i18n] Traduction de la langue QGIS absente → fallback anglais",
        "WARNING"
    )
else:
    qgis_log(
        "[i18n] Aucun QM compatible trouvé → fonctionnement du plugin dans sa langue source",
        "INFO"
    )
# -------------------------------------------------------------------------
# 4️⃣ Installation du translator
# -------------------------------------------------------------------------
if loaded:
    QCoreApplication.installTranslator(translator)

# ----------------------------------------------------------------------
#  Extraction d’outils depuis les ressources
# ----------------------------------------------------------------------
def extract_resource_to_temp(path_in_qrc: str) -> Path:
    """
    Extrait un fichier des ressources vers un fichier temporaire exécutable.
    Utilisé pour distribuer lrelease/lupdate avec le plugin.
    """
    f = QFile(path_in_qrc)
    if not f.exists():
        return None
    if not f.open(QFile.ReadOnly):
        return None

    data = f.readAll()
    f.close()

    tmp = Path(tempfile.gettempdir()) / ("plugin_translator_" + Path(path_in_qrc).name)

    with tmp.open("wb") as out:
        out.write(bytes(data))

    try:
        tmp.chmod(tmp.stat().st_mode | stat.S_IEXEC)
    except Exception:
        pass

    return tmp

def detect_lrelease():
    """Charge lrelease.exe depuis les ressources du plugin si présent."""
    return extract_resource_to_temp(":/tools/lrelease.exe") or None

def detect_lupdate():
    """Prévu si tu veux ajouter lupdate plus tard (actuellement non utilisé)."""
    return extract_resource_to_temp(":/tools/lupdate.exe") or None

# ------------------------------------------------------------
# Détection de la langue du pays d'utilisation
# ------------------------------------------------------------
def get_qgis_locale() -> str:
    """
    Retourne la locale QGIS courante sous forme Qt (ex: fr_FR, en_GB).
    Fallback sûr sur en_GB.
    """
    try:
        from qgis.core import QgsApplication
        loc = QgsApplication.locale()  # ex: 'fr_FR'
        if isinstance(loc, str) and "_" in loc:
            return loc
    except Exception:
        pass
    return "en_GB"

# ------------------------------------------------------------
# Détection pylupdate (QGIS)
# ------------------------------------------------------------
def detect_pylupdate_qgis():
    """
    Tente de localiser pylupdate5 sur une installation QGIS Windows standard.
    Renvoie un tuple (python_qgis, pylupdate5_script_ou_exe).
    """
    python_qgis = Path(sys.exec_prefix) / "python.exe"
    if not python_qgis.exists():
        python_qgis = Path(sys.exec_prefix).parent / "apps" / "Python312" / "python.exe"

    scripts_dir = Path(sys.exec_prefix) / "Scripts"
    if not scripts_dir.exists():
        scripts_dir = Path(sys.exec_prefix).parent / "apps" / "Python312" / "Scripts"

    script_py = scripts_dir / "pylupdate5-script.py"
    if script_py.exists():
        return python_qgis, script_py

    exe = scripts_dir / "pylupdate5.exe"
    if exe.exists():
        return None, exe

    return None, None

# ------------------------------------------------------------
# Chargement UI
# ------------------------------------------------------------
FORM_CLASS, _ = uic.loadUiType(
    os.path.join(os.path.dirname(__file__), "plugin_translator_dialog_base.ui")
)

# ------------------------------------------------------------
# Mapping codes → locales Qt
# ------------------------------------------------------------
LANG_CODE_TO_QT = {
    "fr": "fr_FR",
    "en": "en_GB",
    "es": "es_ES",
    "pt": "pt_PT",
    "it": "it_IT",
    "de": "de_DE",
    "nl": "nl_NL",
    "sv": "sv_SE",
    "no": "nb_NO",
    "da": "da_DK",
    "fi": "fi_FI",
    "pl": "pl_PL",
    "cs": "cs_CZ",
    "sk": "sk_SK",
    "sl": "sl_SI",
    "hr": "hr_HR",
    "sr": "sr_RS",
    "ro": "ro_RO",
    "hu": "hu_HU",
    "bg": "bg_BG",
    "el": "el_GR",
    "ru": "ru_RU",
    "uk": "uk_UA",
    "he": "he_IL",
    "ar": "ar",
    "tr": "tr_TR",
    "zh": "zh_CN",
    "zh-TW": "zh_TW",
    "ja": "ja_JP",
    "ko": "ko_KR",
    "hi": "hi_IN",
}

SOURCE_LANG_CHOICES = {
    "fr_FR": "Français",
    "en_GB": "Anglais",
    "es_ES": "Espagnol",
    "pt_PT": "Portugais",
    "it_IT": "Italien",
    "de_DE": "Allemand",
    "nl_NL": "Néerlandais",
    "sv_SE": "Suédois",
    "nb_NO": "Norvégien",
    "da_DK": "Danois",
    "fi_FI": "Finnois",
    "pl_PL": "Polonais",
    "cs_CZ": "Tchèque",
    "sk_SK": "Slovaque",
    "sl_SI": "Slovène",
    "hr_HR": "Croate",
    "sr_RS": "Serbe",
    "ro_RO": "Roumain",
    "hu_HU": "Hongrois",
    "bg_BG": "Bulgare",
    "el_GR": "Grec",
    "ru_RU": "Russe",
    "uk_UA": "Ukrainien",
    "he_IL": "Hébreu",
    "ar": "Arabe",
    "tr_TR": "Turc",
    "zh_CN": "Chinois",
    "zh_TW": "Chinois traditionnel",
    "ja_JP": "Japonais",
    "ko_KR": "Coréen",
    "hi_IN": "Hindi",
}

# ------------------------------------------------------------
# FONCTIONS UTILITAIRES TS
# ------------------------------------------------------------
# ----------------------------------------------------------------------
#  TRADUCTION AVEC CONTEXTES — fonction unique
# ----------------------------------------------------------------------

def set_ts_header(ts_path: Path, language: str, sourcelanguage: str):
    """
    Met à jour l'entête <TS> :
      - version = 2.1
      - language
      - sourcelanguage
    """
    if not isinstance(ts_path, Path) or not ts_path.exists():
        return

    qgis_log(
        tr("Mise à jour entête TS : {0} (lang={1}, source={2})")
        .format(ts_path.name, language, sourcelanguage),
        "TRACE"
    )

    try:
        tree = ET.parse(ts_path)
        root = tree.getroot()

        # --------------------------------------------------
        # Validation stricte du format TS
        # --------------------------------------------------
        if root.tag != "TS":
            qgis_log(
                tr("Racine TS inattendue dans {0}")
                .format(ts_path),
                "WARNING"
            )
            return

        # --------------------------------------------------
        # Entête TS normalisée
        # --------------------------------------------------
        root.set("version", "2.1")

        if isinstance(language, str) and "_" in language:
            root.set("language", language)

        if isinstance(sourcelanguage, str) and "_" in sourcelanguage:
            root.set("sourcelanguage", sourcelanguage)

        # --------------------------------------------------
        # Écriture sûre (UTF-8 + déclaration XML)
        # --------------------------------------------------
        tree.write(
            ts_path,
            encoding="utf-8",
            xml_declaration=True
        )

        qgis_log(
            tr("Entête TS mis à jour : {0}")
            .format(ts_path.name),
            "DEBUG"
        )

    except ET.ParseError as e:
        qgis_log(
            tr("Erreur XML dans {0} : {1}")
            .format(ts_path.name, e),
            "ERROR"
        )

    except Exception as e:
        qgis_log(
            tr("Erreur lors de la mise à jour de l'entête TS : {0}")
            .format(e),
            "ERROR"
        )


# ------------------------------------------------------------
# DIALOGUE PRINCIPAL
# ------------------------------------------------------------
class TsGeneratorDialog(QtWidgets.QDialog, FORM_CLASS):

    """
    ATTENTION : pour une traduction correcte de la boite de dialogue fournie en ui
    il faut donner en contexte le nom de la balise class contenue dans l'ui
    (Ici, c'est TsGeneratorDialogBase, ligne dans l'ui : <class>TsGeneratorDialogBase</class>
    """
    @staticmethod
    def tr(text: str) -> str:
        return QCoreApplication.translate("TsGeneratorDialogBase", text)

    def __init__(self, parent=None, parent_plugin=None):
        super().__init__(parent)

        self.setupUi(self)
        # ✅ TRADUCTION AUTOMATIQUE DES TEXTES DU .ui
        self.retranslateUi(self)

        self.parent_plugin = parent_plugin
        self.settings = QSettings()
        self.plugin_dir = None

        self.saved_langs = []

        self.setWindowTitle(self.tr("Générateur de fichier pivot .TS"))

        # Détection pylupdate
        self.pylupdate = detect_pylupdate_qgis()
        pyexec, pylu = self.pylupdate
        if pylu is None:
            QMessageBox.warning(
                self,
                self.tr("Attention"),
                self.tr("pylupdate5 introuvable. Aucune génération TS possible.")
            )

        # Connexions
        self.btnBrowseFolder.clicked.connect(self.select_folder)
        self.btnGenerateTS.clicked.connect(self.generate_ts)
        self.comboZone.currentIndexChanged.connect(self.update_languages_list)
        self.listLanguages.itemChanged.connect(self.on_language_item_changed)

        # Définition des zones + chargement des préférences
        self.define_language_zones()
        self.init_source_language_combo()
        self.load_settings()

    # --------------------------------------------------------
    def define_language_zones(self):
        """Définit les zones et leurs langues."""
        # Dictionnaire interne : les clés restent stables (non traduites),
        # on traduira seulement les libellés lors de l'affichage.
        self.LANG_ZONES = {
            "Europe Ouest": {
                "fr": "Français",
                "en": "Anglais",
                "es": "Espagnol",
                "pt": "Portugais",
                "it": "Italien",
                "de": "Allemand",
                "nl": "Néerlandais",
                "sv": "Suédois",
                "no": "Norvégien",
                "da": "Danois",
                "fi": "Finnois",
            },
            "Europe Est": {
                "pl": "Polonais",
                "cs": "Tchèque",
                "sk": "Slovaque",
                "sl": "Slovène",
                "hr": "Croate",
                "sr": "Serbe",
                "ro": "Roumain",
                "hu": "Hongrois",
                "bg": "Bulgare",
                "el": "Grec",
                "ru": "Russe",
                "uk": "Ukrainien",
            },
            "Moyen-Orient": {
                "he": "Hébreu",
                "ar": "Arabe",
                "tr": "Turc",
            },
            "Asie": {
                "zh": "Chinois (simplifié)",
                "zh-TW": "Chinois (traditionnel)",
                "ja": "Japonais",
                "ko": "Coréen",
                "hi": "Hindi",
            }
        }

        # Remplit la comboBox avec les zones (libellé traduit, clé stockée en userData)
        self.comboZone.clear()
        for zone_key in self.LANG_ZONES.keys():
            self.comboZone.addItem(self.tr(zone_key), zone_key)

    # --------------------------------------------------------
    def init_source_language_combo(self):
        """
        Initialise la comboBox de langue source :
        - toutes les langues sont proposées
        - la langue QGIS courante est sélectionnée par défaut
        """
        if not hasattr(self, "comboSourceLanguage"):
            return

        self.comboSourceLanguage.clear()

        for code, label in SOURCE_LANG_CHOICES.items():
            text = f"{self.tr(label)} ({code})"
            self.comboSourceLanguage.addItem(text, code)

        # --- Sélection automatique = langue QGIS ---
        qgis_locale = get_qgis_locale()
        idx = self.comboSourceLanguage.findData(qgis_locale)

        if idx < 0:
            # fallback intelligent : fr_FR → fr_*, en_GB → en_*
            short = qgis_locale.split("_")[0]
            for i in range(self.comboSourceLanguage.count()):
                data = self.comboSourceLanguage.itemData(i)
                if isinstance(data, str) and data.startswith(short):
                    idx = i
                    break

        if idx < 0:
            idx = self.comboSourceLanguage.findData("en_GB")

        self.comboSourceLanguage.setCurrentIndex(max(idx, 0))

    # --------------------------------------------------------
    def load_settings(self):
        """Charge le dernier dossier, préfixe et langues sélectionnées + langue source."""
        last_folder = self.settings.value("/Plugin_Translator/last_folder", "")
        prefix = self.settings.value("/Plugin_Translator/ts_prefix", "")
        lang_str = self.settings.value("/Plugin_Translator/selected_langs", "")

        source_lang = self.settings.value(
            "/Plugin_Translator/source_lang",
            get_qgis_locale()
        )

        if last_folder:
            self.editFolder.setText(last_folder)
            self.plugin_dir = Path(last_folder)

        if prefix:
            self.editPrefix.setText(prefix)

        if isinstance(lang_str, str) and lang_str:
            self.saved_langs = lang_str.split(",")
        else:
            self.saved_langs = []

        # Langue source
        if hasattr(self, "comboSourceLanguage"):
            # Repositionner sur la langue source sauvegardée
            idx = self.comboSourceLanguage.findData(source_lang)
            if idx < 0:
                idx = 0
            self.comboSourceLanguage.setCurrentIndex(idx)

        # Charger les langues pour la zone actuellement sélectionnée
        self.update_languages_list()

    # --------------------------------------------------------
    def update_languages_list(self):
        """Met à jour la liste des langues en fonction de la zone sélectionnée."""
        # Récupère la clé interne de zone (userData)
        zone_key = self.comboZone.currentData()
        langs = self.LANG_ZONES.get(zone_key, {})

        self.listLanguages.blockSignals(True)
        self.listLanguages.clear()

        for code, label in langs.items():
            # Libellé traduit pour l'affichage
            item = QListWidgetItem(f"{self.tr(label)} ({code})")
            item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
            item.setCheckState(Qt.Checked if code in self.saved_langs else Qt.Unchecked)
            item.setData(Qt.UserRole, code)
            self.listLanguages.addItem(item)

        self.listLanguages.blockSignals(False)

    # --------------------------------------------------------
    def on_language_item_changed(self, item: QListWidgetItem):
        """Met à jour self.saved_langs lorsqu'on coche/décoche une langue."""
        code = item.data(Qt.UserRole)
        if not code:
            return

        if item.checkState() == Qt.Checked:
            if code not in self.saved_langs:
                self.saved_langs.append(code)
        else:
            if code in self.saved_langs:
                self.saved_langs.remove(code)

    # ----- Affichage par défaut du dossier plugins du profil en cours d'utilisation------
    # ----- Et réduction à 1/4 de l'écran -----
    def select_folder(self):
        """Ouvre le sélecteur sur le dossier plugins du profil QGIS courant."""

        # --- RÉCUPÉRATION FIABLE DU PROFIL QGIS ACTUEL ---
        try:
            from qgis.core import QgsApplication
            profile_path = Path(QgsApplication.qgisSettingsDirPath())  # ex : .../profiles/default/
            plugin_dir_default = profile_path / "python" / "plugins"
        except Exception:
            plugin_dir_default = Path.home()

        if not plugin_dir_default.exists():
            plugin_dir_default = Path.home()

        # --- DIALOGUE FICHIERS (NON NATIF POUR CONTRÔLER LA TAILLE) ---
        dlg = QFileDialog(self, self.tr("Choisissez un dossier de plugin QGIS"))
        dlg.setFileMode(QFileDialog.Directory)
        dlg.setOption(QFileDialog.ShowDirsOnly, True)
        dlg.setOption(QFileDialog.DontUseNativeDialog, True)  # <<< essentiel pour éviter plein écran
        dlg.setDirectory(str(plugin_dir_default))

        # --- Taille = 1/2 écran (¼ surface) & centré ---
        screen = QApplication.primaryScreen().availableGeometry()
        w = screen.width() // 2
        h = screen.height() // 2

        dlg.resize(w, h)
        dlg.move(
            screen.center().x() - w // 2,
            screen.center().y() - h // 2,
        )

        if dlg.exec_():
            folder = dlg.selectedFiles()[0]
            self.plugin_dir = Path(folder)
            self.editFolder.setText(folder)
            self.editPrefix.setText(self.plugin_dir.name)
            self.settings.setValue("/Plugin_Translator/last_folder", folder)
            self.settings.sync()

    # --------------------------------------------------------
    def get_source_language(self) -> str:
        """
        Renvoie la langue source sélectionnée (locale Qt, ex. 'fr_FR' ou 'en_GB').
        Si la combo n'existe pas ou est vide : 'en_GB' par défaut.
        """
        if hasattr(self, "comboSourceLanguage") and self.comboSourceLanguage.count() > 0:
            code = self.comboSourceLanguage.currentData()
            if isinstance(code, str) and code:
                return code
        return "en_GB"

    # --------------------------------------------------------
    def generate_ts(self):
        """Génère un fichier pivot .ts et des copies par langue."""
        if not self.plugin_dir or not self.plugin_dir.exists():
            QMessageBox.warning(self, self.tr("Erreur"), self.tr("Sélectionnez un dossier valide."))
            return

        # Liste finale des langues sélectionnées (toutes zones confondues)
        languages = list(dict.fromkeys(self.saved_langs))

        if not languages:
            QMessageBox.warning(self, self.tr("Erreur"), self.tr("Aucune langue sélectionnée."))
            return

        # Langue source du pivot (fr_FR ou en_GB)
        source_locale = self.get_source_language()

        # Enregistrer la configuration (le préfixe est conservé mais n'est plus utilisé pour les noms de fichiers)
        prefix = self.editPrefix.text().strip()
        self.settings.setValue("/Plugin_Translator/ts_prefix", prefix)
        self.settings.setValue("/Plugin_Translator/selected_langs", ",".join(languages))
        self.settings.setValue("/Plugin_Translator/source_lang", source_locale)
        self.settings.sync()

        # pylupdate
        try:
            pyexec, pylu = self.pylupdate
        except Exception:
            QMessageBox.critical(self, self.tr("Erreur"), self.tr("pylupdate5 invalide."))
            return

        if pylu is None:
            QMessageBox.critical(self, self.tr("Erreur"), self.tr("pylupdate5 introuvable."))
            return

        # Préparation des chemins
        plugin_name = self.plugin_dir.name  # nom du dossier choisi
        i18n_dir = self.plugin_dir / "i18n"
        i18n_dir.mkdir(exist_ok=True)
        pivot_ts = i18n_dir / f"{plugin_name}.ts"

        # Récupérer tous les fichiers .py (récursivement)
        py_files = [str(p) for p in self.plugin_dir.rglob("*.py")]
        if not py_files:
            QMessageBox.warning(
                self,
                self.tr("Erreur"),
                self.tr("Aucun fichier .py trouvé dans le dossier sélectionné.")
            )
            return
        # Récupérer tous les fichiers .ui (récursivement)
        ui_files = [str(p) for p in self.plugin_dir.rglob("*.ui")]

        sources = py_files + ui_files

        # UI progression
        total_steps = 1 + len(languages)  # 1 pour le pivot, puis 1 par langue
        self.progressBar.setMinimum(0)
        self.progressBar.setMaximum(total_steps)
        self.progressBar.setValue(0)
        self.labelStatus.setText(self.tr("Création du fichier pivot…"))
        QApplication.processEvents()

        # Appel à pylupdate5 pour générer le pivot
        try:
            if pyexec is not None:
                cmd = [str(pyexec), str(pylu)] + sources + ["-ts", str(pivot_ts)]
            else:
                cmd = [str(pylu)] + sources + ["-ts", str(pivot_ts)]

            # --- MASQUER LA FENÊTRE SOUS WINDOWS ---
            kwargs = {}
            if platform.system().lower() == "windows":
                kwargs["creationflags"] = subprocess.CREATE_NO_WINDOW

            # 🔍 TRACE uniquement (jamais en PROD)
            qgis_log(
                f"Commande pylupdate5 : {' '.join(cmd)}",
                "TRACE"
            )

            proc = subprocess.run(
                cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
                **kwargs
            )

            if proc.returncode != 0:
                # ❌ ERREUR réelle → visible même en PROD
                qgis_log(f"pylupdate5 stderr: {proc.stderr}", "ERROR")

                QMessageBox.critical(
                    self,
                    self.tr("Erreur"),
                    self.tr("Échec de l'exécution de pylupdate5.\n\n"
                       "Commande : {0}\n\n"
                       "Erreur : {1}"
                    ).format(' '.join(cmd), proc.stderr)
                )
                return

        except Exception as e:
            QMessageBox.critical(
                self,
                self.tr("Erreur"),
                self.tr("Erreur lors de l'exécution de pylupdate5 :\n{0}").format(e)
            )
            return

        # ------------------------------------------------------------------
        # Pivot TS : langue = langue source (technique)
        # ------------------------------------------------------------------
        set_ts_header(
            pivot_ts,
            language=source_locale,
            sourcelanguage=source_locale
        )

        # ------------------------------------------------------------------
        # ➕ Génération du TS explicite de la langue source
        # ex : plugin_fr.ts / plugin_en.ts
        # ------------------------------------------------------------------
        source_short = source_locale.split("_")[0]  # fr_FR → fr
        source_ts = i18n_dir / f"{plugin_name}_{source_short}.ts"

        try:
            shutil.copyfile(pivot_ts, source_ts)
            set_ts_header(
                source_ts,
                language=source_locale,
                sourcelanguage=source_locale
            )

            self.autofill_source_ts(source_ts)

            qgis_log(
                self.tr("Fichier TS langue source créé : {0}")
                .format(source_ts.name),
                "INFO"
            )

        except Exception as e:
            msg = self.tr(
                "Erreur lors de la création du TS langue source ({0}) : {1}"
            ).format(source_ts.name, e)
            qgis_log(msg, "ERROR")

        self.progressBar.setValue(1)
        self.progressBar.setFormat(self.tr("1 / %d (pivot)") % total_steps)
        self.labelStatus.setText(self.tr("Pivot créé : {0}").format(pivot_ts.name))
        QApplication.processEvents()

        errors = []

        # Créer les copies pour chaque langue sélectionnée
        current_step = 1
        for lang_code in languages:
            current_step += 1
            qt_locale = LANG_CODE_TO_QT.get(lang_code)
            if not qt_locale:
                msg = self.tr("Aucun mapping locale Qt pour le code '{0}'. Fichier ignoré.").format(lang_code)
                qgis_log(msg, "DEBUG")
                errors.append(msg)
                continue

            dest_ts = i18n_dir / f"{plugin_name}_{lang_code}.ts"

            try:
                shutil.copyfile(str(pivot_ts), str(dest_ts))
                # Mettre l'entête avec langue cible + langue source
                set_ts_header(dest_ts, language=qt_locale, sourcelanguage=source_locale)
            except Exception as e:
                msg = self.tr("Erreur lors de la création de {0} : {1}").format(dest_ts.name, e)
                qgis_log(msg, "ERROR")
                errors.append(msg)
                continue

            self.labelStatus.setText(self.tr("Création {0}…").format(dest_ts.name))
            self.progressBar.setValue(current_step)
            self.progressBar.setFormat(self.tr("{0} / {1} ({2})").format(current_step, total_steps, lang_code))
            QApplication.processEvents()

        # FIN DE PROGRESSION
        self.progressBar.setValue(total_steps)
        self.progressBar.setFormat(self.tr("Terminé !"))
        self.labelStatus.setText(self.tr("Terminé."))
        QApplication.processEvents()

        if errors:
            QMessageBox.warning(
                self,
                self.tr("Avertissement"),
                self.tr("Génération terminée avec avertissements :\n\n") + "\n".join(errors)
            )
        else:
            QMessageBox.information(
                self,
                self.tr("Terminé"),
                self.tr("Génération du fichier pivot et des copies de langues effectuée avec succès.")
            )

        # Proposer d'ouvrir le traducteur
        resp = QMessageBox.question(
            self,
            self.tr("Traduction"),
            self.tr("Voulez-vous lancer la traduction maintenant ?"),
            QMessageBox.Yes | QMessageBox.No,
        )

        if resp == QMessageBox.Yes:
            self.close()
            if self.parent_plugin:
                self.parent_plugin.run_translator()
            else:
                QMessageBox.critical(
                    self,
                    self.tr("Erreur"),
                    self.tr("Impossible d'appeler run_translator() : parent_plugin manquant.")
                )

    def autofill_source_ts(self, ts_path: Path):
        """
        Copie <source> → <translation> pour la langue source.
        Supprime le flag unfinished.
        """
        tree = ET.parse(ts_path)
        root = tree.getroot()

        for message in root.findall(".//message"):
            src = message.find("source")
            trn = message.find("translation")

            if src is None or trn is None:
                continue

            trn.text = src.text
            trn.attrib.pop("type", None)

        tree.write(ts_path, encoding="utf-8", xml_declaration=True)

# ----------------------------------------------------------------------
#  Classe principale du plugin QGIS
# ----------------------------------------------------------------------
class TsGenerator:
    """Classe principale du plugin QGIS : gestion interface et actions."""

    @staticmethod
    def tr(text: str) -> str:
        return QCoreApplication.translate("TsGenerator", text)

    def __init__(self, iface):
        """
        Constructor.
        :param iface: instance de QgisInterface (fournie par QGIS)
        """
        self.iface = iface
        self.dlg = None
        self.action = None
        self.action_translator = None  # bouton Traducteur
        self._translator_gui = None

    # ------------------------------------------------------------------
    def initGui(self):
        """Initialise l'interface graphique du plugin dans QGIS."""

        icon_ts = QIcon(":/images/themes/default/mActionEditTable.svg")
        icon_translate = QIcon(str(Path(__file__).parent / "icon.svg"))
        # icon_translate = QIcon(":/images/themes/default/labelingRuleBased.svg")

        # Action : Générateur de fichiers TS
        self.action = QAction(
            icon_ts,
            self.tr("Générateur de fichiers .TS"),
            self.iface.mainWindow()
        )
        self.action.triggered.connect(self.run)

        # Action : Traducteur (Google)
        self.action_translator = QAction(
            icon_translate,
            self.tr("Traducteur Google"),
            self.iface.mainWindow()
        )
        self.action_translator.triggered.connect(self.run_translator)

        # Ajouts dans QGIS
        self.iface.addToolBarIcon(self.action)
        self.iface.addToolBarIcon(self.action_translator)
        self.iface.addPluginToMenu("&Plugin Translator", self.action)
        self.iface.addPluginToMenu("&Plugin Translator", self.action_translator)

    # ------------------------------------------------------------------
    def unload(self):
        """Nettoyage des actions QGIS lors de la désactivation du plugin."""
        if self.action:
            self.iface.removeToolBarIcon(self.action)
            self.iface.removePluginMenu("&Plugin Translator", self.action)
            self.action = None

        if self.action_translator:
            self.iface.removeToolBarIcon(self.action_translator)
            self.iface.removePluginMenu("&Plugin Translator", self.action_translator)
            self.action_translator = None

    # ------------------------------------------------------------------
    def run_translator(self):
        """Lance la fenêtre du traducteur (Google)."""
        try:
            import sip
            from .translate_ts_html_gui_API_google import StartGUI

            # Si la boîte de génération TS est ouverte, on la ferme
            if getattr(self, "dlg", None) is not None:
                try:
                    if self.dlg.isVisible():
                        self.dlg.close()
                except Exception:
                    pass

            # Si la fenêtre traducteur existe déjà et est encore valide : on la remonte
            if getattr(self, "_translator_gui", None) and not sip.isdeleted(self._translator_gui):
                self._translator_gui.raise_()
                self._translator_gui.activateWindow()
                return

            # Optionnel : lrelease intégré (lupdate pas utilisé pour le moment)
            lrelease = detect_lrelease()

            self._translator_gui = StartGUI(
                parent=self.iface.mainWindow(),
                # html_enabled=False,
                # lrelease_path=lrelease,
            )

            # Quand Qt détruit la fenêtre → référence Python = None
            self._translator_gui.destroyed.connect(
                lambda: setattr(self, "_translator_gui", None)
            )

            self._translator_gui.setWindowFlags(Qt.Window | Qt.WindowStaysOnTopHint)
            self._translator_gui.show()
            self._translator_gui.raise_()
            self._translator_gui.activateWindow()

        except Exception as e:
            self.iface.messageBar().pushCritical(
                "Plugin Translator",
                self.tr("Erreur au lancement du traducteur : {}").format(e)
            )

    # ------------------------------------------------------------------
    def run(self):
        """Affiche la boîte de dialogue principale du plugin."""
        if not self.dlg:
            self.dlg = TsGeneratorDialog(self.iface.mainWindow(), parent_plugin=self)

        self.dlg.show()
        self.dlg.raise_()
        self.dlg.activateWindow()


