# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GeoPEC - Software científico para avaliação da acurácia posicional em 
   dados cartográficos
 GeoPEC - Scientific software for assessing positional accuracy in 
   spatial data

 A QGIS plugin [Generated by Plugin Builder]

Tamanho amostral: Definição do tamanho amostral pelas normas ISO 2859, 
  ET-CQDG (Brasil), NBR 13.133 (Brasil), ASPRS (2015), Portugal e 
  amostragem probabilística. 
Sample size: Definition of sample size according to ISO 2859, 
  ET-CQDG (Brazil), NBR 13.133 (Brazil), ASPRS (2015), Portugal standard
  and probabilistic sampling.

  autores: Afonso P. Santos; João Vítor A. Gonçalves; Luis Philippe Ventura
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication, QVariant
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QFileDialog, QMessageBox
from qgis.core import *
from osgeo import gdal
# Initialize Qt resources from file resources.py
from ..resources import *
import processing
# Import the code for the dialog
from ..gui.gerarentrada_dialog import gerarentradaDialog
import os.path
import math


class gerarentrada:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
    #################################
        # Pega o caminho do diretório onde o arquivo atual está localizado
        current_dir = os.path.dirname(os.path.abspath(__file__))
        # Sobe um nível na hierarquia de diretórios para chegar à pasta principal
        self.plugin_dir = os.path.dirname(current_dir)       
        
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(self.plugin_dir,'i18n','pec_{}.qm'.format(locale))
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)
    #################################
        
        #Configurando métodos para não aparecer janelas de erro mais de uma vez
        self.first_start = True
        self.connections_made = False 

        
        # Save reference to the QGIS interface
        self.iface = iface
   
        self.dlg = gerarentradaDialog()
        
        #Ativando métodos de atualização dos combo_Box quando uma layer é adicionada
        #ou removida do projeto
        QgsProject.instance().layersAdded.connect(self.carregaVetor)
        QgsProject.instance().layersRemoved.connect(self.carregaVetor)
            
        
    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('gerarentrada', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        pass
        
    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        self.iface.mainWindow().menuBar().removeAction(self.menu.menuAction())
        del self.menu
        # remove the toolbar
        del self.toolbar


    def carregaVetor(self):
        """Preenche os combobox com as layers vetoriais do tipo ponto"""
        #limpando os comboBox
        self.dlg.comboBox.clear()
        self.dlg.comboBox_2.clear()
        
        #obtendo as instâncias do projeto
        project = QgsProject.instance()
        
        #obtendo as layers
        layers = project.mapLayers()
        
        #obtendo somente as layers de pontos
        point_layers = []
        
        for layer in layers.values():
        
            if layer.type() == QgsMapLayerType.VectorLayer and layer.geometryType() == QgsWkbTypes.PointGeometry:
                point_layers.append(layer)
        
        # Obtendo os nomes das camadas de pontos
        point_layer_names = [layer.name() for layer in point_layers]

        # Preenchendo  os ComboBoxes
        self.dlg.comboBox.addItems(point_layer_names)
        self.dlg.comboBox_2.addItems(point_layer_names)


    def carregaCamada(self):
        """execução do botão para abrir uma layer. Será aberto uma janela de diálogo e a layer escolhida será adicionada ao projeto e ao combobox"""
        camada_abrir = str(QFileDialog.getOpenFileName(caption = "Escolha a camada",filter = "Shapefiles (*.shp)")[0])
    
        if camada_abrir:
    
            # Carregar a camada
            layer = QgsVectorLayer(camada_abrir, str.split(os.path.basename(camada_abrir), ".") [0], "ogr")
        
            # Verificando se a camada possui geometria de ponto
            if layer.geometryType() == QgsWkbTypes.PointGeometry:
                # Adicionar a camada ao projeto
                QgsProject.instance().addMapLayer(layer)
                self.carregaVetor()
            else:
            # Informar ao usuário que a camada não possui geometria de ponto
                QMessageBox.warning(self.dlg, "Erro", "A camada selecionada não possui geometria de ponto.")
                pass
    

    def obt_data(self):
        """Método que extrai os dados das camadas de teste e de referência, faz a junção espacial e gera uma estrutura para salvar os dados necessários"""
        
        #Obtendo o nome das camadas de teste e de referência
        nome_pt_teste = self.dlg.comboBox.currentText()
        nome_pt_ref = self.dlg.comboBox_2.currentText()
        
        #Obtendo as camadas pelo nome
        pt_teste = QgsProject.instance().mapLayersByName(nome_pt_teste)[0]
        pt_ref = QgsProject.instance().mapLayersByName(nome_pt_ref)[0]
        
        '''#Criando cópias temporárias das camadas de teste e de referência
        self.pt_teste = QgsVectorLayer(pt_teste.source(), 'TesteCopiaTemporaria', 'ogr')
        self.pt_ref = QgsVectorLayer(pt_ref.source(), 'RefCopiaTemporaria', 'ogr')'''
        
        self.pt_teste = pt_teste.clone()
        self.pt_ref = pt_ref.clone()
                
        # Verificando se as camadas têm o mesmo SRC
        crs_teste = self.pt_teste.crs()
        crs_ref = self.pt_ref.crs()
        if not crs_teste == crs_ref:
            return False
        
        
        # Removendo todos os atributos das camadas temporárias
        fields_test_names = [field.name() for field in self.pt_teste.fields()]
        self.pt_teste.dataProvider().deleteAttributes([self.pt_teste.fields().indexFromName(name) for name in fields_test_names])

        fields_ref_names = [field.name() for field in self.pt_ref.fields()]
        self.pt_ref.dataProvider().deleteAttributes([self.pt_ref.fields().indexFromName(name) for name in fields_ref_names])

    
        # Adicionando os novos atributos às camadas temporárias
        fields_test = [
            QgsField("IDt", QVariant.Int),
            QgsField("Xxtest", QVariant.Double),
            QgsField("Yytest", QVariant.Double),
            QgsField("Zztest", QVariant.Double),
        ]
        
        fields_ref = [
            QgsField("IDr", QVariant.Int),
            QgsField("Xxref", QVariant.Double),
            QgsField("Yyref", QVariant.Double),
            QgsField("Zzref", QVariant.Double),
        ]
        
        self.pt_teste.dataProvider().addAttributes(fields_test)
        self.pt_ref.dataProvider().addAttributes(fields_ref)
    
        # Atualizando os campos das camadas
        self.pt_teste.updateFields()
        self.pt_ref.updateFields()
  
        # calculando os campos X Y Z
        # =======================================================================
        
        self.pt_teste.startEditing()
        numb = 1
        for ft in self.pt_teste.getFeatures():
            self.pt_teste.changeAttributeValue(ft.id(), self.pt_teste.fields().indexFromName("IDt"), numb)
            self.pt_teste.changeAttributeValue(ft.id(), self.pt_teste.fields().indexFromName("Xxtest"), ft.geometry().asPoint().x())
            self.pt_teste.changeAttributeValue(ft.id(), self.pt_teste.fields().indexFromName("Yytest"), ft.geometry().asPoint().y())
            self.pt_teste.changeAttributeValue(ft.id(), self.pt_teste.fields().indexFromName("Zztest"), ft.geometry().vertexAt(0).z())
            
            numb += 1
        self.pt_teste.commitChanges()

        self.pt_ref.startEditing()

        nn = 1
        for fr in self.pt_ref.getFeatures():
            self.pt_ref.changeAttributeValue(fr.id(), self.pt_ref.fields().indexFromName("IDr"), nn)
            self.pt_ref.changeAttributeValue(fr.id(), self.pt_ref.fields().indexFromName("Xxref"), fr.geometry().asPoint().x())
            self.pt_ref.changeAttributeValue(fr.id(), self.pt_ref.fields().indexFromName("Yyref"), fr.geometry().asPoint().y())
            self.pt_ref.changeAttributeValue(fr.id(), self.pt_ref.fields().indexFromName("Zzref"), fr.geometry().vertexAt(0).z())
            
            nn += 1
        self.pt_ref.commitChanges()
        
        # Realizando o spatial join para juntar os dados de teste e de referência
        # Considera pontos homólogos os pontos mais próximos entre o conjunto dos dados de teste e de referência
        # OBS: Caso o produto possua um efeito sistemático de grande magnitude, esta metodologia de definição de pontos homólogos não será eficiente
        algresult  = processing.run("qgis:joinbynearest", {
                'DISCARD_NONMATCHING' : True, 
                'FIELDS_TO_COPY' : [], 
                'INPUT' : self.pt_teste, 
                'INPUT_2' : self.pt_ref, 
                'MAX_DISTANCE' : None, 
                'NEIGHBORS' : 1, 
                'OUTPUT' : 'memory:JOIN', 
                'PREFIX' : '' })
                
        self.join = algresult['OUTPUT']
        
        # Manipulando os dados do spatial join para gerar uma estrutura com os dados necessário para gerar o arquivo TXT
        data = []
       
        for f in self.join.getFeatures(): 
            
            dict_valores = { "IDt" : f["IDt"],
                            "Xxtest": f["Xxtest"],
                            "Yytest" : f["Yytest"],
                            "Zztest" : f["Zztest"],
                            "Xxref" : f["Xxref"],
                            "Yyref" : f["Yyref"],
                            "Zzref" : f["Zzref"]}
            
            for pt in dict_valores.keys():
                
                #Verificando se existem dados nan ou NULL, caso existam, são substituídos por 0
                
                comp = str(dict_valores[pt])
                
                if  (comp == 'nan' or comp == 'NULL'):
                    
                    dict_valores[pt] = 0
                
                else:
                    
                    pass            
            
            idd, xt, yt, zt, xr, yr, zr = dict_valores["IDt"], dict_valores["Xxtest"], dict_valores["Yytest"], dict_valores["Zztest"], dict_valores["Xxref"], dict_valores["Yyref"], dict_valores["Zzref"]
            data.append(f"{idd}\t{xt}\t{yt}\t{zt}\t{xr}\t{yr}\t{zr}")
  
        #Apagando as camadas temporárias e o join
        del self.join 
        del self.pt_teste
        del self.pt_ref   
                
            
        return data
        
    
    def abrir_caminho(self):
        """execução do botão para definir o local e o nome do arquivo TXT."""
        
        # Abrir uma caixa de diálogo para salvar o arquivo de saída
        file_path, _ = QFileDialog.getSaveFileName(self.dlg, "Salvar arquivo", "", "Arquivos de Texto (*.txt);;Todos os Arquivos (*.*)")
        self.dlg.lineEdit.setText(file_path)
        pass


    def gerar_arquivo(self):
        """Método que gera o arquivo TXT de entrada do GeoPEC Desktop"""
        
        #Obtendo os dados para salvar no relatório
        data = self.obt_data()
      
        #Obtendo o caminho do LineEdit
        file_path = self.dlg.lineEdit.text()
        if file_path:
        
            # Escrevendo os dados no arquivo
            if data:
                with open(file_path, 'w') as file:
                    for line in data:
                        file.write(line + '\n')
                    
                QMessageBox.information(self.dlg, "Sucesso", "Arquivo gerado com sucesso.")
                pass

            else:
                QMessageBox.critical(self.dlg, 'Erro', 'Não foi possível gerar o arquivo de entrada, confira se há dados nas camadas e também se possuem o mesmo SRC.') 
                pass


    def run(self):
        """Run method that performs all the real work"""
        
        if not self.connections_made:
            # show the dialog
            self.carregaVetor()
            
            #Conectando os botões
            self.dlg.pushButton.clicked.connect(self.carregaCamada)
            self.dlg.pushButton_2.clicked.connect(self.carregaCamada)
            self.dlg.pushButton_3.clicked.connect(self.abrir_caminho)
            self.dlg.pushButton_4.clicked.connect(self.gerar_arquivo)
             
            
            self.connections_made = True #Marcar como conectado
        
        self.dlg.show()
    # pass
	