# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PlotGrid
                                 A QGIS plugin
 This plugin create a grid file from a plot boundary file
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-02-07
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Jinha Jung
        email                : jinha@purdue.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, pyqtSignal, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import QgsProject, QgsVectorLayer, QgsFillSymbol, QgsMessageLog, QgsSingleSymbolRenderer

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .plot_grid_dialog import PlotGridDialog
import os.path
from osgeo import ogr
from osgeo import osr
import math
import numpy as np


# Functions
def dotproduct(v1, v2):
    return sum((a * b) for a, b in zip(v1, v2))


def length(v):
    return math.sqrt(dotproduct(v, v))


def angle(v1, v2):
    return math.acos(dotproduct(v1, v2) / (length(v1) * length(v2)))


class PlotGrid:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(self.plugin_dir, "i18n", "PlotGrid_{}.qm".format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&Plot grid tool")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("PlotGrid", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ":/plugins/plot_grid/icon.png"
        self.add_action(icon_path, text=self.tr("Plot grid"), callback=self.run, parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&Plot grid tool"), action)
            self.iface.removeToolBarIcon(action)

    def select_output_file(self):
        """
        Select output file name and updated the lineEdit instance
        """
        filename, _filter = QFileDialog.getSaveFileName(self.dlg, "Select output file", "", "*.geojson")
        self.dlg.lineEdit_output.setText(filename)
        QgsMessageLog.logMessage("File loading complete")

    def _reload_layer(self):
        pg_layer = QgsProject.instance().mapLayersByName("plot_grid")

        if len(pg_layer) == 0:
            vlayer = QgsVectorLayer(self.dlg.lineEdit_output.text(), "plot_grid", "ogr")
            QgsProject.instance().addMapLayer(vlayer)
        elif len(pg_layer) == 1:
            QgsProject.instance().removeMapLayers([pg_layer[0].id()])
            vlayer = QgsVectorLayer(self.dlg.lineEdit_output.text(), "plot_grid", "ogr")
            QgsProject.instance().addMapLayer(vlayer)

        # Styles
        symbol = QgsFillSymbol.createSimple({"color": "transparent", "outline_color": "blue", "width_border": "0.6"})
        renderer = QgsSingleSymbolRenderer(symbol)
        vlayer.setRenderer(renderer)
        vlayer.triggerRepaint()

    def _generate(self):
        # QgsMessageLog.logMessage("Generate button was clicked!")
        # Fetch the currently loaded layers
        layers = QgsProject.instance().layerTreeRoot().children()

        output_fn = self.dlg.lineEdit_output.text()
        nrows = self.dlg.spinBox_nrows.value()
        ncols = self.dlg.spinBox_ncols.value()
        width = self.dlg.doubleSpinBox_width.value()
        height = self.dlg.doubleSpinBox_height.value()

        # Now get spatial reference of the background layer
        selectedLayerIndex = self.dlg.comboBox_input.currentIndex()
        selectedLayer = layers[selectedLayerIndex].layer()
        epsg_code_str = selectedLayer.crs().authid()
        epsg_code = int(epsg_code_str.split(":")[1])

        sproj = osr.SpatialReference()
        sproj.ImportFromEPSG(epsg_code)

        # Create a output file
        driver = ogr.GetDriverByName("GeoJSON")

        # Now open input layer
        path_file_in = selectedLayer.dataProvider().dataSourceUri()
        dataSource = driver.Open(path_file_in, 0)
        print("Opened", path_file_in)

        # Create grid layer
        inLayer = dataSource.GetLayer()

        # Check if the file already exists
        if os.path.exists(output_fn):
            driver.DeleteDataSource(output_fn)

        ds = driver.CreateDataSource(output_fn)

        # Now create a new layer
        layer = ds.CreateLayer("plot_grid", srs=sproj, geom_type=ogr.wkbPolygon)
        featureDefn = layer.GetLayerDefn()

        # Create fields
        prow_defn = ogr.FieldDefn("prow", ogr.OFTInteger)
        pcol_defn = ogr.FieldDefn("pcol", ogr.OFTInteger)
        grow_defn = ogr.FieldDefn("grow", ogr.OFTInteger)
        gcol_defn = ogr.FieldDefn("gcol", ogr.OFTInteger)

        layer.CreateField(prow_defn)
        layer.CreateField(pcol_defn)
        layer.CreateField(grow_defn)
        layer.CreateField(gcol_defn)

        for i in range(0, inLayer.GetFeatureCount()):
            inFeature = inLayer.GetFeature(i)
            prow = inFeature.GetField("row")
            pcol = inFeature.GetField("col")
            geom = inFeature.GetGeometryRef()
            ring = geom.GetGeometryRef(0)
            # Find out ul, ur, ll, lr corners
            p0 = np.delete(np.asarray(ring.GetPoint(0)), [2])
            p1 = np.delete(np.asarray(ring.GetPoint(1)), [2])
            p3 = np.delete(np.asarray(ring.GetPoint(3)), [2])
            p2 = np.delete(np.asarray(ring.GetPoint(2)), [2])
            ang01 = angle(np.array([1.0, 0.0]), p1 - p0)
            ang12 = angle(np.array([1.0, 0.0]), p2 - p1)
            ang23 = angle(np.array([1.0, 0.0]), p3 - p2)
            ang30 = angle(np.array([1.0, 0.0]), p0 - p3)
            if ang01 < ang12 and ang01 < ang23 and ang01 < ang30 and p0[1] > p2[1]:
                # 0 1
                # 3 2
                ul = p0
                ur = p1
                ll = p3
                lr = p2
            elif ang01 < ang12 and ang01 < ang23 and ang01 < ang30 and p0[1] < p2[1]:
                # 3 2
                # 0 1
                ul = p3
                ur = p2
                ll = p0
                lr = p1
            elif ang12 < ang01 and ang12 < ang23 and ang12 < ang30 and p1[1] > p3[1]:
                # 1 2
                # 0 3
                ul = p1
                ur = p2
                ll = p0
                lr = p3
            elif ang12 < ang01 and ang12 < ang23 and ang12 < ang30 and p1[1] < p3[1]:
                # 0 3
                # 1 2
                ul = p0
                ur = p3
                ll = p1
                lr = p2
            elif ang23 < ang01 and ang23 < ang12 and ang23 < ang30 and p2[1] > p0[1]:
                # 2 3
                # 1 0
                ul = p2
                ur = p3
                ll = p1
                lr = p0
            elif ang23 < ang01 and ang23 < ang12 and ang23 < ang30 and p2[1] < p0[1]:
                # 1 0
                # 2 3
                ul = p1
                ur = p0
                ll = p2
                lr = p3
            elif ang30 < ang01 and ang30 < ang12 and ang30 < ang23 and p3[1] > p1[1]:
                # 3 0
                # 2 1
                ul = p3
                ur = p0
                ll = p2
                lr = p1
            elif ang30 < ang01 and ang30 < ang12 and ang30 < ang23 and p3[1] < p1[1]:
                # 2 1
                # 3 0
                ul = p2
                ur = p1
                ll = p3
                lr = p0

            # Draw grid polygon and save
            cen_plot = np.array([(ul[0] + ur[0] + ll[0] + lr[0]) / 4, (ul[1] + ur[1] + ll[1] + lr[1]) / 4])
            v_right = (ur - ul) / length(ur - ul)
            v_down = (ll - ul) / length(ll - ul)
            grid_start = cen_plot + ncols / 2 * width * (-v_right) + nrows / 2 * height * (-v_down)
            for row in range(nrows):
                for col in range(ncols):
                    grid_ul = grid_start + width * col * v_right + height * row * v_down
                    grid_ur = grid_start + width * (col + 1) * v_right + height * row * v_down
                    grid_ll = grid_start + width * col * v_right + height * (row + 1) * v_down
                    grid_lr = grid_start + width * (col + 1) * v_right + height * (row + 1) * v_down

                    # Create ring
                    ring = ogr.Geometry(ogr.wkbLinearRing)
                    ring.AddPoint(grid_ul[0], grid_ul[1])
                    ring.AddPoint(grid_ur[0], grid_ur[1])
                    ring.AddPoint(grid_lr[0], grid_lr[1])
                    ring.AddPoint(grid_ll[0], grid_ll[1])
                    ring.AddPoint(grid_ul[0], grid_ul[1])
                    # Create polygon
                    poly = ogr.Geometry(ogr.wkbPolygon)
                    poly.AddGeometry(ring)
                    outFeature = ogr.Feature(featureDefn)
                    outFeature.SetGeometry(poly)
                    outFeature.SetField("prow", prow)
                    outFeature.SetField("pcol", pcol)
                    outFeature.SetField("gcol", col + 1)
                    outFeature.SetField("grow", row + 1)
                    layer.CreateFeature(outFeature)
                    outFeature = None

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = PlotGridDialog()
            self.dlg.pushButton_output.clicked.connect(self.select_output_file)
            self.dlg.pushButton_generate.clicked.connect(self._generate)

        # Fetch the currently loaded layers
        layers = QgsProject.instance().layerTreeRoot().children()
        # Clear the contents of the comboBox from previous runs
        self.dlg.comboBox_input.clear()
        # Populate the comboBox with names of all the loaded layers
        self.dlg.comboBox_input.addItems([layer.name() for layer in layers])

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass
