# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PlotBoundary
                                 A QGIS plugin
 Generate plot boundary
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-10-21
        git sha              : $Format:%H$
        copyright            : (C) 2020 by Jinha Jung
        email                : jinha@purdue.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, pyqtSignal, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import QgsProject, QgsVectorLayer, QgsFillSymbol, QgsMessageLog, QgsSingleSymbolRenderer
from qgis.gui import QgsMapToolEmitPoint

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .plot_boundary_dialog import PlotBoundaryDialog
import os.path
from osgeo import ogr
from osgeo import osr
import math

def Rotate2D(ax, ay, ix, iy, angle):
    """
    Rotate a point around the anchor point

    Input
    -----
    ax, ay: coordinates of the anchor point
    ix, iy: coordinates of the point to rotate
    angle: angle in degree (counter-clockwise rotation is positive rotation)

    Output
    ------
    x, y : coordinates of a rotated point
    """
    dx = ix - ax
    dy = iy - ay
    # Convert degree into radian
    angle_rad = math.radians(angle)

    # do the rotation
    xp = dx * math.cos(angle_rad) - dy * math.sin(angle_rad) + ax
    yp = dx * math.sin(angle_rad) + dy * math.cos(angle_rad) + ay

    return xp, yp

class PointTool(QgsMapToolEmitPoint):
    def __init__(self, canvas):
        QgsMapToolEmitPoint.__init__(self, canvas)
        self.canvas = canvas

    canvasClicked = pyqtSignal('QgsPointXY')

    def canvasReleaseEvent(self, event):
        self.canvasClicked.emit(event.mapPoint())

class PlotBoundary:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'PlotBoundary_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Plot Boundary')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('PlotBoundary', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/plot_boundary/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Generate plot boundary'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&Plot Boundary'),
                action)
            self.iface.removeToolBarIcon(action)

    def select_output_file(self):
        """
        Select output file name and updated the lineEdit instance
        """
        filename, _filter = QFileDialog.getSaveFileName(self.dlg,
            "Select output file","","*.geojson")
        self.dlg.lineEdit_output.setText(filename)
        QgsMessageLog.logMessage("File loading complete")

    def _on_map_click(self):
        # self.dlg.hide()
        self.point_tool = PointTool(self.iface.mapCanvas())
        self.iface.mapCanvas().setMapTool(self.point_tool)
        self.point_tool.canvasClicked.connect(self._writeLineWidget)

    def _writeLineWidget(self, point):
        x,y = point
        self.dlg.lineEdit_left_coords.setText(str(x))
        self.dlg.lineEdit_up_coords.setText(str(y))
        self.point_tool.canvasClicked.disconnect()
        # self.dlg.show()

    def _reload_layer(self):
        pb_layer = QgsProject.instance().mapLayersByName('plot_boundary')

        if len(pb_layer) == 0:
            vlayer = QgsVectorLayer(self.dlg.lineEdit_output.text(), "plot_boundary", "ogr")
            QgsProject.instance().addMapLayer(vlayer)
        elif len(pb_layer) == 1:
            QgsProject.instance().removeMapLayers( [pb_layer[0].id()] )
            vlayer = QgsVectorLayer(self.dlg.lineEdit_output.text(), "plot_boundary", "ogr")
            QgsProject.instance().addMapLayer(vlayer)

        # Styles
        symbol = QgsFillSymbol.createSimple({'color': 'transparent', 'outline_color': 'red', 'width_border': "0.96"})
        renderer = QgsSingleSymbolRenderer(symbol)
        vlayer.setRenderer(renderer)
        vlayer.triggerRepaint()

    def _generate(self):
        # QgsMessageLog.logMessage("Generate button was clicked!")
        # Fetch the currently loaded layers
        layers = QgsProject.instance().layerTreeRoot().children()

        output_fn = self.dlg.lineEdit_output.text()
        num_row = self.dlg.spinBox_nrows.value()
        num_col = self.dlg.spinBox_ncols.value()
        ulx = float(self.dlg.lineEdit_left_coords.text())
        uly = float(self.dlg.lineEdit_up_coords.text())
        plot_width = self.dlg.doubleSpinBox_width.value()
        plot_height = self.dlg.doubleSpinBox_height.value()
        plot_spacing_x = self.dlg.doubleSpinBox_col_spacing.value()
        plot_spacing_y = self.dlg.doubleSpinBox_row_spacing.value()
        rot_angle = self.dlg.doubleSpinBox_rot_angle.value()

        # Now get spatial reference of the background layer
        selectedLayerIndex = self.dlg.comboBox.currentIndex()
        selectedLayer = layers[selectedLayerIndex].layer()
        epsg_code_str = selectedLayer.crs().authid()
        epsg_code = int(epsg_code_str.split(':')[1])

        sproj = osr.SpatialReference()
        sproj.ImportFromEPSG(epsg_code)

        # Create a output file
        driver = ogr.GetDriverByName("GeoJSON")

        # Check if the file already exists
        if os.path.exists(output_fn):
            driver.DeleteDataSource(output_fn)

        ds = driver.CreateDataSource(output_fn)

        # Now create a new layer
        layer = ds.CreateLayer('plot_boundary', srs = sproj, geom_type = ogr.wkbPolygon)

        # Create fields
        row_defn = ogr.FieldDefn('row', ogr.OFTInteger)
        col_defn = ogr.FieldDefn('col', ogr.OFTInteger)

        layer.CreateField(row_defn)
        layer.CreateField(col_defn)

        for row in range(num_row):
            for col in range(num_col):
                left = ulx + (plot_width + plot_spacing_x) * col
                right = left + plot_width
                up = uly - (plot_height + plot_spacing_y) * row
                down = up - plot_height

                # Apply rotation
                x0, y0 = Rotate2D(ulx, uly, left, up, rot_angle)
                x1, y1 = Rotate2D(ulx, uly, right, up, rot_angle)
                x2, y2 = Rotate2D(ulx, uly, right, down, rot_angle)
                x3, y3 = Rotate2D(ulx, uly, left, down, rot_angle)

                # Create a polygon
                ring_geom = ogr.Geometry(ogr.wkbLinearRing)
                ring_geom.AddPoint(x0, y0)
                ring_geom.AddPoint(x1, y1)
                ring_geom.AddPoint(x2, y2)
                ring_geom.AddPoint(x3, y3)
                ring_geom.AddPoint(x0, y0)

                poly_geom = ogr.Geometry(ogr.wkbPolygon)
                poly_geom.AddGeometry(ring_geom)

                # Attach fields to the feature
                ftDefn = layer.GetLayerDefn()
                ft = ogr.Feature(ftDefn)
                ft.SetGeometry(poly_geom)
                ft.SetField('row', row+1)
                ft.SetField('col', col+1)

                # Add the feature to the layer
                layer.CreateFeature(ft)

                # Destroy before going to next plot
                poly_geom.Destroy()
                ring_geom.Destroy()
                ft.Destroy()

        # Destroy data source to close out the file
        ds = None

        self._reload_layer()

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = PlotBoundaryDialog()
            self.dlg.setWindowFlags(Qt.WindowStaysOnTopHint)
            self.dlg.pushButton_output.clicked.connect(self.select_output_file)
            self.dlg.pushButton_get_coords.clicked.connect(self._on_map_click)
            self.dlg.pushButton_generate.clicked.connect(self._generate)

        # Fetch the currently loaded layers
        layers = QgsProject.instance().layerTreeRoot().children()
        # Clear the contents of the comboBox from previous runs
        self.dlg.comboBox.clear()
        # Populate the comboBox with names of all the loaded layers
        self.dlg.comboBox.addItems([layer.name() for layer in layers])

        # show the dialog
        self.dlg.show()

        # Run the dialog event loop
        result = self.dlg.exec_()

        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            # QgsMessageLog.logMessage("OK button was clicked.")
            pass
