# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PlacesSearch
                                 A QGIS plugin
 Fetch places data from Google Maps API and save to Shapefile.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-03-02
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Yanni Wang
        email                : yanni.wang2000@outlook.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import requests
import time
import geopandas as gpd
from shapely.geometry import Point
from shapely.ops import unary_union
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QFileDialog, QDialog
from qgis.core import QgsVectorLayer, QgsProject, QgsMessageLog, Qgis
from qgis.PyQt.QtGui import QIcon
from .places_search_dialog import PlacesSearchDialog


class PlacesSearch:
    def __init__(self, iface):
        """Initialize the plugin and toolbar button."""
        self.iface = iface
        self.dialog = PlacesSearchDialog()

        # Create an action with an icon
        self.action = QAction(QIcon(self.get_icon_path()), "Search Places", self.iface.mainWindow())
        self.action.triggered.connect(self.run)

    def initGui(self):
        """Add the plugin to the QGIS toolbar and menu."""
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&Plugins", self.action)

    def unload(self):
        """Remove the plugin from the toolbar and menu."""
        self.iface.removeToolBarIcon(self.action)
        self.iface.removePluginMenu("&Plugins", self.action)

    def run(self):
        """Execute the plugin."""
        self.dialog.populate_layers()  # Refresh available polygon layers
        # Run the dialog and check if the user accepted
        result = self.dialog.exec_()
        if result != QDialog.Accepted:
            QgsMessageLog.logMessage("User canceled the search.", "PlacesSearch", Qgis.Info)
            return  # Stop execution if canceled
    
        api_key, keywords, polygon_layer, output_path = self.dialog.get_inputs()

        if not api_key or not keywords or not polygon_layer or not output_path:
            QMessageBox.warning(self.iface.mainWindow(), "Error", "Please enter all required fields.")
            return

        # Convert QGIS polygon layer to GeoDataFrame
        polygon_gdf = gpd.GeoDataFrame.from_features(polygon_layer.getFeatures())
        QgsMessageLog.logMessage(f"Polygon layer CRS in QGIS:{polygon_layer.crs().authid()}", "PlacesSearch", level=Qgis.Info)

        # Check if the polygon layer is in EPSG:4326
        if polygon_layer.crs().authid() != "EPSG:4326":
            QMessageBox.critical(self.iface.mainWindow(), "CRS Error", 
                                 "Selected polygon layer is not in EPSG:4326.\nPlease convert it before running the search.")
            raise RuntimeError("Polygon layer must be in EPSG:4326.")
        
        # Convert the polygon layer to EPSG:4326
        polygon_gdf = polygon_gdf.set_crs(epsg=4326)

        # Merge all polygons into a single polygon
        search_polygon = unary_union(polygon_gdf.geometry)
        min_x, min_y, max_x, max_y = search_polygon.bounds
        center_lat = (min_y + max_y) / 2
        center_lng = (min_x + max_x) / 2
        radius = max(max_x - min_x, max_y - min_y)* 111000  # Convert degrees to meters
        QgsMessageLog.logMessage(f"Center: ({center_lat}, {center_lng}), Radius: {radius}", "PlacesSearch", level=Qgis.Info)

        all_results = []
        for keyword in keywords:
            params = {"location": f"{center_lat},{center_lng}", "radius": radius, "keyword": keyword, "key": api_key}
            all_results.extend(self.fetch_places(params))

        # Remove duplicate places based on place_id
        unique_results = {place["place_id"]: place for place in all_results}.values()

        # Extract relevant information
        summary = []
        for result in unique_results:
            name = result.get("name")
            vicinity = result.get("vicinity")
            lat = result["geometry"]["location"]["lat"]
            lng = result["geometry"]["location"]["lng"]
            point = Point(lng, lat)

            if search_polygon.contains(point):
                summary.append({"name": name, "vicinity": vicinity, "geometry": point})

        if not summary:
            QMessageBox.warning(self.iface.mainWindow(), "No Results", "No places found within the search area.")
            return

        self.save_results_to_qgis(summary, "EPSG:4326", output_path)

    def fetch_places(self, params):
        """Fetch places from Google Places API with adaptive waiting for OVER_QUERY_LIMIT."""
        nearby_search_url = "https://maps.googleapis.com/maps/api/place/nearbysearch/json"
        attempt = 0
        wait_times = [2, 5, 15, 60]
        results = []

        while attempt < len(wait_times):
            response = requests.get(nearby_search_url, params=params)
            if response.status_code == 200:
                data = response.json()
                status = data.get("status")

                if status == "OVER_QUERY_LIMIT":
                    time.sleep(wait_times[attempt])
                    attempt += 1
                    continue
                elif status == "OK":
                    results.extend(data.get("results", []))

                    # Handle pagination
                    next_page_token = data.get("next_page_token")
                    while next_page_token:
                        time.sleep(2)  # Required delay before requesting next page
                        params["pagetoken"] = next_page_token
                        response = requests.get(nearby_search_url, params=params)
                        data = response.json()
                        results.extend(data.get("results", []))
                        next_page_token = data.get("next_page_token")

                    return results
                else:
                    raise RuntimeError(f"Google API error: {status}")

            else:
                raise RuntimeError(f"HTTP error: {response.status_code}")

        raise RuntimeError("Exceeded retry attempts.")

    def get_icon_path(self):
        """Return the absolute path to the toolbar icon."""
        return os.path.join(os.path.dirname(__file__), "icon.png")
    
    def save_results_to_qgis(self, results, crs, output_path):
        """Save the results to a shapefile in QGIS."""
        if not results:
            QgsMessageLog.logMessage("No results to save.", "PlacesSearch", Qgis.Warning)
            return
        
        # Convert results to a GeoDataFrame
        gdf = gpd.GeoDataFrame(results, geometry="geometry", crs=crs)
        gdf.to_file(output_path, driver="ESRI Shapefile")

        self.load_layer_to_qgis(output_path)

        QgsMessageLog.logMessage(f"Results saved to: {output_path}", "PlacesSearch", Qgis.Info)


    def load_layer_to_qgis(self, file_path):
        """Load the saved Shapefile into QGIS."""
        if not os.path.exists(file_path):
            QgsMessageLog.logMessage("File not found, cannot load layer.", "PlacesSearch", Qgis.Critical)
            return

        layer_name = os.path.basename(file_path).replace(".shp", "")
        vector_layer = QgsVectorLayer(file_path, layer_name, "ogr")

        if not vector_layer.isValid():
            QgsMessageLog.logMessage("Failed to load the layer.", "PlacesSearch", Qgis.Critical)
            return

        QgsProject.instance().addMapLayer(vector_layer)
        QgsMessageLog.logMessage(f"Layer loaded: {layer_name}", "PlacesSearch", Qgis.Info)