# -*- coding: utf-8 -*-
import os
from math import cos, radians, sin


class WorldFileTool():
    """handling world file related things."""

    def __init__(self):
        """init."""
        pass

    def rotate_coordinates(self, x, y, deg):
        """Calculate rotated coordinates."""
        # positive degree => clockwise
        _cos = cos(radians(-deg * 1.0))
        _sin = sin(radians(-deg * 1.0))
        if _cos > 0 and _cos < 1e-16:
            _cos = 0
        if _sin > 0 and _sin < 1e-16:
            _sin = 0
        return [x * _cos - y * _sin,
                x * _sin + y * _cos]

    def sanitize_path(self, path):
        """Resolve path variables (~, $HOME, etc...)."""
        path = os.path.expandvars(os.path.expanduser(path))
        return os.path.abspath(path)

    def get_extension(self, extension):
        if len(extension) == 3:
            return str(extension[0]) + str(extension[2]) + 'w'
        else:
            return extension + 'w'

    def get_path(self, image_path):
        tmp = self.sanitize_path(image_path).split(os.extsep)
        path = '.'.join(tmp[:-1])
        ext = tmp[-1]

        return path + '.' + self.get_extension(ext)

    def calc_params(self, x, y, px_size_x, px_size_y,
                    rotation=0.0, w=False, h=False):

        rad = radians(-1.0 * rotation)  # positive => clockwise
        _cos = cos(rad)
        _sin = sin(rad)
        offset = [0, 0]

        # If provide image width and height, image center becomes center
        # of rotation.
        if w and h:
            # center of image => [0, 0]
            # offset: position of top_left after rotation
            offset = self.rotate_coordinates(- w * px_size_x / 2.0,
                                             h * px_size_y / 2.0,
                                             rotation)

        a = px_size_x * _cos
        d = px_size_x * _sin
        b = px_size_y * _sin
        e = - 1.0 * px_size_y * _cos
        c = x + offset[0]
        f = y + offset[1]

        return [a, d, b, e, c, f]

    def save(self, image_path, x, y, px_size_x, px_size_y,
             rotation=0.0, w=False, h=False):

        a, d, b, e, c, f = self.calc_params(x, y, px_size_x, px_size_y,
                                            rotation=0.0, w=False, h=False)
        out = []
        out.append(a)
        out.append(d)
        out.append(b)
        out.append(e)
        out.append(c)
        out.append(f)

        world_file_path = self.get_path(image_path)
        with open(world_file_path, 'w') as f:
            f.write('\n'.join(map(lambda x: str(x), out)))
            return world_file_path

        return False
