# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PianExporter
                                 A QGIS plugin
 Plugin exportuje vrstvu do formátu pro pian AMČR
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QUrl
from qgis.core import QgsProject, QgsCoordinateTransform, QgsCoordinateReferenceSystem, Qgis, QgsMessageLog, QgsMapLayer, QgsGeometry
from qgis.PyQt.QtGui import QIcon, QDesktopServices
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QPushButton

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .pian_exporter_dialog import PianExporterDialog
import os
import csv


class PianExporter:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = self.tr(u'&Pian Exporter')
        self.first_start = None

        # Locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(self.plugin_dir, f'i18n/PianExporter_{locale}.qm')
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

    def tr(self, message):
        return QCoreApplication.translate('PianExporter', message)

    def add_action(self, icon_path, text, callback, enabled_flag=True,
                   add_to_menu=True, add_to_toolbar=True, status_tip=None,
                   whats_this=None, parent=None):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)
        if status_tip:
            action.setStatusTip(status_tip)
        if whats_this:
            action.setWhatsThis(whats_this)
        if add_to_toolbar:
            self.iface.addToolBarIcon(action)
        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)
        return action

    def initGui(self):
        icon_path = ':/plugins/pian_exporter/icon.png'
        self.add_action(icon_path, text=self.tr(u'Pian Exporter'),
                        callback=self.run, parent=self.iface.mainWindow())
        self.first_start = True

    def unload(self):
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)

    def is_vector(self, layer):
            return layer.type() == QgsMapLayer.VectorLayer

    def run(self):
        # Zobrazuje dialog
        dialog = PianExporterDialog(self.iface)
        # Tohle je lepší udělat v dialogu při inicializaci, úplně na konec, TODO ještě se k tomu případně vrátit
        # dialog.layerComboBox.clear()
        # dialog.layerComboBox.addItems(self._get_layer_names())

        def check_geometry(text):
            layer_list = QgsProject.instance().mapLayersByName(text)
            if not layer_list:
                dialog.statusLabel.setText("Vrstva nenalezena!")
                return

            layer = layer_list[0]

            if not self.is_vector(layer):
                dialog.statusLabel.setText("Vybraná vrstva není vektorová!")
                return

            invalid_features = []
            for index, feature in enumerate(layer.getFeatures(), start=1):
                geom = feature.geometry()
                if geom.isEmpty() or not geom.isGeosValid():
                    invalid_features.append(index)

            if invalid_features:
                dialog.statusLabel.setText(f"Neplatná geometrie u prvků: {', '.join(map(str, invalid_features))}")
                self.iface.messageBar().pushMessage(
                    "Chyba geometrie",
                    f"Vrstva obsahuje neplatné geometrie u {len(invalid_features)} prvků.",
                    level=Qgis.Warning,
                    duration=7
                )
                dialog.exportButton.setEnabled(False)
            else:
                dialog.exportButton.setEnabled(True)
                dialog.statusLabel.setText("Geometrie vrstvy je v pořádku.")

        # Spuštění kontroly pro první vybranou vrstvu (pokud existuje)
        if self.iface.activeLayer():
            check_geometry(self.iface.activeLayer().name())

        dialog.layerComboBox.currentTextChanged.connect(check_geometry)
        dialog.exportButton.clicked.connect(lambda: self.export_from_dialog(dialog))
        dialog.exec_()
       
    def export_from_dialog(self, dialog):
        jtsk = 'EPSG:5514'
        wgs84 = 'EPSG:4326'

        # TODO Tohle asi upravit je to až po dialog.exec_()
        layer_name = dialog.layerComboBox.currentText()
        if not layer_name:
            dialog.statusLabel.setText("Vyber vrstvu!")
            return

        layer_list = QgsProject.instance().mapLayersByName(layer_name)
        if not layer_list:
            dialog.statusLabel.setText("Vrstva nenalezena!")
            return
        layer = layer_list[0]
        if layer.type() == QgsMapLayer.RasterLayer:
            dialog.statusLabel.setText("Vybraná vrstva je rastr!")
            return

        crs_original_layer = layer.crs()
        epsg_original = crs_original_layer.authid()
        QgsMessageLog.logMessage(f'Souřadnicový systém vybrané vrstvy je: {epsg_original}.', 'Zprávy', level=Qgis.Info)

        original_file_path = layer.source()
        default_dir = os.path.dirname(original_file_path)
        default_file = os.path.join(default_dir, f"{layer_name}_pian.csv")

        target_crs = QgsCoordinateReferenceSystem(wgs84)
        transformer = QgsCoordinateTransform(layer.crs(), target_crs, QgsProject.instance().transformContext())

        output_file, _ = QFileDialog.getSaveFileName(dialog, "Save CSV", default_file, "CSV Files (*.csv)")
        if not output_file:
            return

        simplify = dialog.simplifyCheckBox.isChecked()
        tolerance = dialog.simplifyToleranceSpinBox.value()

        with open(output_file, mode='w', newline='', encoding='utf-8') as csvfile:
            writer = csv.writer(csvfile, quoting=csv.QUOTE_ALL)
            writer.writerow(['label', 'epsg', 'geometry'])
            epsg = epsg_original
            for index, feature in enumerate(layer.getFeatures(), start=1):
                geom = feature.geometry()
                geom = self.convert_to_simple_XY(geom)
                if simplify:
                    geom = geom.simplify(tolerance) # zjednodušení geometrie, existuje ještě jedna další metoda geometry.simplifyVWS(tolerance). Vyzkoušet, možná.
                if epsg_original not in [wgs84, jtsk]:
                    geom.transform(transformer)
                    epsg = wgs84
                wkt = geom.asWkt()
                label = f'geom_{index}'
                epsg_number = self._get_epsg_number(epsg)
                writer.writerow([label, epsg_number, wkt])

        # Zobrazení zprávy s tlačítkem 
        folder = os.path.dirname(output_file)
        button = QPushButton("Otevři složku")
        button.clicked.connect(lambda: self.open_folder(folder))

        message_widget = self.iface.messageBar().createMessage(f"Vrstva byla exportována do {output_file}")
        message_widget.layout().addWidget(button)  
        self.iface.messageBar().pushWidget(message_widget, Qgis.Success)

        dialog.accept()

    def convert_to_simple_XY(self, geometry):
        if geometry.isMultipart():
            geometry.convertToSingleType()
        geometry_type = geometry.type()
        if geometry_type == 0:  # Point
            return QgsGeometry.fromPointXY(geometry.asPoint())
        elif geometry_type == 1:  # LineString
            return QgsGeometry.fromPolylineXY(geometry.asPolyline())
        elif geometry_type == 2:  # Polygon
            return QgsGeometry.fromPolygonXY(geometry.asPolygon())
        return geometry

    def _get_layer_names(self):
        return [layer.name() for layer in QgsProject.instance().mapLayers().values() if self.is_vector(layer)]

    def _get_epsg_number(self, crs):
        return crs.split(':')[-1]

    def open_folder(self, folder_path):
        if os.path.exists(folder_path):
            QDesktopServices.openUrl(QUrl.fromLocalFile(folder_path))
        else:
            self.iface.messageBar().pushMessage(
                "Chyba",
                f"Složka {folder_path} neexistuje.",
                level=Qgis.Critical,
                duration=5
            )
