# -*- coding: utf-8 -*-
"""
/***************************************************************************
 PianExporter
                                 A QGIS plugin
 Plugin exportuje vrstvu do formátu pro pian
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-09-26
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Marek Lečbych
        email                : lecbych@uapp.cz
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QUrl
from qgis.core import QgsProject, QgsCoordinateTransform, QgsCoordinateReferenceSystem, Qgis, QgsMessageLog, QgsMapLayer, QgsGeometry
from qgis.PyQt.QtGui import QIcon, QDesktopServices
from qgis.PyQt.QtWidgets import QAction, QFileDialog, QInputDialog, QPushButton

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .pian_exporter_dialog import PianExporterDialog
import os.path
import csv


class PianExporter:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'PianExporter_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Pian Exporter')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('PianExporter', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/pian_exporter/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Pian Exporter'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Pian Exporter'),
                action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        # Konstanty, proměnné
        jtsk = 'EPSG:5514'
        wgs84 = 'EPSG:4326'

        # Výběr vrstvy
        layer_names = self._get_layer_names()
        layer_name, ok = QInputDialog.getItem(self.iface.mainWindow(), "Vyber vrstvu k exportu", "Vrstva:", layer_names, 0, False)
        
        if not layer_name or not ok:
            return
        
        # Kontroly vrstvy
        layer = QgsProject.instance().mapLayersByName(layer_name)[0]
        if not layer:
            self.iface.messageBar().pushCritical("Chyba", "Vrstva nenalezena! Zkus to znovu.")
            return
        
        if layer.type() == QgsMapLayer.RasterLayer:
            self.iface.messageBar().pushCritical("Chyba", "Vybraná vrstva je rastr! Zkus to znovu.")
            return

        # Zjištění CRS vrstvy a EPSG
        crs_original_layer = layer.crs()
        epsg_original = crs_original_layer.authid()
        QgsMessageLog.logMessage(f'Souřadnicový systém vybrané vrstvy je: {epsg_original}.', 'Zprávy', level=Qgis.Info)

        # Získání cesty původního souboru
        original_file_path = layer.source()

        # Nastavení výchozí cesty pro uložení
        default_dir = os.path.dirname(original_file_path)  # Získání složky původního souboru
        default_file = os.path.join(default_dir, f"{layer_name}_pian.csv")  # Název nového souboru s cestou

        # Nastavení cílového CRS
        target_crs = QgsCoordinateReferenceSystem(wgs84)
        transformer = QgsCoordinateTransform(layer.crs(), target_crs, QgsProject.instance().transformContext())
        
        # Uložení do CSV
        output_file, _ = QFileDialog.getSaveFileName(self.iface.mainWindow(), "Save CSV", default_file, "CSV Files (*.csv)")
        if not output_file:
            return

        # Export geometrie do CSV
        with open(output_file, mode='w', newline='', encoding='utf-8') as csvfile:
            writer = csv.writer(csvfile,  quoting=csv.QUOTE_ALL)
            writer.writerow(['label', 'epsg', 'geometry'])
            epsg = epsg_original # Bug1 proměnná pro správný zápis epsg do csv, po případné transformaci se přepíše  
            for index, feature in enumerate(layer.getFeatures(), start=1):
                geom = feature.geometry()
                geom = self.convert_to_simple_XY(geom) # Bug2 problém s multigeometriemi
                if (epsg_original not in [wgs84, jtsk]):
                    geom.transform(transformer)
                    epsg = wgs84
                wkt = geom.asWkt()
                label = f'geom_{index}'
                epsg_number = self._get_epsg_number(epsg)
                writer.writerow([label, epsg_number , wkt])
        
        # Zobrazení zprávy s tlačítkem pro otevření složky
        folder = os.path.dirname(output_file)
        button = QPushButton("Otevři složku")
        button.clicked.connect(lambda: self.open_folder(folder))

        # Vytvoření widgetu pro zprávu
        message_widget = self.iface.messageBar().createMessage(f"Vrstva byla exportována do {output_file}")
        message_widget.layout().addWidget(button)  

        # Zobrazení zprávy
        self.iface.messageBar().pushWidget(message_widget, Qgis.Success)

    # převede na jednoduchou geometrii a odstraní případné Z a M hodnoty
    def convert_to_simple_XY(self, geometry):
        if geometry.isMultipart():
            geometry.convertToSingleType()
        geometry_type = geometry.type()
        # QGisu (asi) chybí metody jako hasZ() a hasM() tak to pro jistotu převedeme
        if geometry_type == 0:  # Body
            return QgsGeometry.fromPointXY(geometry.asPoint())
        elif geometry_type == 1:  # LineString
            return QgsGeometry.fromPolylineXY(geometry.asPolyline())
        elif geometry_type == 2:  # Polygon
            return QgsGeometry.fromPolygonXY(geometry.asPolygon())
        return geometry

    def _get_layer_names(self):
        return [layer.name() for layer in QgsProject.instance().mapLayers().values()]

    def open_folder(self, folder_path):
        QDesktopServices.openUrl(QUrl.fromLocalFile(folder_path))
    
    def _get_epsg_number(self, crs):
        return crs.split(':')[-1]
