# -*- coding: utf-8 -*-

"""
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

from qgis.core import (QgsProcessingParameterRasterDestination,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingException)

from pcraster_tools.processing.algorithm import PCRasterAlgorithm


class PCRasterNodirectionAlgorithm(PCRasterAlgorithm):
    """
    Cells with no direction (e.g. flat) get boolean TRUE and with direction get boolean FALSE
    """

    INPUT_RASTER = 'INPUT'
    OUTPUT_RASTER = 'OUTPUT'

    def createInstance(self):  # pylint: disable=missing-function-docstring
        return PCRasterNodirectionAlgorithm()

    def name(self):  # pylint: disable=missing-function-docstring
        return 'nodirection'

    def displayName(self):  # pylint: disable=missing-function-docstring
        return self.tr('nodirection')

    def group(self):  # pylint: disable=missing-function-docstring
        return self.tr('Missing value creation, detection, alteration')

    def groupId(self):  # pylint: disable=missing-function-docstring
        return 'missingvalues'

    def shortHelpString(self):  # pylint: disable=missing-function-docstring
        return self.tr(
            """Cells with no direction (e.g. flat) get boolean TRUE and with direction get boolean FALSE

            <a href="{}">PCRaster documentation</a>

            Parameters:

            * <b>Input directional raster</b> (required) - Raster layer with directional data type (e.g. slope or aspect)
            * <b>Output raster</b> (required) - Boolean raster with True for cells without direction and FALSE for cells with direction
            """
        ).format(PCRasterAlgorithm.documentation_url('op_nodirection.html'))

    def initAlgorithm(self, config=None):  # pylint: disable=missing-function-docstring,unused-argument
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_RASTER,
                self.tr('Directional raster layer')
            )
        )

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT_RASTER,
                self.tr("Output no direction raster")
            )
        )

    def processAlgorithm(self, parameters, context, feedback):  # pylint: disable=missing-function-docstring,unused-argument,too-many-locals
        try:
            from pcraster import (   # pylint: disable=import-outside-toplevel
                setclone,
                readmap,
                report,
                nodirection
            )
        except ImportError as e:
            raise QgsProcessingException('PCRaster library is not available') from e

        input_raster = self.parameterAsRasterLayer(parameters, self.INPUT_RASTER, context)
        setclone(input_raster.dataProvider().dataSourceUri())
        InputRaster = readmap(input_raster.dataProvider().dataSourceUri())
        ResultLayer = nodirection(InputRaster)
        outputFilePath = self.parameterAsOutputLayer(parameters, self.OUTPUT_RASTER, context)

        report(ResultLayer, outputFilePath)

        self.set_output_crs(output_file=outputFilePath, crs=input_raster.crs(), feedback=feedback, context=context)

        return {self.OUTPUT_RASTER: outputFilePath}
