# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ParkingSpacePotential 2.0
                                 A QGIS plugin
 This plugin analyzes the location and capacity of public parking garages and 
 helps to identify street side parking which could be shifted into them.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-07-05
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Lisa-Marie Jalyschko
        email                : lisamarie.jalyschko@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from unittest.mock import inplace

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QRectF, QObject, QVariant
from qgis.PyQt.QtGui import QIcon, QColor
from qgis.PyQt.QtWidgets import QAction, QMenu, QMainWindow, QFileDialog, QCompleter, QSlider
from qgis.PyQt import uic, QtWidgets
from qgis.utils import iface
from qgis.core import QgsProject, QgsVectorLayer, QgsPoint, QgsPointXY, QgsGeometry, QgsMapRendererJob, QgsWkbTypes, QgsField, QgsFillSymbol, QgsProcessingFeatureSourceDefinition, QgsFeatureRequest, QgsMapLayerProxyModel, QgsRasterLayer, QgsCoordinateReferenceSystem, QgsSymbol, QgsRuleBasedRenderer, QgsFeature, NULL, QgsVectorLayer, QgsVectorFileWriter, QgsCoordinateTransformContext
from qgis.gui import QgsMapCanvas, QgsVertexMarker, QgsMapCanvasItem, QgsMapMouseEvent, QgsRubberBand, QgsMapToolPan, QgsMapToolIdentifyFeature, QgsFileWidget
from functools import partial
from qgis import processing

import requests
import time
import json
import pandas as pd
import geopandas as gpd
from shapely.geometry import Point
import matplotlib.pyplot as plt
from urllib.request import urlopen

# Initialize Qt resources from file resources.py
# from .resources import *
# Import the code for the dialog
from .parking_spaces_dialog import ParkingSpacesDialog
from .parking_spaces_second_dialog import ParkingSpacesSecondDialog
import os.path


class ParkingSpaces:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ParkingSpacePotential_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&ParkingSpacePotential 2.0')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ParkingSpacePotential', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/parking_spaces/icon-parking-garage.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Show shiftable parking space'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Parking Space Potential'),
                action)
            self.iface.removeToolBarIcon(action)
            
    def callback(self, dialog, feature, parkList, markList):

        """Code called when the feature is selected by the user"""

        if feature in parkList :
            print ("INFO: {} is already selected!".format(feature.attribute("name")))

        else :
            print("INFO: You selected feature {}".format(feature.attribute("name")))
      
            m = QgsVertexMarker(dialog.canvasWidget)
            m.setColor(QColor(0, 255, 0))
            m.setIconSize(5)
            m.setIconType(QgsVertexMarker.ICON_BOX) # or ICON_CROSS, ICON_X
            m.setPenWidth(3)

            m.setCenter(feature.geometry().asPoint())

            markList.append(m)
            parkList.append(feature)
     
    def map_refresher(self) :
        self.parkhausneu = self.dlg.mMapLayerComboBox.currentLayer()
        
        s = QSettings()
        oldValidation = s.value( "/Projections/defaultBehavior" )
        s.setValue( "/Projections/defaultBehavior", "useGlobal" )
        
        tms = 'type=xyz&url=https://tile.openstreetmap.org/{z}/{x}/{y}.png&zmax=19&zmin=0'
        rasterLyr = QgsRasterLayer(tms, 'OSM', 'wms')
        rasterLyr.setCrs(QgsCoordinateReferenceSystem(3857))
        
        self.dlg.canvasWidget.enableAntiAliasing(True)
        #self.dlg.canvasWidget.setDestinationCrs(QgsCoordinateReferenceSystem(4326))
        self.dlg.canvasWidget.setDestinationCrs(self.parkhausneu.crs())
        self.dlg.canvasWidget.setLayers([self.parkhausneu, rasterLyr])
        self.dlg.canvasWidget.setExtent(self.parkhausneu.extent())
        
        self.dlg.canvasWidget.refresh()
    
        self.dlg.canvasWidget.waitWhileRendering()
        self.feature_identifier.setLayer(self.parkhausneu)
        self.parkhaeuser = self.parkhausneu

        print("INFO: Map refreshed!")
        
    def pp_refresher(self):
        self.pubpark = self.dlg.mMapLayerComboBox_2.currentLayer()
        print("INFO: Layer containing public parking space refreshed!")
        
    def network_refresher(self):
        self.network = self.dlg.mMapLayerComboBox_3.currentLayer()
        print("INFO: Network refreshed!")
        
    def ext_refresher(self, dialog, parkhausLayer, rasterLayer) :
        print("INFO: Extents changed and refreshed!")
        
        dialog.canvasWidget.enableAntiAliasing(True)
        #self.dlg.canvasWidget.setDestinationCrs(QgsCoordinateReferenceSystem(4326))
        dialog.canvasWidget.setDestinationCrs(parkhausLayer.crs())
        dialog.canvasWidget.setLayers([parkhausLayer, rasterLayer])
        dialog.canvasWidget.refresh()
        dialog.canvasWidget.waitWhileRendering()

    def is_geojson(self, data) -> bool:
        """
        Universally and reliably checks if a JSON object conforms to the GeoJSON standard.
        Args:
            data (Any): A JSON object (already loaded as Python dict or list).
        Returns:
            bool: True if the object is GeoJSON-compliant, False otherwise.
        """
        geojson_geometry_types = {
            "Point", "MultiPoint", "LineString", "MultiLineString",
            "Polygon", "MultiPolygon", "GeometryCollection"}

        if not isinstance(data, dict):              # Ensure the input is a dictionary (valid JSON object)
            return False

        json_type = data.get('type', None)

        if json_type == 'FeatureCollection':        # Handle FeatureCollection, ensuring all features are GeoJSON compliant
            return (
                    isinstance(data.get('features', None), list)
                    and all(self.is_geojson(feature) for feature in data['features']))

        if json_type == 'Feature':                  # Handle individual Feature objects
            geometry = data.get('geometry', None)
            return geometry is None or self.is_geojson(geometry)

        if json_type in geojson_geometry_types:     # Check standard geometry types for required structure
            coordinates = data.get('coordinates', None)
            if coordinates is not None:
                return True

            if json_type == 'GeometryCollection':   # Special handling for GeometryCollection type
                geometries = data.get('geometries', None)
                return (
                        isinstance(geometries, list)
                        and all(self.is_geojson(geom) for geom in geometries))

        return False

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = ParkingSpacesDialog()

        dialog = self.dlg

        self.dlg.mMapLayerComboBox.setFilters(QgsMapLayerProxyModel.PointLayer)
        #self.dlg.mMapLayerComboBox_2.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.dlg.mMapLayerComboBox_3.setFilters(QgsMapLayerProxyModel.LineLayer)

        self.dlg.verticalSlider.setMinimum(0)
        self.dlg.verticalSlider.setMaximum(100)
        self.dlg.verticalSlider.setTickPosition(QSlider.TicksBothSides)
        self.dlg.verticalSlider.setTickInterval(20)
        self.dlg.verticalSlider.setSingleStep(10)

        self.dlg.mQgsFileWidget.setStorageMode(QgsFileWidget.GetDirectory)  # set storage mode for output folder selection
        
        # Lists for autocompletion
        suggested_links = [
            'https://api.dashboard.smartcity.ms/parking', # Münster
            'https://services.gis.konstanz.digital/geoportal/rest/services/Fachdaten/Parkplaetze_Parkleitsystem/MapServer/0/query?where=1%3D1&outFields=*&outSR=4326&f=json', # Konstanz
            'https://api.datenplattform.heidelberg.de/ckan/or/mobility/main/offstreetparking/v2/entities?limit=50&api-key=a337781b-e2b1-4834-8e24-1790ded83be0', # Heidelberg
            'http://open-data.dortmund.de/api/explore/v2.1/catalog/datasets/parkhauser/records?limit=20', # Dortmund
        ]
        suggested_paths_attributes = [
            'features' # Konstanz und Münster
        ]
        suggested_paths_name = [
            'attributes,name', # Konstanz
            'properties,NAME' # Münster
        ]
        suggested_paths_capacity = [
            'attributes,max_cap', # Konstanz
            'properties,parkingTotal' # Münster
        ]
        suggested_paths_free_capacity = [
            'attributes,real_fcap',  # Konstanz
            'properties,parkingFree'  # Münster
        ]
        suggested_paths_x = [
            'geometry,x',  # Konstanz
        ]
        suggested_paths_y = [
            'geometry,y',  # Konstanz
        ]

        def completer_setter (suggestion_list) :
            completer =  QCompleter(suggestion_list)
            completer.setCaseSensitivity(Qt.CaseInsensitive)
            completer.setFilterMode(Qt.MatchContains)

            return completer

        self.dlg.mLineEdit_2.setCompleter(completer_setter(suggested_links))
        self.dlg.mLineEdit_9.setCompleter(completer_setter(suggested_paths_attributes))
        self.dlg.mLineEdit_3.setCompleter(completer_setter(suggested_paths_name))
        self.dlg.mLineEdit_5.setCompleter(completer_setter(suggested_paths_capacity))
        self.dlg.mLineEdit_6.setCompleter(completer_setter(suggested_paths_free_capacity))
        self.dlg.mLineEdit_7.setCompleter(completer_setter(suggested_paths_x))
        self.dlg.mLineEdit_8.setCompleter(completer_setter(suggested_paths_y))

        if self.dlg.checkBox_5.isChecked() == False :                       # if no API request
            self.parkhaeuser = self.dlg.mMapLayerComboBox.currentLayer()
        self.pubpark = self.dlg.mMapLayerComboBox_2.currentLayer()
        self.network = self.dlg.mMapLayerComboBox_3.currentLayer()

        s = QSettings()
        oldValidation = s.value( "/Projections/defaultBehavior" )
        s.setValue( "/Projections/defaultBehavior", "useGlobal" )
        
        tms = 'type=xyz&url=https://tile.openstreetmap.org/{z}/{x}/{y}.png&zmax=19&zmin=0'
        rasterLyr = QgsRasterLayer(tms, 'OSM', 'wms')
        rasterLyr.setCrs(QgsCoordinateReferenceSystem(3857))
        
        self.dlg.canvasWidget.enableAntiAliasing(True)
        #self.dlg.canvasWidget.setDestinationCrs(QgsCoordinateReferenceSystem(4326))
        self.dlg.canvasWidget.setDestinationCrs(self.parkhaeuser.crs())
        self.dlg.canvasWidget.setLayers([self.parkhaeuser, rasterLyr])
        self.dlg.canvasWidget.setExtent(self.parkhaeuser.extent())
     
        self.dlg.canvasWidget.refresh()
        self.dlg.canvasWidget.waitWhileRendering()
        
        self.dlg.canvasWidget.extentsChanged.connect(lambda : self.ext_refresher(dialog, self.parkhaeuser, rasterLyr))
        
        parkList = []
        markList = []
        
        self.feature_identifier = QgsMapToolIdentifyFeature(self.dlg.canvasWidget)
        # indicates the layer on which the selection will be done
        self.feature_identifier.setLayer(self.parkhaeuser)
        # use the callback as a slot triggered when the user identifies a feature
        self.feature_identifier.featureIdentified.connect(lambda feature: self.callback(dialog, feature, parkList, markList))
       
        # activation of the map tool
        self.dlg.canvasWidget.setMapTool(self.feature_identifier)
        #self.dlg.canvasWidget.setMapTool(self.panTool)
        
        # show the dialog
        self.dlg.show()            
        
        # trigger a map refresh when a new layer is selected in MapLayerComboBox
        self.dlg.mMapLayerComboBox.layerChanged.connect(self.map_refresher)
        self.dlg.mMapLayerComboBox_2.layerChanged.connect(self.pp_refresher) # ... and refresh for public parking space layer
        self.dlg.mMapLayerComboBox_3.layerChanged.connect(self.network_refresher) # ... and refresh for network layer

        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed

        for ver in markList :
            self.dlg.canvasWidget.scene().removeItem(ver) # remove vertex markers
        
        if result:
            # Do something useful here

            v1 = QgsVectorLayer("Point", "Selected parking facilities", "memory")
            pr = v1.dataProvider()
            
            radius = self.dlg.mLineEdit.value()
            if self.dlg.mLineEdit.isNull() == True :
                print ("WARNING: There is no radius specified! The plugin will assume a radius of 300 Metres.")
                radius = 300
            if int(radius) <= 0 :
                print ("WARNING: The radius ", radius , " is invalid! The plugin will assume a radius of 300 Metres.")
                radius = 300
                    
            if self.dlg.checkBox_5.isChecked() == True :
                # URL for API
                url = self.dlg.mLineEdit_2.value()

                # API request
                response = requests.get(url)
                data = response.json()

                # recursive function to fit a variety of different data structures
                def extract_data(json_obj, keys) :
                    """
                    :param json_obj: The JSON-Object (Dict)
                    :param keys: List of key, which indicate the path to the desired value
                    :return: The extracted value or None if not found
                    """
                    if not keys:
                        return json_obj

                    key = keys[0]
                    
                    if isinstance(json_obj, dict) and key in json_obj : 
                        return extract_data(json_obj[key], keys[1:])

                    elif isinstance(json_obj, dict) and keys[1:] in json_obj[key] :
                        return extract_data(json_obj[key][keys[1:]], keys[2:])

                    else :
                        return None

                path_a = self.dlg.mLineEdit_9.value().split(",")
                records = extract_data(data, path_a)

                path_name = self.dlg.mLineEdit_3.value().split(",")
                path_cap = self.dlg.mLineEdit_5.value().split(",")
                path_free = self.dlg.mLineEdit_6.value().split(",")
                path_geoX = self.dlg.mLineEdit_7.value().split(",")
                path_geoY = self.dlg.mLineEdit_8.value().split(",")

                if self.is_geojson(data):                           # check if GeoJSON

                    print('INFO: The Data Response from the API is GeoJSON!')
                    gdf = gpd.read_file(url)  # read GeoJSON directly from URL

                    gdf = gdf[[path_name[-1], path_cap[-1], path_free[-1], 'geometry']]
                    new_colnames = {path_name[-1]:'name', path_cap[-1]:'capacity', path_free[-1]:'free'}
                    gdf.rename(columns=new_colnames, inplace=True)

                    if 'utilized' not in gdf.columns:
                        gdf['utilized'] = None # initialize field, float, two decimals
                        gdf['utilized'] = gdf['utilized'].astype('float64')
                        gdf['utilized'] = gdf['utilized'].round(decimals=2)

                    for index, row in gdf.iterrows():
                        gdf.at[index, 'utilized'] = round(1- (int(gdf.at[index, 'free']) or 0) / int(gdf.at[index, 'capacity']),2)

                    lyr = QgsVectorLayer(gdf.to_json(), "PH_from_API_GeoJSON", "ogr")
                    QgsProject.instance().addMapLayer(lyr)

                    layers = QgsProject.instance().mapLayersByName(lyr.name())
                    exist = True if layers and len(layers) == 1 else False

                else:
                    print('INFO: The Data Response from the API is *not* GeoJSON!')

                    # Write to DataFrame with geo-field
                    df = pd.DataFrame([{
                        "name": extract_data(rec, path_name),
                        "capacity": extract_data(rec, path_cap),
                        "free": extract_data(rec, path_free),
                        "utilized": float((round(1 - int(extract_data(rec, path_free) or 0) / int(extract_data(rec, path_cap) or 0),2))) or 0,
                        "geometry": Point(extract_data(rec, path_geoX), extract_data(rec, path_geoY))
                        } for rec in records])

                    # Write to GeoDataFrame
                    gdf = gpd.GeoDataFrame(df, geometry="geometry", crs="EPSG:4326")

                    lyr = QgsVectorLayer(gdf.to_json(), "PH_from_API_JSON", "ogr")

                    if QgsProject.instance().mapLayersByName(lyr.name()) :  # check, if there is already a layer with that name
                        print('WARNING: There is already a layer named "PH_from_API_JSON". Please rename the layer or delete it.')
                        lyr.setName(lyr.name() + '_new')
                        QgsProject.instance().addMapLayer(lyr)

                    else :
                        QgsProject.instance().addMapLayer(lyr)

                    layers = QgsProject.instance().mapLayersByName(lyr.name())
                    exist = True if layers else False

                if exist :
                    print('INFO: The Layer was loaded from API successfully: ' + str(exist))
                    self.parkhaeuser = layers[0]
                    self.second_dlg = ParkingSpacesSecondDialog()
                    dialog = self.second_dlg

                    self.second_dlg.canvasWidget.enableAntiAliasing(True)
                    # self.dlg.canvasWidget.setDestinationCrs(QgsCoordinateReferenceSystem(4326))
                    self.second_dlg.canvasWidget.setDestinationCrs(self.parkhaeuser.crs())
                    self.second_dlg.canvasWidget.setLayers([self.parkhaeuser, rasterLyr])
                    self.second_dlg.canvasWidget.setExtent(self.parkhaeuser.extent())

                    self.second_dlg.canvasWidget.refresh()
                    self.second_dlg.canvasWidget.waitWhileRendering()

                    self.second_dlg.canvasWidget.extentsChanged.connect(lambda: self.ext_refresher(dialog, self.parkhaeuser, rasterLyr))

                    self.feature_identifier = QgsMapToolIdentifyFeature(self.second_dlg.canvasWidget)
                    # indicates the layer on which the selection will be done
                    self.feature_identifier.setLayer(self.parkhaeuser)
                    # use the callback as a slot triggered when the user identifies a feature
                    self.feature_identifier.featureIdentified.connect(
                        lambda feature: self.callback(dialog, feature, parkList, markList))

                    # activation of the map tool
                    self.second_dlg.canvasWidget.setMapTool(self.feature_identifier)
                    # self.dlg.canvasWidget.setMapTool(self.panTool)

                    self.second_dlg.show()

                else :
                    print('ERROR: Loading Layer from API was unsuccessful or there are multiple layers with the same name')

                result_2 = self.second_dlg.exec_()

                if result_2 :
                    print(parkList)

            for feat in parkList:
                for field in feat.fields():
                    pr.addAttributes([QgsField(field.name(), field.type())]) # add all the other attributes from input layer
                    
                v1.updateFields()
                pr.addFeature(feat)

            v1.updateExtents()
            
            distance = processing.run("native:serviceareafromlayer", 
                {'INPUT': self.network,
                'STRATEGY':0,                                                       # 0: by distance, 1: by travel time
                'DIRECTION_FIELD':'',
                'VALUE_FORWARD':'',
                'VALUE_BACKWARD':'',
                'VALUE_BOTH':'',
                'DEFAULT_DIRECTION':2,
                'SPEED_FIELD':'',
                'DEFAULT_SPEED':50,                                                 # in km/h, not relevant if by distance
                'TOLERANCE':0,
                'START_POINTS': v1,
                'TRAVEL_COST': radius,                                              # max. distance in meters or in hours
                'INCLUDE_BOUNDS': False,
                'OUTPUT_LINES':'TEMPORARY_OUTPUT'})
            
            area = processing.run("native:convexhull",
                {'INPUT':distance['OUTPUT_LINES'],
                'OUTPUT':'TEMPORARY_OUTPUT'})
                           
            a_dp = area['OUTPUT'].dataProvider() 
            a_dp.addAttributes([QgsField("shifted", QVariant.Int)])
            a_dp.addAttributes([QgsField("new_util", QVariant.Double)])
                    
            area['OUTPUT'].updateFields()
            QgsProject.instance().addMapLayer(area['OUTPUT'])
            
            selection = processing.run("native:selectbylocation",
                {'INPUT' : self.pubpark,
                'INTERSECT' : area['OUTPUT'],
                'METHOD' : 0,
                'PREDICATE' : [5,6] })                                              # overlaps & are within
            
            centroids = processing.run("native:centroids",
                {'ALL_PARTS': False,
                'INPUT' : QgsProcessingFeatureSourceDefinition(self.pubpark.id(), selectedFeaturesOnly=True),
                'OUTPUT' : 'TEMPORARY_OUTPUT'})
            
            waitList = []
            resultList = []
            partialList = []
            lowList = []
            
            dupList = []
            
            nearDict = {}
            resultDict = {}
            dupDict = {}
            capDict = {}
            restrDict = {}
            indDict = {}
            
            nom = 1
            
            for feat in v1.getFeatures():
                start_point_geom = feat.geometry()
                
                shortest = processing.run("native:shortestpathpointtolayer",
                    {'INPUT': self.network,
                    'STRATEGY':0,
                    'DIRECTION_FIELD':'',
                    'VALUE_FORWARD':'',
                    'VALUE_BACKWARD':'',
                    'VALUE_BOTH':'',
                    'DEFAULT_DIRECTION':2,
                    'SPEED_FIELD':'',
                    'DEFAULT_SPEED':50,
                    'TOLERANCE':0,
                    'START_POINT': start_point_geom,
                    'END_POINTS': centroids['OUTPUT'],
                    'OUTPUT':'TEMPORARY_OUTPUT'})
                    
                sh_dp = shortest['OUTPUT'].dataProvider()  
                sh_dp.addAttributes([QgsField("shiftable_into", QVariant.String)])
                    
                shortest['OUTPUT'].updateFields()
                
                dfeatIds = []                                                       # creating a list for the features to be deleted
                
                for f in shortest['OUTPUT'].getFeatures():
                    if f['cost'] > int(radius) :
                        dfeatIds.append(f.id())
                        
                def get_cost(feat):
                    return feat['cost']   
               
                shortest['OUTPUT'].dataProvider().deleteFeatures(dfeatIds)
                shortest['OUTPUT'].triggerRepaint()
                
                nearList = sorted(shortest['OUTPUT'].getFeatures(), key = get_cost)
                        
                nearDict[nom] = nearList
                print("INFO: Parking garage " , feat['name'] , " corresponds to number " , nom)
              
                nom += 1
            
            geotype = QgsWkbTypes.displayString(self.pubpark.wkbType())
            print('\nINFO: Der Geotyp der Parkplatzdaten ist' , geotype)
            

            
            for item in nearDict.items() :                                          # key (item[0]) = nom, value (item[1]) = list of features
                for f in item[1] :
                    checkID = f['id']
                    f['shiftable_into'] = str(item[0])                              # get key and add as attribute
                    
                    if checkID in resultDict.keys() :
                        oldF = resultDict.get(checkID)
                        
                        if f['cost'] < oldF['cost'] :
                            #dupList
                            dupDict[oldF['id']] = resultDict[checkID]               # add duplets to seperate dict
                            del resultDict[checkID]                                 # delete old feature
                            resultDict[f['id']] = f                                 # replace with new one
                            
                        else :
                            dupDict[checkID] = f                                    # add duplets to seperate dict
                        
                    else :
                        resultDict[f['id']] = f                                     # add to resultDict

            restrictionList = []

            if self.dlg.checkBox.isChecked() == True :                              # check for ,disabled' flag
                restrictionList.append('d_cap')
            if self.dlg.checkBox_2.isChecked() == True :                            # check for ,short-term' flag
                restrictionList.append('st_cap')
            if self.dlg.checkBox_3.isChecked() == True :                            # check for ,residential' flag
                restrictionList.append('r_cap')
            if self.dlg.checkBox_4.isChecked() == True :                            # check for ,loading and delivery' flag
                restrictionList.append('ld_cap')
            if self.dlg.checkBox_9.isChecked() == True :                            # check for ,charging station' flag
                restrictionList.append('e_cap')

            nom2 = 1
            
            for feat in v1.getFeatures():                                           # for each Parkhaus
                capCount = int(0)
                u_cap = int(round(int(feat['capacity']) * (float(feat['utilized'])))) # utilization in absolute numbers (util)

                if self.dlg.checkBox_7.isChecked() == True :
                    tU = float(self.dlg.verticalSlider.value() / 100)
                    free_cap = (int(feat['capacity']) * tU) - u_cap

                elif v1.fields().indexOf('target_util') == -1 :                   # check, whether field 'target_util' exists (index = -1 means it doesn't)
                    tU = 1
                    free_cap = int(feat['capacity']) - u_cap

                else :
                    tU = float(feat['target_util'])
                    free_cap = (int(feat['capacity']) * tU) - u_cap

                print('\nINFO: The calculated free capacity for ' , feat['name'] , ' is ' , free_cap , '. The target utilization is ' , int(tU * 100) , '%.')

                endresult = QgsVectorLayer(geotype, "Parking Spaces", "memory")
                dp = endresult.dataProvider()

                for resF in resultDict.values() :                                   # ... get all features ...
                    for field in resF.fields():
                        dp.addAttributes([QgsField(field.name(), field.type())])
                        
                    endresult.updateFields()
                    
                    if resF['shiftable_into'] == str(nom2) :                        # ... which are 'shiftable into' it ...
                        permittedCap = 0
                        permittedCap += int(resF['capacity'])

                        for restriction in restrictionList :
                            if resF.fieldNameIndex(restriction) == -1 :
                                print('ERROR: There is no field ', restriction,
                                      ' and thus no restricted capacity specified. The plugin will assume a capacity restriction of 0.')
                            else :
                                if resF[restriction] != NULL :                          # go through all flagged restrictions...
                                    permittedCap -= int(resF[restriction])              # ... substract them from the total capacity

                        if permittedCap == 0 :                                      # check if no permittedCap left, if yes: add to lowList
                            lowList.append(resF)
                            capCount += permittedCap                                # should be +0 but just to be sure...

                        elif permittedCap < int(resF['capacity']) :                 # check whether cap was reduced by restrictions and pp is therefore not fully shiftable
                            restrDict[resF['id']] = permittedCap                    # add  to restrDict to check waitList later
                            print('INFO: ' + resF['id'] + ' with a permittedCap of ' + str(permittedCap) + ' was added to the restrDict.')
                            if capCount + permittedCap <= free_cap :                # if free_cap big enough, add to partialList immediately
                                partialList.append(resF)
                                capCount += permittedCap
                            else :
                                print('INFO: The restricted PP ' + str(resF['id']) + ' was added to the waitList!')

                        else :
                            if capCount + permittedCap <= free_cap :                    # ... and check the capacity
                                resultList.append(resF)                                 # add to resultList
                                capCount += permittedCap

                            else :
                                waitList.append(resF)
                                print ('INFO: ' , resF['id'] , ' was added to the waitList. The capCount for ', feat['name'], 'is currently ' , capCount , ' with a free capacity of' , free_cap , '.')
                            
                free = int(free_cap - capCount)
                indList = [u_cap, free_cap, tU]                                         # List of indeces for final result

                capDict[nom2] = free
                indDict[nom2] = indList

                nom2 += 1             
                
            for waitF in waitList :                                                 # check, whether parking spaces which don't fit in
                waitID = waitF['id']                                                #  nearest garage could be shifted into another
                
                if waitID in dupDict.keys() :
                    dupF = dupDict.get(waitID)
                    capacity = capDict.get(int (dupF['shiftable_into']))
                    
                    if capacity == 0 :
                        lowList.append(dupF)
                        
                    else :
                        if waitID in restrDict.keys() and restrDict.get(waitID) <= capacity :
                            partialList.append(dupF)                                # if parking space fits but has restrictions it is added to partialList (yellow)
                            capacity -= restrDict[waitID]
                            capDict[int(dupF['shiftable_into'])] = capacity
                        elif int(dupF['capacity']) <= capacity :                      # if parking space fits and has no restrictions it is added to resultList (green)
                            resultList.append(dupF)
                            capacity -= int(dupF['capacity'])
                            capDict[int(dupF['shiftable_into'])] = capacity
                            
                            print ("INFO: " , dupF['id'] , " is a duplicate and was appended to " , dupF['shiftable_into'] , "!")
                            print ("INFO: Left capacity for " , dupF['shiftable_into'] , ":" , capDict.get(int(dupF['shiftable_into'])))
                        
                        else :
                            lowList.append(dupF)                                    # if parking space doesn't fit it is added to lowList (red)
                        
                else :
                    lowList.append(waitF)

            lowCount = 0
            
            for l in lowList :
                lowCount += int(l['capacity'])
                
            print("\nINFO: Total parking space capacity within the radius which can *not* be shifted:" , lowCount, "\n")
            
            nom3 = 1
            
            for feat in v1.getFeatures() :
                lowCapCount = 0
                
                for l in lowList :
                    if l['shiftable_into'] == str(nom3) :
                        lowCapCount += int(l['capacity'])
                        
                util = indDict[nom3][0]
                free_cap = indDict[nom3][1]

                leftCap = capDict.get(nom3)
                shift = free_cap - int(leftCap)
                newUtil = round((shift + util) / int(feat['capacity']), 2)

                indDict[nom3].append(shift)             # not necessary?
                indDict[nom3].append(newUtil)           # not necessary?
                
                print ("\nRESULT: Shiftable parking space into " , feat['name'] , ":" , shift)
                print("RESULT: ", feat['name'], " would be utilized by ", float(newUtil) * 100, " %. The target utilization is ", indDict[nom3][2] * 100 , " %")
                print ("RESULT: " , feat['name'] , "is currently utilized by " , float(feat['utilized']) * 100 , " %")
                print("RESULT: Total parking space capacity within the radius of " , feat['name'] , " which can *not* be shifted:" , lowCapCount)
                print ("\n")

                aID = feat['id']
                area['OUTPUT'].selectByExpression("\"id\" = '{}' ".format(aID))
                
                for p in area['OUTPUT'].selectedFeatures() :
                    field_idx_iU = area['OUTPUT'].fields().indexOf('new_util')
                    iU = newUtil
                    attr_value = {field_idx_iU : iU}
                    a_dp.changeAttributeValues({p.id():attr_value})

                    field_idx_s = area['OUTPUT'].fields().indexOf('shifted')
                    sh = shift
                    attr_value = {field_idx_s : sh}
                    a_dp.changeAttributeValues({p.id(): attr_value})
                    
                area['OUTPUT'].commitChanges()
                
                nom3 += 1
                    
            rules = (
                ('Low Utilization', '"new_util" <=0.5', 'green', 0.3),
                ('Mid Utilization', '"new_util" >0.5 AND "new_util" <=0.7 ', 'yellow', 0.3),
                ('High Utilization', '"new_util" >0.7 AND "new_util" <=0.9', 'orange', 0.3),
                ('Max Utilization', '"new_util" >0.9', 'red', 0.3))
                
            newsymbol = QgsSymbol.defaultSymbol(area['OUTPUT'].geometryType())
            newrenderer = QgsRuleBasedRenderer(newsymbol)
                
            root_rule = newrenderer.rootRule()
                
            for label, expression, color_name, opacity in rules:                                                            
                rule = root_rule.children()[0].clone()                              # create a clone (i.e. a copy) of the default rule                                                                    
                rule.setLabel(label)                                                # set the label, expression and color
                rule.setFilterExpression(expression)
                rule.symbol().setColor(QColor(color_name))
                rule.symbol().setOpacity(opacity)
                    
                root_rule.appendChild(rule)
                
            root_rule.removeChildAt(0)
              
            area['OUTPUT'].setRenderer(newrenderer)
            area['OUTPUT'].triggerRepaint()
            area['OUTPUT'].setName('Parking Garages')
                    
            QgsProject.instance().addMapLayer(area['OUTPUT'])
            
            for resF in resultList :            # Adding the features which are completely shiftable (GREEN)
                for field in resF.fields():
                    dp.addAttributes([QgsField(field.name(), field.type())])
                dp.addAttributes([QgsField('result', QVariant.String)])           # add new attribute: 'result'
                endresult.updateFields()

                keyID = resF['id'] 
                self.pubpark.selectByExpression("\"id\" = '{}' ".format(keyID))
                selection = self.pubpark.selectedFeatures()
                
                for f in selection :
                    dp.addFeature(f)
                    field_idx_s = resF.fields().indexOf('shiftable_into')           # connecting to geometries and adding 'shiftable_into'
                    shiftable = resF['shiftable_into']
                    attr_value = {field_idx_s : shiftable}
                    dp.changeAttributeValues({f.id():attr_value})
                    
                    field_idx_c = resF.fields().indexOf('cost')                     # connecting to geometries and adding 'cost'
                    cost = resF['cost']
                    attr_value = {field_idx_c : cost}
                    dp.changeAttributeValues({f.id():attr_value})

                    endresult.commitChanges()

                    field_idx_r = endresult.fields().indexOf('result')
                    endresult.startEditing()
                    endresult.changeAttributeValue(f.id(), field_idx_r, 'yes')

                endresult.commitChanges()
                
            for partF in partialList :
                keyID = partF['id']
                self.pubpark.selectByExpression("\"id\" = '{}' ".format(keyID))
                selection = self.pubpark.selectedFeatures()

                for f in selection:
                    dp.addFeature(f)

                    field_idx_s = partF.fields().indexOf(
                        'shiftable_into')  # connecting to geometries and adding 'shiftable_into'
                    shiftable = partF['shiftable_into']
                    attr_value = {field_idx_s: shiftable}
                    dp.changeAttributeValues({f.id(): attr_value})

                    field_idx_c = partF.fields().indexOf('cost')  # connecting to geometries and adding 'cost'
                    cost = partF['cost']
                    attr_value = {field_idx_c: cost}
                    dp.changeAttributeValues({f.id(): attr_value})

                    endresult.commitChanges()

                    field_idx_r = endresult.fields().indexOf('result')
                    endresult.startEditing()
                    endresult.changeAttributeValue(f.id(), field_idx_r, 'partially')

                endresult.commitChanges()

            for lowF in lowList :                                                   # Adding the features which are NOT shiftable, but spatially close (RED)
                
                keyID = lowF['id']
                self.pubpark.selectByExpression("\"id\" = '{}' ".format(keyID))
                selection = self.pubpark.selectedFeatures()
                
                for f in selection :
                    dp.addFeature(f)
                    
                    field_idx_s = lowF.fields().indexOf('shiftable_into')           # connecting to geometries and adding 'shiftable_into'
                    shiftable = lowF['shiftable_into']
                    attr_value = {field_idx_s : shiftable}
                    dp.changeAttributeValues({f.id():attr_value})
                    
                    field_idx_c = lowF.fields().indexOf('cost')                     # connecting to geometries and adding 'cost'
                    cost = lowF['cost']
                    attr_value = {field_idx_c : cost}
                    dp.changeAttributeValues({f.id():attr_value})

                    endresult.commitChanges()

                    field_idx_r = endresult.fields().indexOf('result')
                    endresult.startEditing()
                    endresult.changeAttributeValue(f.id(), field_idx_r, 'no')

                endresult.commitChanges()

            endresult.updateExtents()

            rules2 = (
                ('Fully shiftable', '"result" = \'yes\'', 'green', 1.4),
                ('Partially shiftable', '"result" = \'partially\'', 'yellow', 1.4),
                ('Not shiftable', '"result" = \'no\'', 'red', 1.4))

            symbolr = QgsSymbol.defaultSymbol(endresult.geometryType())
            rendererr = QgsRuleBasedRenderer(symbolr)

            root_rule2 = rendererr.rootRule()

            for label, expression, color_name, width in rules2:
                rule = root_rule2.children()[0].clone()  # create a clone (i.e. a copy) of the default rule
                rule.setLabel(label)  # set the label, expression and color
                rule.setFilterExpression(expression)
                rule.symbol().setColor(QColor(color_name))

                if geotype == 'MultiLineString' or geotype == 'LineString':
                    rule.symbol().setWidth(width)

                root_rule2.appendChild(rule)

            root_rule2.removeChildAt(0)

            endresult.setRenderer(rendererr)
            endresult.triggerRepaint()
            endresult.setName('Parking Spaces')
                
            endresult.triggerRepaint()

            QgsProject.instance().addMapLayer(endresult)

            if self.dlg.checkBox_6.isChecked() == True :
                options = QgsVectorFileWriter.SaveVectorOptions()
                options.driverName = 'csv'
                options.fileEncoding = "UTF-8"
                timestring = time.strftime("%Y%m%d-%H%M%S")

                if self.dlg.mQgsFileWidget.filePath() == False :
                    print('ERROR: There is no output directory specified! Writing .csv was unsuccessful.')

                else :
                    path_pg = os.path.join(
                        self.dlg.mQgsFileWidget.filePath(),
                        '{}_parking_garages.csv'.format(timestring))

                    path_ps = os.path.join(
                        self.dlg.mQgsFileWidget.filePath(),
                        '{}_parking_spaces.csv'.format(timestring))

                    QgsVectorFileWriter.writeAsVectorFormatV3(endresult, path_ps, QgsCoordinateTransformContext(), options)
                    QgsVectorFileWriter.writeAsVectorFormatV3(area['OUTPUT'], path_pg, QgsCoordinateTransformContext(), options)

                    print('INFO: The result data for the selected parking garages was written to' , path_pg)
                    print('INFO: The result data for the shiftable parking spaces was written to' , path_ps)

            self.pubpark.removeSelection() # just in case...
            area['OUTPUT'].removeSelection()
            self.dlg.close()
           
        else : 
            for ver in markList : 
                self.dlg.canvasWidget.scene().removeItem(ver)