# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ParallelBuffers
                                 A QGIS plugin
 Create lateral buffers from selected lines with distance control.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-08-01
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Javier Moreno Rico
        email                : j.morenorico.formacion@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

from .resources import *
from .ParallelBuffers_dialog import ParallelBuffersDialog

import os.path
from qgis.core import (
    QgsProcessingFeatureSourceDefinition,
    QgsFeatureRequest,
    QgsProject
)
import processing


class ParallelBuffers:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ParallelBuffers_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        self.actions = []
        self.menu = self.tr(u'&ParallelBuffers')
        self.first_start = None

    def tr(self, message):
        return QCoreApplication.translate('ParallelBuffers', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None
    ):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)
        return action

    def initGui(self):
        icon_path = ':/plugins/ParallelBuffers/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Create lateral buffers'),
            callback=self.run,
            parent=self.iface.mainWindow()
        )
        self.first_start = True

    def unload(self):
        for action in self.actions:
            self.iface.removePluginMenu(self.tr(u'&ParallelBuffers'), action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Main logic to create lateral buffers from selected lines."""
        if self.first_start:
            self.first_start = False
            self.dlg = ParallelBuffersDialog()

        self.dlg.show()
        result = self.dlg.exec_()
        if not result:
            return

        # Obtener los valores introducidos por el usuario en la GUI
        offset_value = self.dlg.spinOffsetDistance.value()
        distance_value = self.dlg.spinBufferDistance.value()

        layer = self.iface.activeLayer()
        if not layer:
            self.iface.messageBar().pushMessage("Error", "No layer selected", level=3)
            return

        def eliminar_offset_cero(layer_memoria):
            layer_memoria.startEditing()
            for f in layer_memoria.getFeatures():
                if f['offset'] == 0:
                    layer_memoria.deleteFeature(f.id())
            layer_memoria.commitChanges()
            return layer_memoria

        source = QgsProcessingFeatureSourceDefinition(
            layer.source(),
            selectedFeaturesOnly=True,
            featureLimit=-1,
            geometryCheck=QgsFeatureRequest.GeometryAbortOnInvalid
        )

        # Paralela izquierda (offset negativo)
        paralela_izquierda = processing.run("native:arrayoffsetlines", {
            'INPUT': source,
            'COUNT': 1,
            'OFFSET': -offset_value,
            'SEGMENTS': 8,
            'JOIN_STYLE': 0,
            'MITER_LIMIT': 1,
            'OUTPUT': 'memory:paralela_izquierda'
        })['OUTPUT']
        paralela_izquierda = eliminar_offset_cero(paralela_izquierda)

        buffer_izquierda = processing.run("native:singlesidedbuffer", {
            'INPUT': paralela_izquierda,
            'DISTANCE': distance_value,
            'SIDE': 1,
            'SEGMENTS': 8,
            'JOIN_STYLE': 0,
            'MITER_LIMIT': 2,
            'OUTPUT': 'memory:buffer_izquierda'
        })['OUTPUT']

        # Paralela derecha (offset positivo)
        paralela_derecha = processing.run("native:arrayoffsetlines", {
            'INPUT': source,
            'COUNT': 1,
            'OFFSET': offset_value,
            'SEGMENTS': 8,
            'JOIN_STYLE': 0,
            'MITER_LIMIT': 1,
            'OUTPUT': 'memory:paralela_derecha'
        })['OUTPUT']
        paralela_derecha = eliminar_offset_cero(paralela_derecha)

        buffer_derecha = processing.run("native:singlesidedbuffer", {
            'INPUT': paralela_derecha,
            'DISTANCE': distance_value,
            'SIDE': 0,
            'SEGMENTS': 8,
            'JOIN_STYLE': 0,
            'MITER_LIMIT': 2,
            'OUTPUT': 'memory:buffer_derecha'
        })['OUTPUT']

        # Fusionar los buffers
        merged_buffers = processing.run("native:mergevectorlayers", {
            'LAYERS': [buffer_izquierda, buffer_derecha],
            'CRS': buffer_izquierda.crs().authid(),
            'OUTPUT': 'memory:buffer_final'
        })['OUTPUT']

        QgsProject.instance().addMapLayer(merged_buffers)

        self.iface.messageBar().pushMessage("Info", "Buffers creados correctamente.", level=0)
