#! python3  # noqa: E265

import math
import os

from qgis.core import (
    Qgis,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsCoordinateTransformContext,
    QgsDataSourceUri,
    QgsDistanceArea,
    QgsGeometry,
    QgsProject,
    QgsVectorTileLayer,
)
from qgis.gui import QgsMapCanvas, QgsMapToolPan, QgsVertexMarker
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtGui import QColor

from panoramax.gui.dlg_image_viewer import DlgImageViewer
from panoramax.service.api_client import ApiClient
from panoramax.toolbelt import PlgLogger, PlgOptionsManager

TILED_LAYER_STYLE_PATH = os.path.abspath(
    os.path.join(os.path.dirname(__file__), "../resources/styles/style.json")
)
TILED_LAYER_URI = "&".join(
    [
        f"{key}={value}"
        for key, value in {
            "type": "xyz",
            "url": "https://explore.panoramax.fr/api/map/{z}/{x}/{y}.mvt",
            "styleUrl": f"file://{TILED_LAYER_STYLE_PATH}",
            "zmin": 0,
            "zmax": 15,
        }.items()
    ]
)
# EPSG codes for commonly used CRS
EPSG_3857 = 3857  # EPSG:3857 - Web Mercator
EPSG_4326 = 4326  # EPSG:4326 - WGS84

# Image search radius in meters
IMAGE_SEARCH_RADIUS = 300

# API Url (managed with instance registration dlg in the future)
API_URL = "https://explore.panoramax.fr"


class PanoramaxMapTool(QgsMapToolPan):

    def __init__(self, canvas: QgsMapCanvas, iface):
        """
        Custom map control to display closest panoramax picture from clicked coordinates.
        :param canvas: map canvas
        :type canvas: QgsMapCanvas
        :param iface: qgis interface
        :type iface: QgsInterface
        """
        super().__init__(canvas)
        self.canvas = canvas
        QgsMapToolPan.__init__(self, self.canvas)
        self.iface = iface
        self.drag_start_position = None

        # toolbelt
        self.log = PlgLogger().log
        self.plg_settings_mngr = PlgOptionsManager()

        # map tool
        self.current_picture = None
        self.current_heading = None
        self.dlg_image_viewer = DlgImageViewer(iface.mainWindow())
        self.imageViewerDockArea = Qt.RightDockWidgetArea

        self.dlg_image_viewer.dockLocationChanged.connect(
            self.on_image_dock_location_changed
        )
        self.marker_triangle = QgsVertexMarker(self.canvas)
        self.marker_circle = QgsVertexMarker(self.canvas)
        self.dlg_image_viewer.pano_widget.orientationChanged.connect(
            self.update_marker_orientation
        )
        self.dlg_image_viewer.btn_right.clicked.connect(self.on_btn_right_clicked)
        self.dlg_image_viewer.btn_left.clicked.connect(self.on_btn_left_clicked)

    @property
    def api_client(self):
        current_url = API_URL
        # to manage later :
        # current_url = PlgOptionsManager().get_plg_settings().instance_url
        if not hasattr(self, "_api_client") or self._api_client.base_url != current_url:
            self._api_client = ApiClient(current_url)
        return self._api_client

    def toolName(self):
        return "Panoramax"

    def on_image_dock_location_changed(self, area):
        self.imageViewerDockArea = area

    def update_marker_orientation(self, angle):
        """
        Update map marker orientation
        :param angle: angle between 0° and 360° from the north
        :type angle: float
        """
        self.current_heading = angle
        self.marker_triangle.setRotation(angle)

    def transformed_geom(self, geometry, source_crs, target_crs) -> QgsGeometry:
        """
        Reproject and buffer geom
        :param geometry: Geometry point
        :type geometry: QgsGeometry
        :param source_crs: CRS code source
        :param source_crs: string
        :param target_crs: CRS code target
        :param target_crs: string
        :return: transformed geometry
        :rtype: QgsGeometry
        """
        pt_geom = QgsGeometry().fromWkt(geometry.asWkt())
        xform = QgsCoordinateTransform(source_crs, target_crs, QgsProject.instance())
        pt_geom.transform(xform)
        buffered_geom = pt_geom.buffer(1, 8)
        bb_geom = QgsGeometry.fromRect(buffered_geom.boundingBox())
        return bb_geom

    def canvasPressEvent(self, event):
        """On press mouse"""
        if event.button() != Qt.LeftButton:
            return
        self.drag_start_position = event.pos()

    def canvasMoveEvent(self, event):
        """On move mouse"""
        # pass the event to the parent class to enable panning
        super().canvasMoveEvent(event)
        self.iface.mapCanvas().setCursor(Qt.ArrowCursor)

    def canvasReleaseEvent(self, event):
        """On release mouse"""
        self.iface.mapCanvas().setCursor(Qt.ArrowCursor)
        if event.button() != Qt.LeftButton:
            return
        if self.drag_start_position is None:
            return
        if (event.pos() - self.drag_start_position).manhattanLength() > 3:
            # this was a drag, not a click
            self.drag_start_position = None
            super().canvasReleaseEvent(event)
            return

        # add waiting marker during image download
        prev_cursor = self.iface.mapCanvas().cursor()
        self.iface.mapCanvas().setCursor(Qt.WaitCursor)

        # Handle mouse release events
        point = self.toMapCoordinates(event.pos())
        point_geom = QgsGeometry.fromPointXY(point)

        # Convert the point to metric coordinates
        self.canvas_to_metric_coordinates(point_geom)

        # create buffer around the clicked point
        buffer_segments = 2
        search_buffer_geom = point_geom.buffer(IMAGE_SEARCH_RADIUS, buffer_segments)

        # reproject the buffer geometry from the map canvas CRS to WGS84
        self.metric_to_geographic_coordinates(search_buffer_geom)

        self.log(
            message=self.tr(f"Buffer (geographic): {search_buffer_geom.asWkt()}"),
            log_level=Qgis.MessageLevel.Info,
        )
        # search area as the buffer bbox
        search_area = search_buffer_geom.boundingBox().asWktCoordinates()
        search_area_param = search_area.replace(" ", ",").replace(",,", ",")

        # Get picture infos from API
        try:
            self.log(
                message=self.tr(f"Searching picture with area: {search_area_param}"),
                log_level=Qgis.MessageLevel.Info,
            )
            found_picture = self.api_client.search_picture(search_area_param)
            if found_picture:
                self.current_picture = found_picture
            else:
                return
        except Exception as e:
            self.log(
                message=self.tr(f"Error while searching picture: {str(e)}"),
                log_level=Qgis.MessageLevel.Info,
                push=True,
                duration=5,
            )
            return
        # Get image from url info
        self.display_image()
        self.iface.mapCanvas().setCursor(prev_cursor)
        self.iface.mainWindow().setCursor(Qt.ArrowCursor)

    def canvas_to_metric_coordinates(self, geometry):
        """
        Transforms the given geometry from the current map canvas coordinate reference system to EPSG:3857 (Web Mercator) coordinates.

        Note: The transformation is applied in-place to the provided geometry and the function returns None.

        :param geometry: QgsGeometry, the geometry to be transformed
        :type geometry: QgsGeometry
        :return: None
        :rtype: None
        """

        source_crs = self.canvas.mapSettings().destinationCrs()
        dest_crs = QgsCoordinateReferenceSystem(EPSG_3857)
        canvas_to_3857 = QgsCoordinateTransform(
            source_crs, dest_crs, QgsProject.instance()
        )
        geometry.transform(canvas_to_3857)

    def metric_to_geographic_coordinates(self, geometry):
        source_crs = QgsCoordinateReferenceSystem(EPSG_3857)
        dest_crs = QgsCoordinateReferenceSystem(EPSG_4326)
        source_to_dest = QgsCoordinateTransform(
            source_crs, dest_crs, QgsProject.instance()
        )
        geometry.transform(source_to_dest)

    def geographic_to_canvas_coordinates(self, geometry):
        """
        Transforms the given geometry from WGS84 (EPSG:4326) coordinates to the current map canvas coordinate reference system.
        :param sourceCrs: QgsCoordinateReferenceSystem, the source CRS (WGS84)
        :param destCrs: QgsCoordinateReferenceSystem, the destination CRS (current map canvas CRS)
        :type destCrs: QgsCoordinateReferenceSystem
        :param geometry: QgsGeometry, the geometry to be transformed
        :type geometry: QgsGeometry
        """

        source_crs = QgsCoordinateReferenceSystem(EPSG_4326)  # WGS84
        dest_crs = self.canvas.mapSettings().destinationCrs()
        wgs84_to_canvas = QgsCoordinateTransform(
            source_crs, dest_crs, QgsProject.instance()
        )
        geometry.transform(wgs84_to_canvas)

    def display_image(self):
        """
        Displays the current image in the image viewer dialog, updates sequence information,
        and creates map markers for the image location.

        Handles errors during image and sequence retrieval, and returns early if self.current_picture is None.
        """
        try:
            self.log(
                message=self.tr(
                    f"Getting image from URL: {self.current_picture.get_url()}"
                ),
                log_level=Qgis.MessageLevel.Info,
            )
            image_pixmap = self.api_client.get_image_by_url(
                self.current_picture.get_url()
            )
            self.log(message=f"current_picture_url : {self.current_picture.get_url()}")
        except Exception as e:
            self.log(
                message=self.tr(f"Error while getting image from API: {str(e)}"),
                log_level=Qgis.MessageLevel.Critical,
                push=False,
            )
            return

        # Get Sequence infos
        if not self.current_picture.get_sequence_id():
            self.log(
                message=self.tr("No sequence found for the picture"),
                log_level=Qgis.MessageLevel.Critical,
                push=True,
            )
            return
        try:
            sequence = self.api_client.get_sequence(
                self.current_picture.get_sequence_id()
            )
        except Exception as e:
            self.log(
                message=self.tr(f"Error while getting sequence from API: {str(e)}"),
                log_level=Qgis.MessageLevel.Critical,
                push=True,
            )
            raise
        pictures_count = sequence.get_pictures_count()

        # add or show image viewer widget if not already visible
        if not self.dlg_image_viewer.isVisible():
            try:
                self.iface.addDockWidget(
                    self.imageViewerDockArea, self.dlg_image_viewer
                )
            except Exception as e:
                self.log(
                    message=self.tr(f"Error opening image viewer : {str(e)}"),
                    log_level=Qgis.MessageLevel.Critical,
                )
        else:
            if not self.dlg_image_viewer.isVisible():
                self.dlg_image_viewer.show()
        if self.current_picture is not None:
            self.dlg_image_viewer.set_content_infos(
                self.current_picture, pictures_count
            )

        # Display image in the viewer dialog without given bearing
        self.log(
            message=f"Image field of view: {self.current_picture.get_field_of_view()}",
            log_level=Qgis.MessageLevel.Info,
        )
        self.dlg_image_viewer.set_pixmap(
            image_pixmap, self.current_picture.get_field_of_view()
        )

        # compute the road bearing in the direct way
        road_bearing = self.compute_bearing(
            self.current_picture.get_geom(),
            self.current_picture.get_next_geom(),
        )

        if road_bearing and not self.current_picture.get_heading():
            self.dlg_image_viewer.pano_widget.initial_heading = road_bearing
            self.dlg_image_viewer.initial_orientation = road_bearing

        if not road_bearing and self.current_picture.get_heading():
            self.dlg_image_viewer.pano_widget.initial_heading = (
                self.current_picture.get_heading()
            )
            self.dlg_image_viewer.initial_orientation = (
                self.current_picture.get_heading()
            )

        if road_bearing and self.current_picture.get_heading():
            self.dlg_image_viewer.pano_widget.initial_heading = (
                self.current_picture.get_heading()
            )
            self.dlg_image_viewer.initial_orientation = road_bearing
        # If no road bearing and no heading, keep initial heading to 0
        self.log(
            message=f"Road bearing: {road_bearing}, Picture heading: {self.current_picture.get_heading()}",
            log_level=Qgis.MessageLevel.Info,
        )

        # Center image to the road bearing or the heading explicitely changed by user
        self.dlg_image_viewer.pano_widget.center_on_heading(
            road_bearing if self.current_heading is None else self.current_heading
        )

        # Create marker for the picture location
        marker_geom = QgsGeometry.fromPointXY(self.current_picture.get_geom())
        self.geographic_to_canvas_coordinates(marker_geom)
        self.marker_triangle.setCenter(marker_geom.asPoint())
        self.marker_circle.setCenter(marker_geom.asPoint())
        self.marker_triangle.setIconSize(30)
        self.marker_circle.setIconSize(20)
        self.marker_triangle.setColor(QColor(255, 255, 255))
        self.marker_circle.setColor(QColor(255, 255, 255))
        self.marker_triangle.setFillColor(QColor(26, 35, 126))
        self.marker_circle.setFillColor(QColor(30, 136, 229))
        self.marker_triangle.setIconType(
            QgsVertexMarker.IconType.ICON_INVERTED_TRIANGLE
        )
        self.marker_circle.setIconType(QgsVertexMarker.IconType.ICON_CIRCLE)

        # Set rotation of the map marker to the road bearing
        self.marker_triangle.setRotation(
            float(
                road_bearing if self.current_heading is None else self.current_heading
            )
        )
        self.marker_triangle.setPenWidth(2)
        self.marker_circle.setPenWidth(2)
        self.marker_triangle.setZValue(1000)
        self.marker_circle.setZValue(1001)

    def compute_bearing(self, point1, point2):
        """
        Computes the bearing between two points.

        If either point is None, returns None.

        :param point1: First point as QgsPointXY
        :param point2: Second point as QgsPointXY
        :return: Bearing in degrees, or None if either point is None
        """
        if point1 is None or point2 is None:
            return None

        distance_area = QgsDistanceArea()
        context = QgsCoordinateTransformContext()
        distance_area.setSourceCrs(
            QgsCoordinateReferenceSystem(EPSG_4326), context
        )  # WGS84

        bearing = math.degrees(distance_area.bearing(point1, point2)) % 360
        return bearing

    def on_btn_right_clicked(self):
        """Handle right button click."""
        nav_direction = self.dlg_image_viewer.btn_right.property("to")
        self.nav_to(nav_direction)

    def on_btn_left_clicked(self):
        """Handle left button click."""
        nav_direction = self.dlg_image_viewer.btn_left.property("to")
        self.nav_to(nav_direction)

    def nav_to(self, direction: str):
        """
        Navigate to the next or previous picture in the sequence.
        :param direction: "next" or "prev"
        :type direction: str
        If no picture is available in the sequence, logs a message and returns.
        :return: None
        :rtype: None
        """

        self.iface.mainWindow().setCursor(Qt.WaitCursor)
        if direction == "next":
            if not self.current_picture.get_next():
                self.log(
                    message=self.tr("No next picture available in the sequence"),
                    log_level=Qgis.MessageLevel.Info,
                    push=True,
                    duration=5,
                )
            else:
                # Get next picture from API
                next_picture_id = self.current_picture.get_next()
                self.current_picture = self.api_client.get_picture(
                    next_picture_id, self.current_picture.get_sequence_id()
                )
        elif direction == "prev":
            if not self.current_picture.get_previous():
                self.log(
                    message=self.tr("No previous picture available in the sequence"),
                    log_level=Qgis.MessageLevel.Info,
                    push=True,
                    duration=5,
                )
            else:
                # Get previous picture from API
                previous_picture_id = self.current_picture.get_previous()
                try:
                    self.current_picture = self.api_client.get_picture(
                        previous_picture_id, self.current_picture.get_sequence_id()
                    )
                except Exception as e:
                    self.log(
                        message=self.tr(
                            f"Error while getting previous picture from API: {str(e)}"
                        ),
                        log_level=Qgis.MessageLevel.Critical,
                        push=True,
                        duration=5,
                    )
                    return
        else:
            self.log(
                message=self.tr(f"Unknown direction: {direction}"),
                log_level=Qgis.MessageLevel.Critical,
                push=True,
                duration=5,
            )
            return
        self.display_image()
        self.iface.mainWindow().setCursor(Qt.ArrowCursor)

    def activate(self):
        """Activate the custom map tool"""
        self.canvas.setCursor(Qt.ArrowCursor)
        self.log(
            message=self.tr("Panoramax map tool activated"),
            log_level=Qgis.MessageLevel.Info,
        )
        self.current_heading = None
        # Check if panoramax layer is set
        for layer in QgsProject.instance().layerTreeRoot().findLayers():
            if layer.name() == "Panoramax":
                self.log(
                    message=f"Panoramax layer found : {layer.name()}",
                    log_level=Qgis.MessageLevel.Info,
                )
                return
        # Check if a panoramax tiled layer exists
        vector_source = QgsDataSourceUri(TILED_LAYER_URI)
        self.log(
            message=f"No panoramax layer, creating it with source {vector_source}",
            log_level=Qgis.MessageLevel.Info,
        )
        vector_layer = QgsVectorTileLayer()
        vector_layer.setDataSource(TILED_LAYER_URI, "Panoramax", "panoramax_tile_layer")
        QgsProject.instance().addMapLayer(vector_layer)
        error = ""
        warnings = []
        vector_layer.loadDefaultStyleAndSubLayers(error, warnings)
        self.log(
            message=f"Error loading style : {error}", log_level=Qgis.MessageLevel.Info
        )
        self.log(
            message=f"Warnings loading style : {warnings}",
            log_level=Qgis.MessageLevel.Info,
        )

    def deactivate(self):
        """Deactivate the custom map tool"""
        self.log(
            message=self.tr("Panoramax map tool deactivated"),
            log_level=Qgis.MessageLevel.Info,
        )
        self.current_heading = None
        if self.dlg_image_viewer.isVisible():
            print("removing dock widget")
            self.iface.removeDockWidget(self.dlg_image_viewer)
        super().deactivate()
