import datetime

from qgis.core import QgsPointXY


class Picture:
    def __init__(self, picture_object):
        """
        Initialize a Picture object.

        :param picture_object: The picture object from the API response.
        :type picture_object: dict
        """
        self.id = picture_object["id"]
        self.sequence_id = picture_object["collection"]

        # retrieve infos from links
        self.previous_geom = None
        self.next_geom = None
        self.instance_url = None
        self.instance_name = None
        self.license_href = None
        self.previous = None
        self.next = None
        for el in picture_object.get("links", []):
            if el.get("rel") == "prev":
                self.previous = el.get("id")
                self.previous_geom = QgsPointXY(
                    *el.get("geometry", {}).get("coordinates", [])
                )
            elif el.get("rel") == "next":
                self.next = el.get("id")
                self.next_geom = QgsPointXY(
                    *el.get("geometry", {}).get("coordinates", [])
                )
            elif el.get("rel") == "via":
                self.instance_url = el.get("href")
                self.instance_name = el.get("instance_name")
            elif el.get("rel") == "license":
                self.license_href = el.get("href")

        # get properties
        properties = picture_object.get("properties", {})
        self.heading = properties.get("view:azimuth")
        self.url = properties.get("geovisio:image")
        self.author = properties.get("geovisio:producer")
        self.date = properties.get("datetime")
        self.license_name = properties.get("license")
        interior_orientation = properties.get("pers:interior_orientation")
        self.field_of_view = interior_orientation.get("field_of_view")

        # attribute not always present:
        self.rank_in_sequence = properties.get("geovisio:rank_in_collection")

        # get geometry
        geometry = picture_object.get("geometry", {})
        if not geometry or "coordinates" not in geometry:
            raise ValueError("Invalid geometry in picture object")
        coordinates = geometry["coordinates"]
        self.geom = QgsPointXY(*coordinates)

    def get_id(self):
        return self.id

    def get_sequence_id(self):
        return self.sequence_id

    def get_rank_in_sequence(self):
        return self.rank_in_sequence

    def get_heading(self) -> float:
        return self.heading

    def get_previous(self):
        return self.previous

    def get_next(self):
        return self.next

    def get_url(self):
        return self.url

    def get_instance(self):
        return f"<a href='{self.instance_url}'>{self.instance_name}</a>"

    def get_author(self):
        return self.author

    def get_date(self):
        return datetime.datetime.fromisoformat(self.date).strftime(
            "%d/%m/%Y - %H:%M:%S"
        )

    def get_license(self):
        return f"<a href='{self.license_href}'>{self.license_name}</a>"

    def get_field_of_view(self):
        return self.field_of_view

    def get_geom(self):
        return self.geom

    def get_previous_geom(self):
        return self.previsous_geom

    def get_next_geom(self):
        return self.next_geom

    def __str__(self):
        return f"Picture(id={self.id}, sequence_id={self.sequence_id}, rank_in_sequence={self.get_rank_in_sequence()})"
